package cn.crushes.cloud.core.common.model;

import cn.crushes.cloud.core.common.constants.RespConstant;
import cn.crushes.cloud.core.common.enums.EnumInterface;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.ToString;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.io.Serializable;

@ToString
@ApiModel(description = "响应实体")
public class R<T> implements Serializable {

    @ApiModelProperty(value = "是否成功", required = true)
    private Boolean success;
    @ApiModelProperty(value = "响应编码", required = true)
    private Integer code;
    @ApiModelProperty(value = "响应信息", required = true)
    private String message;
    @ApiModelProperty(value = "承载数据")
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private T data;


    public static <T> Mono<R<T>> ok(Mono<T> monoBody) {
        return monoResponseCreate(monoBody, RespConstant.SUCCESS_CODE, RespConstant.SUCCESS_MSG, true);
    }

    public static <T> Mono<R<T>> ok(Mono<T> monoBody, String msg) {
        return monoResponseCreate(monoBody, RespConstant.SUCCESS_CODE, msg, true);
    }

    public static <T> Mono<R<T>> ok(Mono<T> monoBody, int code, String msg) {
        return monoResponseCreate(monoBody, code, msg, true);
    }

    public static <T> Mono<R<T>> failed(Mono<T> monoBody) {
        return monoResponseCreate(monoBody, RespConstant.ERROR_CODE, RespConstant.ERROR_MSG, false);
    }

    public static <T> Mono<R<T>> failed(Mono<T> monoBody, String msg) {
        return monoResponseCreate(monoBody, RespConstant.ERROR_CODE, msg, false);
    }

    public static <T> Mono<R<T>> failed(Mono<T> monoBody, int code, String msg) {
        return monoResponseCreate(monoBody, code, msg, false);
    }

    public static <T> Flux<R<T>> ok(Flux<T> monoBody) {
        return fluxResponseCreate(monoBody, RespConstant.SUCCESS_CODE, RespConstant.SUCCESS_MSG, true);
    }

    public static <T> Flux<R<T>> ok(Flux<T> monoBody, String msg) {
        return fluxResponseCreate(monoBody, RespConstant.SUCCESS_CODE, msg, true);
    }

    public static <T> Flux<R<T>> ok(Flux<T> monoBody, int code, String msg) {
        return fluxResponseCreate(monoBody, code, msg, true);
    }

    public static <T> Flux<R<T>> failed(Flux<T> monoBody) {
        return fluxResponseCreate(monoBody, RespConstant.ERROR_CODE, RespConstant.ERROR_MSG, false);
    }

    public static <T> Flux<R<T>> failed(Flux<T> monoBody, String msg) {
        return fluxResponseCreate(monoBody, RespConstant.ERROR_CODE, msg, false);
    }

    public static <T> Flux<R<T>> failed(Flux<T> monoBody, int code, String msg) {
        return fluxResponseCreate(monoBody, code, msg, false);
    }


    public static <T> Mono<R<T>> ok() {
        return responseCreate(RespConstant.SUCCESS_CODE, RespConstant.SUCCESS_MSG, true);
    }

    public static <T> Mono<R<T>> failed() {
        return responseCreate(RespConstant.ERROR_CODE, RespConstant.ERROR_MSG, false);
    }

    public static <T> Mono<R<T>> ok(T data) {
        return responseCreate(data, RespConstant.SUCCESS_CODE, RespConstant.SUCCESS_MSG, true);
    }

    public static <T> Mono<R<T>> ok(T data, String msg) {
        return responseCreate(data, RespConstant.SUCCESS_CODE, msg, true);
    }

    public static <T> Mono<R<T>> ok(int code, String msg) {
        return responseCreate(code, msg, true);
    }

    public static <T> Mono<R<T>> ok(T data, int code, String msg) {
        return responseCreate(data, code, msg, true);
    }

    public static <T> Mono<R<T>> failed(T data) {
        return responseCreate(data, RespConstant.ERROR_CODE, RespConstant.ERROR_MSG, false);
    }

    public static <T> Mono<R<T>> failed(T data, String msg) {
        return responseCreate(data, RespConstant.ERROR_CODE, msg, false);
    }

    public static <T> Mono<R<T>> failed(int code, String msg) {
        return responseCreate(code, msg, false);
    }

    public static <T> Mono<R<T>> failed(T data, int code, String msg) {
        return responseCreate(data, code, msg, false);
    }

    private static <T> Mono<R<T>> responseCreate(int code, String msg, Boolean success) {
        final R<T> r = new R<>();
        r.setCode(code);
        r.setMessage(msg);
        r.setSuccess(success);
        return Mono.just(r);
    }

    private static <T> Mono<R<T>> responseCreate(T data, int code, String msg, Boolean success) {
        Mono<T> monoData = Mono.just(data);
        return getMono(code, msg, success, monoData);
    }

    private static <T> Mono<R<T>> monoResponseCreate(Mono<T> monoData, int code, String msg, Boolean success) {
        return getMono(code, msg, success, monoData);
    }

    private static <T> Mono<R<T>> getMono(int code, String msg, Boolean success, Mono<T> monoData) {
        return monoData.map(x -> {
            final R<T> r = new R<>();
            r.setCode(code);
            r.setData(x);
            r.setMessage(msg);
            r.setSuccess(success);
            return r;
        }).switchIfEmpty(
                R.ok(code, msg));
    }

    private static <T> Flux<R<T>> fluxResponseCreate(Flux<T> fluxData, int code, String msg, Boolean success) {
        return fluxData.map(list -> {
            final R<T> r = new R<>();
            r.setCode(code);
            r.setData(list);
            r.setMessage(msg);
            r.setSuccess(success);
            return r;
        });
    }

    public static <T> Mono<R<T>> error(EnumInterface failure) {
        return responseCreate(failure.getCode(), failure.getMsg(), false);
    }


    public Boolean isSuccess() {
        return success;
    }

    public String getMessage() {
        return message;
    }

    public Integer getCode() {
        return code;
    }

    public T getData() {
        return data;
    }

    public R<T> setSuccess(Boolean success) {
        this.success = success;
        return this;
    }

    public R<T> setMessage(String message) {
        this.message = message;
        return this;
    }

    public R<T> setCode(Integer code) {
        this.code = code;
        return this;
    }

    public R<T> setData(T data) {
        this.data = data;
        return this;
    }
}
