package cn.crushes.cloud.core.cache.aspect;

import cn.crushes.cloud.core.cache.annotations.CacheMono;
import cn.crushes.cloud.core.cache.manager.ReactiveCacheManager;
import cn.crushes.cloud.core.cache.model.CacheType;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import reactor.core.publisher.Mono;

import java.lang.reflect.Method;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Slf4j
@AllArgsConstructor
public class CacheService {

    private final ReactiveCacheManager reactiveCacheManager;

    public Mono<?> getOrCache(ProceedingJoinPoint joinPoint) {
        MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
        Method method = methodSignature.getMethod();
        CacheMono cacheMono = method.getAnnotation(CacheMono.class);
        CacheType cacheType = cacheMono.cacheType();
        long expire = cacheMono.expire();
        String key = cacheMono.key();
        List<Object> request = new ArrayList<>();
        Collections.addAll(request, joinPoint.getArgs());
        return Mono.just(request)
                .flatMap(reqList -> lookup(request,key,cacheType))
                .switchIfEmpty(proceedAndSave(joinPoint, request,key,expire,cacheType));
    }

    private Mono<Object> lookup(List<Object> request,String cachePrefix,CacheType cacheType) {
        return this.reactiveCacheManager.apply(cacheType, provider -> provider.lookup(request, cachePrefix)
                .name("reactive.cache.lookup")
                .metrics()
                .doOnSuccess(o -> log.debug("cache found: " + o))
                .timeout(Duration.ofMillis(300), Mono.fromCallable(() -> {
                    log.warn("Timeout on lookup cache {}", request);
                    return Mono.empty();
                }))
                .onErrorResume(throwable -> {
                    log.warn("Error while lookup cache" + throwable.getMessage());
                    return Mono.empty();
                })).orElseThrow(()->new RuntimeException("Invalid cache type"));
    }

    @SneakyThrows
    private Mono<?> proceedAndSave(ProceedingJoinPoint joinPoint, List<Object> request,String cachePrefix,long expire,CacheType cacheType) {
        Mono<?> result = (Mono) joinPoint.proceed();
        return this.reactiveCacheManager.apply(cacheType,provider -> result.doOnSuccess(response ->
                provider.save(request, response,cachePrefix,Duration.ofMillis(expire))
                        .name("reactive.cache.lookup")
                        .metrics()
                        .doOnSuccess(o -> log.debug("cached result: {}", response))
                        .timeout(Duration.ofMillis(300), Mono.defer(() -> {
                            log.warn("Timeout while caching the response {}", response);
                            return Mono.empty();
                        }))
                        .doOnError(throwable -> log.debug("Error while saving the response", throwable))
                        .onErrorResume(throwable -> Mono.empty())
                        .subscribe()
        )).orElseThrow(()->new RuntimeException("Invalid cache type"));
    }

//    public void clearAll() {
//        log.info("Clearing all tmp cache...");
//        redisTemplate.getConnectionFactory().getReactiveConnection().serverCommands().flushDb().subscribe();
//    }

}
