package cn.crushes.cloud.core.auth.filter;

import cn.crushes.cloud.core.auth.props.TokenProperties;
import cn.crushes.cloud.core.common.constants.MyConstant;
import cn.crushes.cloud.core.common.constants.Oauth2Constant;
import cn.crushes.cloud.core.common.utils.ResponseUtil;
import cn.crushes.cloud.core.common.utils.SecurityUtil;
import cn.crushes.cloud.core.common.utils.TokenUtil;
import io.jsonwebtoken.Claims;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.Ordered;
import org.springframework.data.redis.core.ReactiveRedisTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebFilter;
import org.springframework.web.server.WebFilterChain;
import reactor.core.publisher.Mono;

/**
 * @author youta
 */
@Slf4j
@Component
@AllArgsConstructor
public class PreAuthFilter implements WebFilter, Ordered {
    private final ReactiveRedisTemplate<String, Object> redisService;
    private final TokenProperties properties;

    @Override
    public Mono<Void> filter(ServerWebExchange exchange, WebFilterChain chain) {
        if (!properties.getEnable()) {
            return chain.filter(exchange);
        }
        //　如果在忽略的url里，则跳过
        String path = exchange.getRequest().getURI().getPath();
        String requestUrl = exchange.getRequest().getURI().getRawPath();
        if (ignore(path) || ignore(requestUrl)) {
            return chain.filter(exchange);
        }

        // 验证token是否有效
        ServerHttpResponse resp = exchange.getResponse();
        String headerToken = exchange.getRequest().getHeaders().getFirst(Oauth2Constant.HEADER_TOKEN);
        if (headerToken == null) {
            return unauthorized(resp, "没有携带Token信息");
        }
        String token = TokenUtil.getToken(headerToken);
        Claims claims = SecurityUtil.getClaims(token);
        if (claims == null) {
            return unauthorized(resp, "token已过期或验证不正确！");
        }
        String userId = String.valueOf(claims.get("userId"));
        if (!StringUtils.hasText(userId)) {
            return unauthorized(resp, "登录超时，请重新登录");
        }
//        Boolean hasKey = this.redisService.hasKey(Oauth2Constant.RE_LOGIN_KEY + userId);
        return redisService.hasKey(Oauth2Constant.RE_LOGIN_KEY + userId).flatMap(aBoolean -> {
            if (aBoolean) {
                return unauthorized(resp, "登录超时，请重新登录");
            } else {
                return chain.filter(exchange);
            }
        });
    }

    private Mono<Void> unauthorized(ServerHttpResponse resp, String msg) {
        return ResponseUtil.webFluxResponseWriter(resp, MyConstant.JSON_UTF8, HttpStatus.UNAUTHORIZED, msg);
    }

    private boolean ignore(String path) {
        return properties.getIgnoreUrl().stream()
                .map(url -> url.replace("/**", ""))
                .anyMatch(path::startsWith);
    }

    @Override
    public int getOrder() {
        return -200;
    }

}
