package cn.crushes.cloud.core.auth.aspect;

import cn.crushes.cloud.core.auth.annotation.PreAuth;
import cn.crushes.cloud.core.common.constants.Oauth2Constant;
import cn.crushes.cloud.core.common.context.ReactiveRequestContextHolder;
import cn.crushes.cloud.core.common.entity.LoginUser;
import cn.crushes.cloud.core.common.exception.TokenException;
import cn.crushes.cloud.core.common.utils.SecurityUtil;
import cn.crushes.cloud.core.common.utils.StringPool;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.PatternMatchUtils;
import org.springframework.util.StringUtils;
import reactor.core.publisher.Mono;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

/**
 * @author youta
 */
@Aspect
@Component
@AllArgsConstructor
@Slf4j
public class PreAuthAspect {

    /**
     * 所有权限标识
     */
    private static final String ALL_PERMISSION = "*:*:*";

    @Around("@annotation(cn.crushes.cloud.core.auth.annotation.PreAuth)")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        Signature signature = point.getSignature();
        MethodSignature methodSignature = (MethodSignature) signature;
        Method method = methodSignature.getMethod();
        PreAuth preAuth = method.getAnnotation(PreAuth.class);
        if (ObjectUtils.isEmpty(preAuth)) {
            return point.proceed();
        }
        String perm = preAuth.hasPerm();
        if (!StringUtils.hasText(perm)) {
            return point.proceed();
        }
        Mono<?> mono = (Mono<?>) point.proceed();

        return ReactiveRequestContextHolder.getRequest().flatMap(request -> {
            LoginUser currentUser = SecurityUtil.getCurrentUser(request);
            // 如果用户是超级管理员，则直接跳过权限验证
            if (currentUser.getUsername().equalsIgnoreCase(Oauth2Constant.SUPER_ADMIN)) {
                return mono;
            }
            if (StringUtils.hasText(perm)) {
                if (!hasPermissions(Arrays.stream(String.valueOf(currentUser.getAuthorities()).split(StringPool.COMMA)).collect(Collectors.toList()), perm)) {
                    return Mono.error(new TokenException("权限验证不通过"));
                }
            }
            return mono;
        });
    }

    /**
     * 判断是否包含权限
     *
     * @param authorities 权限列表
     * @param permission  权限字符串
     * @return 用户是否具备某权限
     */
    private boolean hasPermissions(Collection<String> authorities, String permission) {
        return authorities.stream().filter(StringUtils::hasText)
                .anyMatch(x -> ALL_PERMISSION.contains(x) || PatternMatchUtils.simpleMatch(permission, x));
    }
}
