package cn.coufran.springboot.starter.http;

import cn.coufran.springboot.starter.http.content.HttpBodyContent;
import cn.coufran.springboot.starter.http.content.HttpRequestBodyContent;
import cn.coufran.springboot.starter.http.exception.HttpException;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.entity.EntityBuilder;
import org.apache.hc.client5.http.entity.mime.MultipartEntityBuilder;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.core5.http.*;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * HTTP客户端
 * @author Coufran
 * @version 2.0.0
 * @since 1.0.0
 */
public class HttpClient {
    /** 日志Logger */
    private static final Logger LOGGER = LoggerFactory.getLogger(HttpClient.class);

    /** 默认的客户端实例 */
    public static HttpClient INSTANCE_DEFAULT = new HttpClient();

    /**
     * 执行HTTP请求
     * @param request HTTP请求
     * @return HTTP响应
     * @throws HttpException HTTP异常
     */
    public HttpResponse execute(HttpRequest request) {
        LOGGER.info("request:{}", request);

        // Request适配
        // 构造URL
        StringBuilder url = new StringBuilder(request.url()).append("?");
        request.params().forEach((key, value) -> {
            try {
                key = URLEncoder.encode(key, StandardCharsets.UTF_8.name());
                value = URLEncoder.encode(value, StandardCharsets.UTF_8.name());
            } catch (UnsupportedEncodingException e) {
                throw new IllegalStateException(e);
            }
            url.append(key).append("=").append(value).append("&");
        });
        url.deleteCharAt(url.length() - 1);

        // 构造ApacheRequest
        ClassicHttpRequest apacheRequest;
        switch (request.method()) {
            case GET:
                apacheRequest = new HttpGet(url.toString());
                break;
            case POST:
                HttpPost apachePostRequest = new HttpPost(url.toString());
                if (request.contents() != null) {
                    if (request.contentType().getMimeType().startsWith("multipart/")) {
                        MultipartEntityBuilder entityBuilder = MultipartEntityBuilder.create()
                                .setContentType(request.contentType());
                        for (HttpRequestBodyContent content : request.contents()) {
                            content.accept(entityBuilder);
                        }
                        apachePostRequest.setEntity(entityBuilder.build());
                    } else {
                        EntityBuilder entityBuilder = EntityBuilder.create()
                                .setContentType(request.contentType())
                                .setContentEncoding(StandardCharsets.UTF_8.name());
                        if(request.contents().isEmpty()) {
                            entityBuilder.setParameters();
                        } else {
                            request.contents().get(0).accept(entityBuilder);
                        }
                        apachePostRequest.setEntity(entityBuilder.build());
                    }
                }
                apacheRequest = apachePostRequest;
                break;
            default:
                throw new IllegalStateException("unsupported method " + request.method());
        }

        // 设置请求头
        Map<String, String> headers = request.headers();
        headers.forEach((name, value) -> {
            apacheRequest.addHeader(name, value);
        });

        // 请求
        CloseableHttpClient apacheClient = HttpClients.createDefault();
        CloseableHttpResponse apacheResponse;
        try {
            apacheResponse = apacheClient.execute(apacheRequest);
        } catch (IOException e) {
            throw new HttpException(e);
        }

        // Response适配
        int statusCode = apacheResponse.getCode();
//        Header[] responseHeaders = apacheResponse.getHeaders();
//        for (Header responseHeader : responseHeaders) {
//            String name = responseHeader.getName();
//            String value = responseHeader.getValue();
//        }
        HttpEntity responseEntity = apacheResponse.getEntity();
        ContentType contentType = ContentType.parseLenient(responseEntity.getContentType());
        byte[] body;
        try {
            body = EntityUtils.toByteArray(responseEntity);
        } catch (IOException e) {
            throw new HttpException(e);
        }
        HttpResponse response = new HttpResponse(statusCode, contentType, body);

        try {
            apacheResponse.close();
            apacheClient.close();
        } catch (IOException e) {
            throw new HttpException(e);
        }

        LOGGER.info("response:{}", response);
        return response;
    }


}
