package itez.kit;

import java.net.NetworkInterface;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.security.SecureRandom;
import java.util.Date;
import java.util.Enumeration;
import java.util.Random;

/**
 * UUID Generator
 * 
 * 说明：
 * 1、生成的UUID长度：18个字符
 * 2、生成的UUID中含有机器特征码及容器特征码，可在分布式集群环境中保持唯一性
 * 3、生成的UUID可作为排序索引
 * 4、可将生成的UUID字符串转换为时间格式
 */
public final class EUid {
	private static final int LOW_ORDER_THREE_BYTES = 0x00ffffff;
	private static final int MACHINE_IDENTIFIER = createMachineIdentifier();
	private static final short PROCESS_IDENTIFIER = createProcessIdentifier();
	private static long timestamp;
	private static int counter = 0;

	/**
	 * 生成唯一UID标识 *
	 * 
	 * @category 生成唯一UID标识
	 * @return
	 */
	public final static synchronized String generator() {
		long t = System.currentTimeMillis();
		if (t == timestamp) {
			counter++;
			if (counter >= 1679615) counter = 1;
		} else {
			Random rand = new Random(t);
			counter = rand.nextInt(1679615);
			timestamp = t;
		}
		StringBuffer sb = new StringBuffer();
		sb.append(EStr.addPrefix(Long.toString(t / 1000, 35), 8, "zzzzzzzz"));
		sb.append(EStr.addPrefix(Long.toString(MACHINE_IDENTIFIER + PROCESS_IDENTIFIER, 36), 6));
		sb.append(EStr.addPrefix(Integer.toString(counter, 36), 4));
		return sb.toString().replace("-", "_").toUpperCase();
	}

	/**
	 * 返回机器码，用于集群环境保持UUID唯一性
	 * 
	 * @category 返回机器码，用于集群环境保持UUID唯一性
	 * @return
	 */
	private static int createMachineIdentifier() {
		int machinePiece;
		try {
			StringBuilder sb = new StringBuilder();
			Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
			while (e.hasMoreElements()) {
				NetworkInterface ni = e.nextElement();
				sb.append(ni.toString());
				byte[] mac = ni.getHardwareAddress();
				if (mac != null) {
					ByteBuffer bb = ByteBuffer.wrap(mac);
					try {
						sb.append(bb.getChar());
						sb.append(bb.getChar());
						sb.append(bb.getChar());
					} catch (BufferUnderflowException shortHardwareAddressException) {

					}
				}
			}
			machinePiece = sb.toString().hashCode();
		} catch (Throwable t) {
			machinePiece = (new SecureRandom().nextInt());
		}
		machinePiece = machinePiece & LOW_ORDER_THREE_BYTES;
		return machinePiece;
	}

	/**
	 * 返回JVM的运行时线程码，用于集群环境保持UUID唯一性
	 * 
	 * @category 返回JVM的运行时线程码，用于集群环境保持UUID唯一性
	 * @return
	 */
	private static short createProcessIdentifier() {
		short processId;
		try {
			String processName = java.lang.management.ManagementFactory.getRuntimeMXBean().getName();
			if (processName.contains("@")) {
				processId = (short) Integer.parseInt(processName.substring(0, processName.indexOf('@')));
			} else {
				processId = (short) java.lang.management.ManagementFactory.getRuntimeMXBean().getName().hashCode();
			}
		} catch (Throwable t) {
			processId = (short) new SecureRandom().nextInt();
		}
		return processId;
	}

	/**
	 * 将UUID转换为日期时间
	 * 
	 * @category 将UUID转换为日期时间
	 * @param tmuid
	 * @return
	 */
	public static Date parseUuid(String tmuid) {
		if (tmuid.length() < 8)
			return null;
		String s = tmuid.substring(0, 8).toLowerCase();
		long l = Long.parseLong(s.replace("z", "0").replace("_", "-"), 35) * 1000;
		return new Date(l);
	}
	
	public static void main(String[] args) {
//		System.out.println(EUid.generator());
//		String salt = "ne7gY4ZHcnEmEZTE-yJAxWIu1DaQW_dh";
//		System.out.println(HashKit.md5("7735243" + salt));
		for(int i=0; i<20; i++){
			System.out.println(EUid.generator());
		}
	}
}