package itez.kit;

import java.io.File;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * String处理工具集
 * 
 * @category String处理工具集
 * @author netwild
 *
 */
public class EStr {

	public static final String Encoding = "UTF-8";
	public static final String FileSep = File.separator; //本地文件系统分隔符
	private static final String PREFIX_STRING_ZERO = "0000000000"; //默认用于补齐空位的前缀字符串

    // 字符集GBK
    public static final Charset GBK = Charset.forName("GBK");
    // 字符集ISO-8859-1
    public static final Charset ISO_8859_1 = Charset.forName("ISO-8859-1");
    // 字符集utf-8
    public static final Charset UTF_8 = Charset.forName("UTF-8");
	
	/**
	 * 判断字符串是否为Null
	 * 
	 * @category 返回字符串是否为Null
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean isNull(Object para){
		return null == para;
	}
	
	/**
	 * 判断对象数组中是否含有NULL
	 * @param strings
	 * @return
	 */
	public static boolean hasNull(Object...paras){
		if(paras == null) return true;
		for(Object para : paras){
			if(para == null) return true;
		}
		return false;
	}
	
	/**
	 * 判断对象数组中是否全部都为NULL
	 * @param strings
	 * @return
	 */
	public static boolean allNull(Object...paras){
		if(paras == null) return true;
		for(Object para : paras){
			if(para != null) return false;
		}
		return true;
	}

	/**
	 * 判断字符串是否不为Null
	 * 
	 * @category 判断字符串是否不为Null
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean notNull(Object para){
		return !isNull(para);
	}
	
	/**
	 * 如果字符串为Null，则返回默认字符串；否则原样返回
	 * 
	 * @category 如果字符串为Null，则返回默认字符串；否则原样返回
	 * @param str 原字符串
	 * @param defaultStr 默认字符串
	 * @return
	 */
	public static Object ifNull(Object str, Object defaultStr){
		return isNull(str) ? defaultStr : str;
	}
	
	/**
	 * 判断字符串是否为空（包括Null或者""）
	 * 
	 * @category 判断字符串是否为空（包括Null或者""）
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean isEmpty(String str){
		return null == str || str.equals("");
	}
	
	/**
	 * 判断字符串数组是否含有空值（包括Null或者""）
	 * 
	 * @category 判断字符串是否不为空（包括Null或者""）
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean hasEmpty(String...strings){
		if (strings == null) return true;
		for (String str : strings) {
			if (isEmpty(str)) return true;
		}
		return false;
	}
	
	/**
	 * 判断字符串数组是否全部都为空（包括Null或者""）
	 * 
	 * @category 判断字符串是否不为空（包括Null或者""）
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean allEmpty(String...strings){
		if (strings == null) return true;
		for (String str : strings) {
			if (notEmpty(str)) return false;
		}
		return true;
	}
	
	/**
	 * 判断字符串是否不为空（包括Null或者""）
	 * 
	 * @category 判断字符串是否不为空（包括Null或者""）
	 * @param str 待检测的原字符串
	 * @return
	 */
	public static boolean notEmpty(String str){
		return !isEmpty(str);
	}
	
	/**
	 * 判断对象是否为空（包括Null或者""）
	 * 
	 * @category 判断对象是否为空（包括Null或者""）
	 * @param obj
	 * @return
	 */
	public static boolean notEmpty(Object obj){
		return obj == null ? false : notEmpty(obj.toString());
	}
	
	/**
	 * 如果字符串为空，则返回默认字符串；否则原样返回
	 * 
	 * @category 如果字符串为空，则返回默认字符串；否则原样返回
	 * @param str 原字符串
	 * @param defaultStr 默认字符串
	 * @return
	 */
	public static String ifEmpty(String str, String defaultStr){
		return isEmpty(str) ? defaultStr : str;
	}
	
	/**
	 * 取字符串的前几位字符
	 * 
	 * @category 取字符串的前几位字符
	 * @param str
	 * @param len
	 * @return
	 */
	public static String left(String str, int count){
		return isNull(str) ? "" : count < str.length() ? str.substring(0, count) : str;
	}
	
	/**
	 * 取字符串的后几位字符
	 * 
	 * @category 取字符串的后几位字符
	 * @param str
	 * @param len
	 * @return
	 */
	public static String right(String str, int count){
		if(isNull(str)) return "";
		int len = str.length();
		if(len < count) return str;
		return str.substring(len - count);
	}
	
	/**
	 * 补齐字符串前缀
	 * 
	 * @category 补齐字符串前缀
	 * @param str 原字符串
	 * @param len 补齐之后的字符串长度
	 * @return
	 */
	public static String addPrefix(int number, int len){
		return addPrefix(number + "", len, PREFIX_STRING_ZERO);
	}
	
	/**
	 * 补齐字符串前缀
	 * 
	 * @category 补齐字符串前缀
	 * @param str 原字符串
	 * @param len 补齐之后的字符串长度
	 * @return
	 */
	public static String addPrefix(String str, int len){
		return addPrefix(str, len, PREFIX_STRING_ZERO);
	}
	
	/**
	 * 补齐字符串前缀
	 * 
	 * @category 补齐字符串前缀
	 * @param str 原字符串
	 * @param len 补齐之后的字符串长度
	 * @param prefix 用于补齐的前缀字符串
	 * @return
	 */
	public static String addPrefix(String str, int len, String prefix){
		str = prefix + str;
		return right(str, len);
	}

	/**
	 * 首字母转小写
	 * @param s
	 * @return
	 */
	public static String toLowerCaseFirst(String s) {
        if(Character.isLowerCase(s.charAt(0))){
        	return s;
        } else {
        	return (new StringBuilder()).append(Character.toLowerCase(s.charAt(0))).append(s.substring(1)).toString();
        }
    }
	
	/**
	 * 首字母转大写
	 * @param s
	 * @return
	 */
    public static String toUpperCaseFirst(String s) {
        if(Character.isUpperCase(s.charAt(0))){
        	return s;
        } else {
        	return (new StringBuilder()).append(Character.toUpperCase(s.charAt(0))).append(s.substring(1)).toString();
        }
    }
    
    /**
     * 将列表转换为逗号间隔的字符串
     * @param arr
     * @return
     */
    public static String parseArray(List<String> list){
    	return parseArray(list.toArray(new String[list.size()]), ",");
    }
    
    /**
     * 将数组转换为逗号间隔的字符串
     * @param arr
     * @return
     */
    public static String parseArray(String[] arr){
    	return parseArray(arr, ",");
    }
    
    /**
     * 将数组转换为指定字符间隔的字符串
     * @param arr
     * @param sept
     * @return
     */
    public static String parseArray(String[] arr, String sept){
    	if(arr == null) return "";
    	return Arrays.stream(arr).collect(Collectors.joining(sept));
    }
    
    /**
     * 以逗号为分隔符转换为数组
     * @param str
     * @return
     */
    public static String[] toArray(String str){
    	return toArray(str, ",");
    }
    
    /**
     * 以指定符号作为分隔符，将字符串转换为数组
     * @param str
     * @param sep
     * @return
     */
    public static String[] toArray(String str, String sept){
    	return str.replace(" ", "").split(sept);
    }
    
    /**
     * 将ID字符串列表转换成SQL in 参数
     * @category a, b, c ==> a, b, c
     * @param ids
     * @return
     */
    public static String ida2sqlIn(Object...ida){
    	return Arrays.stream(ida).map(id -> String.format("'%s'", id.toString().trim())).collect(Collectors.joining(","));
    }
    
    /**
     * 将ID字符串列表转换成SQL in 参数
     * @category a, b, c ==> a, b, c
     * @param ids
     * @return
     */
    public static String ida2sqlIn(String...ida){
    	return Arrays.stream(ida).map(id -> String.format("'%s'", id.trim())).collect(Collectors.joining(","));
    }
    
    /**
     * 将ID字符串列表转换成SQL in 参数
     * @category a, b, c ==> 'a', 'b', 'c'
     * @param ids
     * @return
     */
    public static String ids2sqlIn(String ids){
    	String format = "'%s'";
    	return String.format(format, ids.replace(" ", "").replace(",", "','"));
    }
    
    /**
     * 在数组中查找，返回索引
     * @param arr
     * @param keyword
     * @return
     */
    public static int findInArr(String[] arr, String keyword){    	
    	int ind = -1;
    	for(String item : arr){
    		ind++;
    		if(item.equals(keyword)) return ind;
    	}
    	return -1;
    }
    
    /**
     * 从传入的多个参数中，返回第一个非NULL且非空的
     * @param objects
     * @return
     */
    public static String findUseful(Object...objects){
    	return Arrays.stream(objects).filter(EStr::notEmpty).findFirst().orElse("").toString();
    }
    
    /**
     * 将String转换为byte[]数组
     * @param str
     * @return
     */
    public static byte[] toByte(String str){
    	if(str == null) return null;
    	return str.getBytes(UTF_8);
    }
    
    /**
     * 将byte[]数组转换为String
     * @param bytes
     * @return
     */
    public static String parse(byte[] bytes){
    	if(bytes == null) return null;
    	return new String(bytes, UTF_8);
    }
	
}
