package itez.kit;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * 日期处理工具类
 * 
 * @category 日期处理工具类
 * @author netwild
 *
 */
public class EDate {

	public static final String PATTERN_Y = "yyyy"; // 年
	public static final String PATTERN_YM = "yyyy-MM"; // 年-月
	public static final String PATTERN_MD = "MM-dd"; // 月-日
	public static final String PATTERN_YMD = "yyyy-MM-dd"; // 年-月-日
	public static final String PATTERN_YMD_HM = "yyyy-MM-dd HH:mm"; // 年-月-日 时：分
	public static final String PATTERN_HM = "HH:mm"; // 时：分
	public static final String PATTERN_YMD_HMS = "yyyy-MM-dd HH:mm:ss"; // 年-月-日 时：分：秒
	public static final String PATTERN_YMD_HMS_S = "yyyy-MM-dd HH:mm:ss:SSS"; // 年-月-日 时：分：秒：毫秒
	
	public static enum DateUtil { DAY, HOUR, MINUTE, SECOND }
	public static enum DatePart { YEAR, MONTH, DAY, HOUR, MINUTE, SECOND }

	/**
	 * 获取当前时间
	 * 
	 * @category 获取当前时间
	 * @return
	 */
	public static Date getDate() {
		return new Date();
	}

	/**
	 * 获取当前时间的时间戳
	 * 
	 * @category 获取当前时间的时间戳
	 * @return
	 */
	public static long getTime() {
		return new Date().getTime();
	}
	
	/**
	 * 日期时间格式化
	 * 
	 * @category 日期时间格式化
	 * @param date
	 * @param pattern
	 * @return
	 */
	public static String format(Date date) {
		return format(date, PATTERN_YMD);
	}
	
	/**
	 * 日期时间格式化
	 * 
	 * @category 日期时间格式化
	 * @param date
	 * @param pattern
	 * @return
	 */
	public static String format(Date date, String pattern) {
		DateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}

	/**
	 * 将字符串转换为日期时间
	 * 
	 * @category 将字符串转换为日期时间
	 * @param date
	 * @param pattern
	 * @return
	 */
	public static Date parse(String date, String pattern) {
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		try {
			return format.parse(date);
		} catch (Exception e) {
			return getDate();
		}
	}
	
	/**
	 * 自动判断日期格式后，将字符串转换为日期时间
	 * 
	 * @param date
	 * @return
	 */
	public static Date parse(String date){
		String pattern = parsePattern(date);
		return EStr.isNull(pattern) ? getDate() : parse(date, pattern);
	}
	
	/**
	 * 自动判断指定字符串与相应日期时间格式匹配的模板
	 * 无法匹配时，返回null
	 * 
	 * @param date
	 * @return
	 */
	private static String parsePattern(String date){
		String pattern = null;
		if(ERegex.has(date, "^\\d{4}$")){// 年
			pattern = PATTERN_Y;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}$")){// 年-月
			pattern = PATTERN_YM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}$")){// 年-月-日
			pattern = PATTERN_YMD;
		}else if(ERegex.has(date, "^\\d{1,2}\\-\\d{1,2}$")){// 月-日
			pattern = PATTERN_MD;
		}else if(ERegex.has(date, "^\\d{1,2}\\:\\d{1,2}$")){// 时：分
			pattern = PATTERN_HM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分
			pattern = PATTERN_YMD_HM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分：秒
			pattern = PATTERN_YMD_HMS;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}\\:\\d{1,3}$")){// 年-月-日 时：分：秒：毫秒
			pattern = PATTERN_YMD_HMS_S;
		}
		return pattern;
	}
	
	/**
	 * 返回日期中的某个区间
	 * 
	 * @category 返回日期中的某个区间
	 * @param dp
	 * @param dt
	 * @return
	 */
	public static String part(DatePart dp, Date dt){
		if(dp == DatePart.YEAR){// 年
			return format(dt, "yyyy");
		}else if(dp == DatePart.MONTH){// 月
			return format(dt, "MM");
		}else if(dp == DatePart.DAY){// 日
			return format(dt, "dd");
		}else if(dp == DatePart.HOUR){// 时
			return format(dt, "HH");
		}else if(dp == DatePart.MINUTE){// 分
			return format(dt, "mm");
		}else{// 秒
			return format(dt, "ss");
		}
	}

	/**
	 * 两个日期的时间差，返回"X天X小时X分X秒"
	 * 
	 * @category 两个日期的时间差，返回"X天X小时X分X秒"
	 * @param start
	 * @param end
	 * @return
	 */
	public static String diffMsg(Date start, Date end) {
		long diff = (end.getTime() - start.getTime()) / 1000;// 转换成秒
		long day = diff / (24 * 3600);
		long hour = diff % (24 * 3600) / 3600;
		long minute = diff % 3600 / 60;
		long second = diff % 60;
		StringBuilder sb = new StringBuilder();
		sb.append(day);
		sb.append("天");
		sb.append(hour);
		sb.append("小时");
		sb.append(minute);
		sb.append("分");
		sb.append(second);
		sb.append("秒");
		return sb.toString();
	}
	
	/**
	 * 返回两个日期的间隔
	 * 
	 * @category 返回两个日期的间隔
	 * @param dateUtil 间隔单位：DateUtil.DAY（天）/ DateUtil.HOUR（小时）/ DateUtil.MINUTE（分钟）/ DateUtil.SECOND（秒）
	 * @param start
	 * @param end
	 * @return
	 */
	public static int diff(DateUtil dateUtil, Date start, Date end){
		int diffLen = 0;
		if(dateUtil == DateUtil.DAY){// 相差天数
			diffLen = 24 * 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.HOUR){// 相差小时数
			diffLen = 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.MINUTE){// 相差分钟数
			diffLen = 60 * 1000;
		}else if(dateUtil == DateUtil.SECOND){// 相差秒数
			diffLen = 1000;
		}
		return (int)((end.getTime() - start.getTime()) / diffLen);
	}
	
	/**
	 * 返回两个日期的间隔
	 * @param start
	 * @param end
	 * @return
	 */
	public static Duration diff(Date start, Date end){
		LocalDateTime ld1 = toLocalDateTime(start);
		LocalDateTime ld2 = toLocalDateTime(end);
		Duration duration = Duration.between(ld1, ld2);
		return duration;
	}
	
	/**
	 * 日期计算
	 * 
	 * @category 日期计算
	 * @param dateUtil 间隔单位：DateUtil.DAY（天）/ DateUtil.HOUR（小时）/ DateUtil.MINUTE（分钟）/ DateUtil.SECOND（秒）
	 * @param start
	 * @param end
	 * @return
	 */
	public static Date add(DateUtil dateUtil, Date start, int diff){
		int diffLen = 0;
		if(dateUtil == DateUtil.DAY){// 相差天数
			diffLen = 24 * 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.HOUR){// 相差小时数
			diffLen = 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.MINUTE){// 相差分钟数
			diffLen = 60 * 1000;
		}else if(dateUtil == DateUtil.SECOND){// 相差秒数
			diffLen = 1000;
		}
		return new Date(start.getTime() + diff * diffLen);
	}
	
	/**
	 * 日期计算
	 * @param start
	 * @param diff
	 * @param calendarType
	 * @return
	 */
	private static Date addUtil(Date start, int diff, int calendarType){
		Calendar calendar = new GregorianCalendar();
		calendar.setTime(start);
		calendar.add(calendarType, diff);
		return calendar.getTime();
	}
	
	/**
	 * 年份计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addYear(Date start, int diff){
		return addUtil(start, diff, Calendar.YEAR);
	}
	
	/**
	 * 月份计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addMonth(Date start, int diff){
		return addUtil(start, diff, Calendar.MONTH);
	}
	
	/**
	 * 天数计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addDay(Date start, int diff){
		return addUtil(start, diff, Calendar.DATE);
	}
	
	/**
	 * 小时计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addHour(Date start, int diff){
		return addUtil(start, diff, Calendar.HOUR);
	}
	
	/**
	 * 分钟计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addMinute(Date start, int diff){
		return addUtil(start, diff, Calendar.MINUTE);
	}
	
	/**
	 * 分钟计算
	 * @param start
	 * @param diff
	 * @return
	 */
	public static Date addSecond(Date start, int diff){
		return addUtil(start, diff, Calendar.SECOND);
	}

	/**
	 * 得到某一天是星期几
	 * 
	 * @category 得到某一天是星期几
	 * @param date 日期
	 * @return String 星期几
	 */
	@SuppressWarnings("static-access")
	public static String getDateInWeek(Date date) {
		String[] weekDays = { "星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六" };
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		int dayIndex = calendar.get(calendar.DAY_OF_WEEK) - calendar.SUNDAY;
		if (dayIndex < 0) dayIndex = 0;
		return weekDays[dayIndex];
	}
	
	/**
	 * 根据日期月份，获取月份的开始和结束日期
	 * 
	 * @param date
	 * @return
	 */
	public static Date[] getMonthDate(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);

		// 得到前一个月的第一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
		Date start = calendar.getTime();

		// 得到前一个月的最后一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
		Date end = calendar.getTime();

		return new Date[]{start, end};
	}
	
    /**
     * 判断传入的时间是否在当前时间之后，返回boolean值
     * true: 过期
     * false: 还没过期
     *
     * @param date
     * @return
     */
    public static boolean isExpire(Date date) {
    	if (date.before(getDate())) return true;
    	return false;
    }
	
	/**
	 * 格式化为JDK8的LocalDate类型
	 * @param date
	 * @return
	 */
	public static LocalDate toLocalDate(Date date){
		int year = Integer.parseInt(part(DatePart.YEAR, date));
		int month = Integer.parseInt(part(DatePart.MONTH, date));
		int day = Integer.parseInt(part(DatePart.DAY, date));
		LocalDate ldate = LocalDate.of(year, month, day);
		return ldate;
	}
	
	/**
	 * 格式化为JDK8的LocalTime类型
	 * @param date
	 * @return
	 */
	public static LocalTime toLocalTime(Date date){
		int hour = Integer.parseInt(part(DatePart.HOUR, date));
		int minute = Integer.parseInt(part(DatePart.MINUTE, date));
		int second = Integer.parseInt(part(DatePart.SECOND, date));
		LocalTime ldate = LocalTime.of(hour, minute, second);
		return ldate;
	}
	
	/**
	 * 格式化为JDK8的LocalDateTime类型
	 * @param date
	 * @return
	 */
	public static LocalDateTime toLocalDateTime(Date date){
		LocalDate ldate = toLocalDate(date);
		LocalTime ltime = toLocalTime(date);
		LocalDateTime ldatetime = LocalDateTime.of(ldate, ltime);
		return ldatetime;
	}

	public static void main(String[] args) {
		Date dd = getDate();
		Date dd1 = addYear(dd, 1);
		Date[] dds = getMonthDate(dd);
		System.out.println(format(dds[0]));
		System.out.println(format(dds[1]));
		System.out.println(format(dd));
		System.out.println(format(dd1));
		
		String d1 = "2018-01-26 23:25:00";
		String d2 = "2018-01-27 00:25:00";
		LocalDateTime ld1 = toLocalDateTime(parse(d1));
		LocalDateTime ld2 = toLocalDateTime(parse(d2));
		Duration duration = Duration.between(ld1, ld2);
		System.out.println(d1);
		System.out.println(d2);
		System.out.println(duration.toMinutes());
		
//		Date d1 = parse("2017-01-15 20:40:00", PATTERN_YMD_HMS);
//		Date d2 = parse("2017-01-20 22:49:20", PATTERN_YMD_HMS);
//		String diff = diffMsg(d1, d2);
//		int diff = diff(DateUtil.SECOND, d1, d2);
//		System.out.println(diff);
//		Date d3 = add(DateUtil.SECOND, d1, -1);
//		System.out.println(format(d3, PATTERN_YMD_HMS));
//		System.out.println(getDateInWeek(d1));
//		System.out.println(format(d3, "mm"));
//		String dd = "1981";
//		System.out.println(parsePattern(dd));
//		dd = "1981-07-07";
//		System.out.println(parsePattern(dd));
//		dd = "07-07";
//		System.out.println(parsePattern(dd));
//		dd = "1981-07";
//		System.out.println(parsePattern(dd));
//		dd = "11:17";
//		System.out.println(parsePattern(dd));
//		dd = "1981-07-07 11:17";
//		System.out.println(parsePattern(dd));
//		dd = "1981-07-07 11:17:25";
//		System.out.println(parsePattern(dd));
//		dd = "1981-07-07 11:17:25:111";
//		System.out.println(parsePattern(dd));
	}
	/*
	*/

}
