package itez.kit;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.jfinal.log.Log;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 类实例创建者创建者
 * Created by michael on 17/3/21.
 */
public class EClass {

    public static Log log = Log.getLog(EClass.class);
	private static final Map<String, Object> singletons = new ConcurrentHashMap<>();

	/**
	 * 获取类加载器
	 * @return
	 */
	public static ClassLoader loader() {
		return Thread.currentThread().getContextClassLoader();
	}
	
	/**
	 * 装载类，仅将类的class文件加载到JVM内存中，不执行类中的静态代码块
	 * 
	 *  √ 1）加载：将class文件加载到JVM内存中。
	 *  ✘  2）链接：校验class文件、为类变量分配存储空间并设置类变量初始值、引用常量等
	 *  ✘  3）初始化：执行类变量赋值和静态代码块
	 * 
	 * @param className
	 * @return
	 */
	public static Class<?> loadClass(String className){
		try {
			return loader().loadClass(className);
		} catch (ClassNotFoundException e) {
			log.error("无法找到类：" + className);
		}
		return null;
	}
	
	/**
	 * 装载并初始化类
	 * 
	 *  √ 1）加载：将class文件加载到JVM内存中。
	 *  √ 2）链接：校验class文件、为类变量分配存储空间并设置类变量初始值、引用常量等
	 *  √ 3）初始化：执行类变量赋值和静态代码块
	 * 
	 * @param className
	 * @return
	 */
	public static Class<?> forName(String className){
		return forName(className, true);
	}

	/**
	 * 装载并初始化类
	 * 
	 *  √ 1）加载：将class文件加载到JVM内存中。
	 *  √ 2）链接：校验class文件、为类变量分配存储空间并设置类变量初始值、引用常量等
	 *  √ 3）初始化：执行类变量赋值和静态代码块
	 *  
	 * @param className
	 * @param init 是否对类进行初始化
	 * @return
	 */
	public static Class<?> forName(String className, boolean init){
		try {
			return Class.forName(className, init, loader());
		} catch (ClassNotFoundException e) {
			log.error("无法找到类：" + className);
		}
		return null;
	}

    /**
     * 创建类的新实例（执行类的无参构造方法）
     *
     * @param <T>
     * @param clazz
     * @return
     */
    public static <T> T newInstance(Class<T> clazz) {
        try {
            Constructor<T> constructor = clazz.getDeclaredConstructor();
            constructor.setAccessible(true);
            return (T) constructor.newInstance();
        } catch (Exception e) {
            log.error("无法创建类的实例：" + clazz + "\n" + e.toString(), e);
        }
        return null;
    }
    
    /**
     * 创建类的新实例（执行类的有参构造方法）
     * @param clazz
     * @param paramsTypes
     * @param args
     * @return
     */
    public static <T> T newInstance(Class<T> clazz, Class<?>[] paramsTypes, Object... args){
    	try {
        	Constructor<T> constructor = clazz.getConstructor(paramsTypes);
            constructor.setAccessible(true);
            return (T) constructor.newInstance(args);
		} catch (Exception e) {
			log.error("无法创建类的实例：" + clazz + "\n" + e.toString(), e);
		}
    	return null;
    }
    
    /**
     * 创建类的新实例（执行类的默认构造方法）
     *
     * @param <T>
     * @param clazzName
     * @return
     */
    @SuppressWarnings("unchecked")
	public static <T> T newInstance(String clazzName) {
        try {
            Class<T> clazz = (Class<T>) forName(clazzName, false);
            return newInstance(clazz);
        } catch (Exception e) {
            log.error("无法创建类的实例：" + clazzName + "\n" + e.toString(), e);
        }
        return null;
    }

    /**
     * 获取类的单列实例（无参构造方法）
     *
     * @param clazz
     * @param <T>
     * @return
     */
	@SuppressWarnings("unchecked")
	public static <T> T singleton(Class<T> clazz) {
		String singleKey = clazz.getName();
        Object object = singletons.get(singleKey);
        if (object == null) {
            synchronized (clazz) {
                object = singletons.get(singleKey);
                if (object == null) {
                    object = newInstance(clazz);
                    if (object != null) {
                        singletons.put(singleKey, object);
                    } else {
                        Log.getLog(clazz).error("cannot new newInstance!!!!");
                    }

                }
            }
        }
        return (T) object;
    }

    /**
     * 获取类的单列实例（有参构造方法）
     *
     * @param clazz
     * @param <T>
     * @return
     */
	@SuppressWarnings("unchecked")
	public static <T> T singleton(Class<T> clazz, Class<?>[] paramsTypes, Object... args) {
		String singleKey = clazz.getName();
		String paramsKey = Arrays.stream(paramsTypes).map(cls -> cls.getName()).collect(Collectors.joining(""));
		singleKey += paramsKey;
        Object object = singletons.get(singleKey);
        if (object == null) {
            synchronized (clazz) {
                object = singletons.get(singleKey);
                if (object == null) {
                    object = newInstance(clazz, paramsTypes, args);
                    if (object != null) {
                        singletons.put(singleKey, object);
                    } else {
                        Log.getLog(clazz).error("cannot new newInstance!!!!");
                    }

                }
            }
        }
        return (T) object;
    }
    
    /**
     * 返回类的全路径
     * 
     * 与Method.toGenericString()的区别是，不包含方法返回值以及抛错相关的定义描述
     * 
     * @param clazz
     * @return
     */
    public static String getClassFullPath(Class<?> clazz){
    	return clazz.getName();
    }
    
    /**
     * 返回类的全路径
     * 
     * 与Method.toGenericString()的区别是，不包含方法返回值以及抛错相关的定义描述
     * 
     * @param method
     * @return
     */
    public static String getClassFullPath(Method method){
    	Class<?> clazz = getUsefulClass(method.getDeclaringClass());
    	return getClassFullPath(clazz);
    }
    
    /**
     * 返回方法的全路径
     * 
     * 与Method.toGenericString()的区别是，不包含方法返回值以及抛错相关的定义描述
     * 
     * @param method
     * @return
     */
    public static String getMethodFullPath(Method method){
    	String methodName = method.getName();
    	Class<?>[] params = method.getParameterTypes();
    	String paramsClass = Arrays.stream(params).map(p -> p.getName()).collect(Collectors.joining(","));
    	return String.format("%s(%s)", methodName, paramsClass);
    }

	/**
	 * 获取代理类的实际类名
	 * @param clazz
	 * @return
	 */
    public static Class<?> getUsefulClass(Class<?> clazz) {
        //ControllerTest$ServiceTest$$EnhancerByGuice$$40471411#hello
        //com.demo.blog.Blog$$EnhancerByCGLIB$$69a17158
        return clazz.getName().indexOf("$$EnhancerBy") == -1 ? clazz : clazz.getSuperclass();
    }


    
    
    /**
     * 类的get/set方法缓存，用于减少对类的反射工作
     */
    private static Multimap<Class<?>, Method> classGetMethodsCache = ArrayListMultimap.create();
    private static Multimap<Class<?>, Method> classSetMethodsCache = ArrayListMultimap.create();

    /**
     * 获取 某class 下的所有get方法
     * 
     * @param clazz
     * @return
     */
    public static Collection<Method> getClassGetMethods(Class<?> clazz) {
    	return getClassAttrMethods(clazz, "get");
    }
    
    /**
     * 获取 某class 下的所有set方法
     * 
     * @param clazz
     * @return
     */
    public static Collection<Method> getClassSetMethods(Class<?> clazz) {
    	return getClassAttrMethods(clazz, "set");
    }
    
    /**
     * 获取 某class 下的所有get/set方法
     *
     * @param clazz
     * @return
     */
    public static Collection<Method> getClassAttrMethods(Class<?> clazz, String getOrSet) {
    	Integer parameterCount = null;
    	Multimap<Class<?>, Method> cache = null;
    	if(getOrSet.equalsIgnoreCase("get")){
    		parameterCount = 0;
    		cache = classGetMethodsCache;
    	}else{
    		parameterCount = 1;
    		cache = classSetMethodsCache;
    	}
        Collection<Method> retMethods = cache.get(clazz);
        if (retMethods == null || retMethods.isEmpty()) {
            Method[] methods = clazz.getMethods();
            for (Method method : methods) {
                if (method.getName().startsWith(getOrSet) && method.getName().length() > 3 && method.getParameterCount() == parameterCount) {
                	cache.put(clazz, method);
                	retMethods.add(method);
                }
            }
        }
        return retMethods;
    }


    /**
     * 确定class是否可以被加载
     * @param className 完整类名
     * @param classLoader 类加载
     * @return {boolean}
     */
    public static boolean isPresent(String className, ClassLoader classLoader) {
        try {
            Class.forName(className, true, classLoader);
            return true;
        }
        catch (Throwable ex) {
            // Class or one of its dependencies is not present...
            return false;
        }
    }


}

