/**
 * 
 */
package itez.jwinner.kit;

import java.io.File;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * <p>
 * String / Object 工具类
 * </p>
 * 
 * <p>Copyright(C) 2017-2020 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2020年4月20日 下午3:22:42
 */
public class JStr {

    // 默认变量
    public static final String Encoding = "UTF-8";                  //默认字符集
    public static final String FileSep = File.separator;            //本地文件系统分隔符

    // 字符集
    public static final Charset UTF_8 = StandardCharsets.UTF_8;
    public static final Charset ISO_8859_1 = StandardCharsets.ISO_8859_1;

    //*******************************************  空对象判断  ***********************************************/

    /**
     * 判断对象是否为NULL
     * @param para 对象
     * @return boolean
     */
    public static boolean isNull(Object para){
        return null == para;
    }

    /**
     * 判断对象是否不为NULL
     * @param para 对象
     * @return boolean
     */
    public static boolean notNull(Object para){
        return !isNull(para);
    }

    /**
     * 如果对象为Null，则返回默认值
     * @param orig 对象
     * @param def 默认值
     * @return Object
     */
    public static Object ifNull(Object orig, Object def){
        return isNull(orig) ? def : orig;
    }

    /**
     * 判断对象数组中是否含有NULL
     * @param paras 对象数组
     * @return boolean
     */
    public static boolean hasNull(Object...paras){
        if(null == paras) return true;
        return Arrays.stream(paras).anyMatch(JStr::isNull);
    }

    /**
     * 判断对象数组是否全部为NULL
     * @param paras 对象数组
     * @return boolean
     */
    public static boolean allNull(Object...paras){
        if(null == paras) return true;
        return Arrays.stream(paras).allMatch(JStr::isNull);
    }

    /**
     * 判断字符串是否为空
     * @param str 字符串
     * @return boolean
     */
    public static boolean isEmpty(String str){
        return isNull(str) || str.equals("");
    }

    /**
     * 判断字符串是否不为空
     * @param str 字符串
     * @return boolean
     */
    public static boolean notEmpty(String str){
        return !isEmpty(str);
    }

    /**
     * 如果字符串为空，则返回默认值
     * @param orig 字符串
     * @param def 默认值
     * @return String
     */
    public static String ifEmpty(String orig, String def){
        return isEmpty(orig) ? def : orig;
    }

    /**
     * 判断字符串数组中是否含有空元素
     * @param paras 字符数组
     * @return boolean
     */
    public static boolean hasEmpty(String...paras){
        if(null == paras) return true;
        return Arrays.stream(paras).anyMatch(JStr::isEmpty);
    }

    /**
     * 判断字符串数组中是否全部为空元素
     * @param paras 字符数组
     * @return boolean
     */
    public static boolean allEmpty(String...paras){
        if(null == paras) return true;
        return Arrays.stream(paras).allMatch(JStr::isEmpty);
    }

    //*******************************************  字符串截取  ***********************************************/

    /**
     * 截取字符串的前几位字符
     * @param str 字符串
     * @param count 字符个数
     * @return String
     */
    public static String left(String str, int count){
        if(isEmpty(str)) return "";
        int len = str.length();
        if(len < count) return str;
        return str.substring(0, count);
    }

    /**
     * 截取字符串的后几位字符
     * @param str 字符串
     * @param count 字符个数
     * @return String
     */
    public static String right(String str, int count){
        if(isEmpty(str)) return "";
        int len = str.length();
        if(len < count) return str;
        return str.substring(len - count);
    }

    /**
     * 数字左侧补0
     * @param number 数字
     * @param len 位数
     * @return String
     */
    public static String addPrefix(Integer number, int len){
        if(isNull(number)) number = 0;
        return addPrefix(number + "", len);
    }

    /**
     * 字符串左侧补0
     * @param str 字符串
     * @param len 位数
     * @return String
     */
    public static String addPrefix(String str, int len){
        return addPrefix(str, len, "0000000000");
    }

    /**
     * 字符串左侧补齐指定字符
     * @param str 字符串
     * @param len 位数
     * @param prefix 前缀
     * @return String
     */
    public static String addPrefix(String str, int len, String prefix){
        if(isNull(str)) str = "";
        str = prefix + str;
        return right(str, len);
    }

    /**
     * 首字母转小写
     * @param s 字符串
     * @return String
     */
    public static String toLowerCaseFirst(String s) {
        if(Character.isLowerCase(s.charAt(0))){
            return s;
        } else {
            return Character.toLowerCase(s.charAt(0)) + s.substring(1);
        }
    }

    /**
     * 首字母转大写
     * @param s 字符串
     * @return String
     */
    public static String toUpperCaseFirst(String s) {
        if(Character.isUpperCase(s.charAt(0))){
            return s;
        } else {
            return Character.toUpperCase(s.charAt(0)) + s.substring(1);
        }
    }

    //*******************************************  字符串拼接  ***********************************************/
    
    /**
     * <p>
     * 字符串拼接
     * </p>
     * 
     * @param param
     * @return
     */
    public static String concat(String...params){
        if(isNull(params)) return "";
        return String.join("", params);
    }

    //*******************************************  列表处理  ***********************************************/

    /**
     * 将字符串列表转换为逗号间隔的字符串
     * @param list  字符串列表
     * @return String
     */
    public static String join(List<String> list){
        if(isNull(list)) return "";
        return join(list.toArray(new String[0]));
    }

    /**
     * 将字符串数组转换为逗号间隔的字符串
     * @param arr  字符串数组
     * @return String
     */
    public static String join(String[] arr){
        return join(arr, ",");
    }

    /**
     * 将字符串数组转换为指定字符间隔的字符串
     * @param arr  字符串数组
     * @param sept 分隔符
     * @return String
     */
    public static String join(String[] arr, String sept){
        if(isNull(arr)) return "";
        return String.join(sept, arr);
    }

    /**
     * 将字符串按逗号转换为数组
     * @param str 字符串
     * @return String[]
     */
    public static String[] toArray(String str){
        return toArray(str, ",");
    }

    /**
     * 将字符串按指定字符转换为数组
     * @param str 字符串
     * @param sept 分隔符
     * @return String[]
     */
    public static String[] toArray(String str, String sept){
        if(isNull(str)) str = "";
        return str.trim().split(sept);
    }

    /**
     * 将字符串按逗号转换为列表
     * @param str 字符串
     * @return List
     */
    public static List<String> toList(String str){
        return toList(str, ",");
    }

    /**
     * 将字符串按指定字符转换为列表
     * @param str 字符串
     * @param sept 分隔符
     * @return List
     */
    public static List<String> toList(String str, String sept){
        String[] arr = toArray(str, sept);
        return Arrays.asList(arr);
    }

    //*******************************************  查找匹配  ***********************************************/

    /**
     * 从字符串数组中返回第一个非空元素
     * @param s 字符串
     * @return String
     */
    public static String findUseful(String...s){
        if(isNull(s)) return null;
        return Arrays.stream(s).filter(JStr::notEmpty).findFirst().orElse(null);
    }

    /**
     * 在数组中查找，返回匹配元素的索引
     * @param arr 字符串数组
     * @param keyword 关键字
     * @return String
     */
    public static int findInArr(String[] arr, String keyword){
        if(isNull(arr)) return -1;
        for(int i=0, len=arr.length; i<len; i++){
            String item = arr[i];
            if(notNull(item) && item.equals(keyword)) return i;
        }
        return -1;
    }

    //*******************************************  类型转换  ***********************************************/

    /**
     * 字符串转Byte
     * @param str 字符串
     * @return byte[]
     */
    public static byte[] toByte(String str){
        return toByte(str, UTF_8);
    }
    
    /**
     * <p>
     * 字符串转Byte
     * </p>
     * 
     * @param str 字符串
     * @param charset 字符集
     * @return byte[]
     */
    public static byte[] toByte(String str, Charset charset){
        if(isNull(str)) return null;
        return str.getBytes(charset);
    }

    /**
     * 将Byte转字符串
     * @param bytes 字节
     * @return String
     */
    public static String parse(byte[] bytes){
        return parse(bytes, UTF_8);
    }
    
    /**
     * <p>
     * 将Byte转字符串
     * </p>
     * 
     * @param bytes 字节
     * @param charset 字符集
     * @return
     */
    public static String parse(byte[] bytes, Charset charset){
        if(isNull(bytes)) return null;
        return new String(bytes, charset);
    }

    //*******************************************  随机数  ***********************************************/

    /**
     * 生产随机数
     * @param min 最小值
     * @param max 最大值
     * @return Integer
     */
    public static Integer rand(Integer min, Integer max){
        Random rand = new Random();
        return rand.nextInt(max - min + 1) + min;
    }

    //*******************************************  SQL IN ID 处理  ***********************************************/

    /**
     * 将参数数组处理为逗号间隔的、两边加单引号的SQL IN检索格式
     * @param ida ID数组
     * @return 'aa', 'bb', 'cc', ...
     */
    public static String ida2sqlIn(Object...ida){
        if(isNull(ida)) ida = new String[]{""};
        return Arrays.stream(ida).map(id -> String.format("'%s'", id.toString().trim())).collect(Collectors.joining(","));
    }

    /**
     * 将普通字符串处理为逗号间隔的、两边加单引号的SQL IN检索格式
     * @param ids ID字符串
     * @return 'aa', 'bb', 'cc', ...
     */
    public static String ids2sqlIn(String ids){
        if(isNull(ids)) ids = "";
        return String.format("'%s'", ids.replace(",", "','"));
    }
    
}
