/**
 * 
 */
package itez.jwinner.kit;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;

/**
 * <p>
 * 日期工具类
 * </p>
 * 
 * <p>Copyright(C) 2017-2020 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2020年4月20日 下午9:39:25
 */
public class JDate {

	public static final String PATTERN_Y = "yyyy"; // 年
	public static final String PATTERN_YM = "yyyy-MM"; // 年-月
	public static final String PATTERN_MD = "MM-dd"; // 月-日
	public static final String PATTERN_YMD = "yyyy-MM-dd"; // 年-月-日
	public static final String PATTERN_YMD_HM = "yyyy-MM-dd HH:mm"; // 年-月-日 时：分
	public static final String PATTERN_HM = "HH:mm"; // 时：分
	public static final String PATTERN_YMD_HMS = "yyyy-MM-dd HH:mm:ss"; // 年-月-日 时：分：秒
	public static final String PATTERN_YMD_HMS_S = "yyyy-MM-dd HH:mm:ss:SSS"; // 年-月-日 时：分：秒：毫秒

    //*******************************************  返回当前日期时间   ***********************************************/

	/**
	 * <p>
	 * 获取当前时间
	 * </p>
	 * 
	 * @return Date
	 */
	public static Date getDate() {
		return new Date();
	}

	/**
	 * <p>
	 * 获得当前的时间戳
	 * </p>
	 * 
	 * @return long
	 */
	public static long getTime() {
		return new Date().getTime();
	}

    //*******************************************  字符串到日期型转换   ***********************************************/

	/**
	 * <p>
	 * 日期字符串类型转为日期类型
	 * </p>
	 * 
	 * @param date 日期字符串
	 * @return Date
	 */
	public static Date parse(String date){
		String pattern = parsePattern(date);
		return JStr.isEmpty(pattern) ? getDate() : parse(date, pattern);
	}

	/**
	 * <p>
	 * 日期字符串类型转为日期类型
	 * </p>
	 * 
	 * @param date 日期字符串
	 * @param pattern 格式化模板
	 * @return Date
	 */
	public static Date parse(String date, String pattern) {
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		try {
			return format.parse(date);
		} catch (Exception e) {
			return getDate();
		}
	}

    //*******************************************  Date与LocalDateTime转换   ***********************************************/

	/**
	 * <p>
	 * Date转LocalDate
	 * </p>
	 * 
	 * @param date Date
	 * @return LocalDate
	 */
	public static LocalDate date2localDate(Date date){
		Instant instant = date.toInstant();
		ZoneId zoneId  = ZoneId.systemDefault();
		return instant.atZone(zoneId).toLocalDate();
	}
	
	/**
	 * <p>
	 * Date转LocalTime
	 * </p>
	 * 
	 * @param date Date
	 * @return LocalTime
	 */
	public static LocalTime date2localTime(Date date){
		Instant instant = date.toInstant();
		ZoneId zoneId  = ZoneId.systemDefault();
		return instant.atZone(zoneId).toLocalTime();
	}
	
	/**
	 * <p>
	 * Date转LocalDateTime
	 * </p>
	 * 
	 * @param date Date
	 * @return LocalDateTime
	 */
	public static LocalDateTime date2localDateTime(Date date){
		Instant instant = date.toInstant();
		ZoneId zoneId  = ZoneId.systemDefault();
		return instant.atZone(zoneId).toLocalDateTime();
	}
	
	/**
	 * <p>
	 * LocalDate转Date
	 * </p>
	 * 
	 * @param localDate LocalDate
	 * @return Date
	 */
	public static Date localDate2date(LocalDate localDate){
		ZoneId zoneId = ZoneId.systemDefault();
		Instant instant = localDate.atStartOfDay(zoneId).toInstant();
		return Date.from(instant);
	}
	
	/**
	 * <p>
	 * LocalTime转Date
	 * </p>
	 * 
	 * @param localTime LocalTime
	 * @return Date
	 */
	public static Date localTime2date(LocalTime localTime){
		LocalDate localDate = LocalDate.now();
		LocalDateTime localDateTime = LocalDateTime.of(localDate, localTime);
		return localDateTime2date(localDateTime);
	}
	
	/**
	 * <p>
	 * localDateTime转Date
	 * </p>
	 * 
	 * @param localDateTime LocalDateTime
	 * @return Date
	 */
	public static Date localDateTime2date(LocalDateTime localDateTime){
		ZoneId zoneId = ZoneId.systemDefault();
		Instant instant = localDateTime.atZone(zoneId).toInstant();
		return Date.from(instant);
	}

    //*******************************************  日期格式化   ***********************************************/

	/**
	 * <p>
	 * 日期格式化
	 * </p>
	 * 
	 * @param date 日期
	 * @return 日期字符串
	 */
	public static String format(Date date) {
		return format(date, PATTERN_YMD);
	}
	
	/**
	 * <p>
	 * 日期格式化
	 * </p>
	 * 
	 * @param date 日期
	 * @param pattern 格式化模板
	 * @return 日期字符串
	 */
	public static String format(Date date, String pattern) {
		DateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}

    //*******************************************  返回日期组成部分   ***********************************************/
	
	public static Integer getYear(Date date){
		LocalDate localDate = date2localDate(date);
		return localDate.getYear();
	}

	public static Integer getMonth(Date date){
		LocalDate localDate = date2localDate(date);
		return localDate.getMonthValue();
	}

	public static Integer getDay(Date date){
		LocalDate localDate = date2localDate(date);
		return localDate.getDayOfMonth();
	}
	
	public static Integer getDayOfYear(Date date){
		LocalDate localDate = date2localDate(date);
		return localDate.getDayOfYear();
	}
	
	public static Integer getDayOfWeek(Date date){
		LocalDate localDate = date2localDate(date);
		return localDate.getDayOfWeek().getValue();
	}

	public static Integer getHour(Date date){
		LocalTime localTime = date2localTime(date);
		return localTime.getHour();
	}

	public static Integer getMinute(Date date){
		LocalTime localTime = date2localTime(date);
		return localTime.getMinute();
	}

	public static Integer getSecond(Date date){
		LocalTime localTime = date2localTime(date);
		return localTime.getSecond();
	}

    //*******************************************  日期比较   ***********************************************/

	/**
	 * <p>
	 * 日期间隔
	 * </p>
	 * 
	 * @param start 起始日期
	 * @param end 截止日期
	 * @return Duration
	 */
	public static Duration diff(Date start, Date end){
		LocalDateTime ld1 = date2localDateTime(start);
		LocalDateTime ld2 = date2localDateTime(end);
		Duration duration = Duration.between(ld1, ld2);
		return duration;
	}
	
	/**
	 * <p>
	 * 两个日期是否由小到大
	 * </p>
	 * 
	 * @param start
	 * @param end
	 * @return
	 */
	public static boolean isAsc(Date start, Date end){
		LocalDateTime ld1 = date2localDateTime(start);
		LocalDateTime ld2 = date2localDateTime(end);
		return ld1.isBefore(ld2);
	}
	
	/**
	 * <p>
	 * 两个日期是否由大到小
	 * </p>
	 * 
	 * @param start
	 * @param end
	 * @return
	 */
	public static boolean isDesc(Date start, Date end){
		LocalDateTime ld1 = date2localDateTime(start);
		LocalDateTime ld2 = date2localDateTime(end);
		return ld1.isAfter(ld2);
	}
	
	/**
	 * <p>
	 * 指定日期是否已超期
	 * </p>
	 * 
	 * @param date
	 * @return
	 */
	public static boolean isExpire(Date date){
    	if (date.before(getDate())) return true;
    	return false;
	}

    //*******************************************  日期计算   ***********************************************/

	public static Date addYear(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusYears(num);
		return localDateTime2date(localDateTime);
	}

	public static Date addMonth(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusMonths(num);
		return localDateTime2date(localDateTime);
	}

	public static Date addDay(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusDays(num);
		return localDateTime2date(localDateTime);
	}
	
	public static Date addWeek(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusWeeks(num);
		return localDateTime2date(localDateTime);
	}
	
	public static Date addHour(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusHours(num);
		return localDateTime2date(localDateTime);
	}
	
	public static Date addMinute(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusMinutes(num);
		return localDateTime2date(localDateTime);
	}
	
	public static Date addSecond(Date date, Integer num){
		LocalDateTime localDateTime = date2localDateTime(date);
		localDateTime.plusSeconds(num);
		return localDateTime2date(localDateTime);
	}

    //*******************************************  其他   ***********************************************/
	
	/**
	 * <p>
	 * 返回日期所在月份的月初和月末两天日期
	 * </p>
	 * 
	 * @param date
	 * @return
	 */
	public static Date[] getMonthScope(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);

		// 得到前一个月的第一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
		Date start = calendar.getTime();

		// 得到前一个月的最后一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
		Date end = calendar.getTime();

		return new Date[]{start, end};
	}

    //*******************************************  私有方法   ***********************************************/

	/**
	 * <p>
	 * 自动判断指定字符串与相应日期时间格式匹配的模板
	 * </p>
	 * <p>
	 * 无法匹配时，返回null
	 * </p>
	 * 
	 * @param date 日期字符串
	 * @return String
	 */
	private static String parsePattern(String date){
		String pattern = null;
		if(JRegex.has(date, "^\\d{4}$")){// 年
			pattern = PATTERN_Y;
		}else if(JRegex.has(date, "^\\d{4}\\-\\d{1,2}$")){// 年-月
			pattern = PATTERN_YM;
		}else if(JRegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}$")){// 年-月-日
			pattern = PATTERN_YMD;
		}else if(JRegex.has(date, "^\\d{1,2}\\-\\d{1,2}$")){// 月-日
			pattern = PATTERN_MD;
		}else if(JRegex.has(date, "^\\d{1,2}\\:\\d{1,2}$")){// 时：分
			pattern = PATTERN_HM;
		}else if(JRegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分
			pattern = PATTERN_YMD_HM;
		}else if(JRegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分：秒
			pattern = PATTERN_YMD_HMS;
		}else if(JRegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}\\:\\d{1,3}$")){// 年-月-日 时：分：秒：毫秒
			pattern = PATTERN_YMD_HMS_S;
		}
		return pattern;
	}
	
}
