package itez.tp.impl.weixin.mp;

import itez.core.util.RetryUtils;
import itez.kit.EHttp;
import itez.kit.restful.EMap;
import itez.tp.ApiCommon;
import itez.tp.ApiCommon.TP;
import itez.tp.ApiConf;
import java.util.concurrent.Callable;

/**
 * 认证并获取 access_token API
 * http://mp.weixin.qq.com/wiki/index.php?title=%E8%8E%B7%E5%8F%96access_token
 *
 * AccessToken默认存储于内存中，可设置存储于redis或者实现IAccessTokenCache到数据库中实现分布式可用
 *
 * 具体配置：
 * <pre>
 * ApiConfigKit.setAccessTokenCache(new RedisAccessTokenCache());
 * </pre>
 */
public class AccessTokenApi {

    // "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential&appid=APPID&secret=APPSECRET";
    private static final String url = "https://api.weixin.qq.com/cgi-bin/token?grant_type=client_credential";

    /**
     * 从缓存中获取 access token，如果未取到或者 access token 不可用则先更新再获取
     * @return AccessToken accessToken
     */
    public static AccessToken getAccessToken() {
    	ApiConf api = ApiCommon.getApi(TP.WXMP);
    	String appId = api.getVal("appId");
    	String appSecret = api.getVal("appSecret");
    	AccessToken token = getAvailableAccessToken(appId);
        if (token != null) return token;
        return refreshAccessTokenIfNecessary(appId, appSecret);
    }

    private static AccessToken getAvailableAccessToken(String appId) {
    	AccessToken token = Common.getToken(appId);
        if (token != null && token.isAvailable()) return token;
        return null;
    }

    /**
     * 直接获取 accessToken 字符串，方便使用
     * @return String accessToken
     */
    public static String getAccessTokenStr() {
        return getAccessToken().getAccessToken();
    }

    /**
     * synchronized 配合再次获取 token 并检测可用性，防止多线程重复刷新 token 值
     * @param appId
     * @param appSecret
     * @return
     */
    private static synchronized AccessToken refreshAccessTokenIfNecessary(String appId, String appSecret) {
        AccessToken token = getAvailableAccessToken(appId);
        if (token != null) return token;
        return refreshAccessToken(appId, appSecret);
    }

    /**
     * 无条件强制更新 access token 值，不再对 cache 中的 token 进行判断
     * @param appId
     * @param appSecret
     * @return
     */
    public static AccessToken refreshAccessToken(String appId, String appSecret) {
        // 最多三次请求
        final EMap paras = EMap.by("appid", appId).set("secret", appSecret);
        AccessToken token = RetryUtils.retryOnException(3, new Callable<AccessToken>() {
			@Override
            @SuppressWarnings("unchecked")
            public AccessToken call() throws Exception {
            	String json = EHttp.me.get(url, paras);
                return new AccessToken(json);
            }
        });
        if (null != token) Common.setToken(appId, token);
        return token;
    }

}
