package itez.plat.site.service.impl;

import itez.core.runtime.service.EModelService;
import itez.core.wrapper.dbo.model.Query;
import itez.core.wrapper.dbo.model.Querys;
import itez.kit.EArr;
import itez.kit.EFile;
import itez.kit.EJson;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.EUid;
import itez.kit.log.ExceptionUtil;
import itez.kit.restful.EMap;
import itez.kit.zip.ZipKit;
import itez.core.runtime.service.Define;
import itez.plat.site.model.Backup;
import itez.plat.site.model.Channel;
import itez.plat.site.model.Content;
import itez.plat.site.model.Info;
import itez.plat.site.model.Navi;
import itez.plat.site.model.NaviItem;
import itez.plat.site.model.Tags;
import itez.plat.site.service.BackupService;
import itez.plat.site.service.ChannelService;
import itez.plat.site.service.ContentService;
import itez.plat.site.service.InfoService;
import itez.plat.site.service.NaviItemService;
import itez.plat.site.service.NaviService;
import itez.plat.site.service.TagsService;

import java.io.File;
import java.io.IOException;
import java.util.List;

import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.jfinal.plugin.activerecord.Record;

/**
 * 由JWinner Service Generator自动生成。
 */
@Define
@Singleton
public class BackupServiceImpl extends EModelService<Backup> implements BackupService {
	
	@Inject
	InfoService infoSer;
	
	@Inject
	ChannelService chnSer;
	
	@Inject
	TagsService tagSer;
	
	@Inject
	NaviService naviSer;
	
	@Inject
	NaviItemService naviItemSer;
	
	@Inject
	ContentService contentSer;
	
	@Override
	public Backup getByCaption(String caption) {
		Querys qs = Querys.and(Query.eq("caption", caption));
		return selectFirst(qs);
	}

	/**
	 * <p>
	 * 创建新备份
	 * </p>
	 * 
	 * @param bak 备份记录
	 * @param domain 子域
	 */
	@Override
	public void create(Backup bak) {
		//创建备份记录
		save(bak);
		
		//开启备份线程
		new Thread(new Runnable() {
			@Override
			public void run() {
				createEvent(bak);
			}
		}).start();
	}

	/**
	 * <p>
	 * 创建新备份（线程）
	 * </p>
	 * 
	 * @param bak 备份记录
	 * @param domain 子域
	 */
	private void createEvent(Backup bak){
		try {
			BakDirs dirs = new BakDirs(bak.getDomain());
			dirs.initDirs();
			BakAble able = new BakAble(bak);
			backup_cfg(bak, dirs);				//备份描述文件
			backup_data(bak, dirs, able);		//网站数据
			backup_temp(bak, dirs, able);		//网站模板
			backup_res(bak, dirs, able);		//网站资源文件
			backup_upload(bak, dirs, able);		//网站上传文件
			update(bak.setBakDirName(dirs.getBakDirName()).setState(1));	//更新备份状态
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
		}
	}
	
	/**
	 * <p>
	 * 备份描述文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @throws IOException
	 */
	private void backup_cfg(Backup bak, BakDirs dirs) throws IOException{
		File bakFile = new File(dirs.getBakRoot().concat("/backup.json"));
		bakFile.createNewFile();
		bak.setBakDirName(dirs.getBakDirName());
		EFile.write(bakFile, EJson.toJson(bak));
	}
	
	/**
	 * <p>
	 * 网站备份：基础数据
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @throws IOException
	 */
	private void backup_data(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnData()) return;
		String domain = bak.getDomain();
		
		//基础信息
		if(able.isEnInfo()){
			Info info = infoSer.getInfo(domain);
			File infoFile = new File(dirs.getBakDataRoot().concat("/info.json"));
			infoFile.createNewFile();
			EFile.write(infoFile, EJson.toJson(info));
		}
		//栏目
		if(able.isEnChannel()){
			List<Record> chns = chnSer.getBackChannels(domain);
			File channelFile = new File(dirs.getBakDataRoot().concat("/channel.json"));
			channelFile.createNewFile();
			EFile.write(channelFile, EJson.toJson(chns));
		}
		//标签
		if(able.isEnTag()){
			List<Tags> tags = tagSer.getTags(domain);
			File tagFile = new File(dirs.getBakDataRoot().concat("/tag.json"));
			tagFile.createNewFile();
			EFile.write(tagFile, EJson.toJson(tags));
		}
		//导航
		if(able.isEnNav()){
			List<Record> navis = naviSer.getBackNavis(domain);
			File naviFile = new File(dirs.getBakDataRoot().concat("/navi.json"));
			naviFile.createNewFile();
			EFile.write(naviFile, EJson.toJson(navis));
			List<Record> naviItems = naviItemSer.getBackNavis(domain);
			File naviItemFile = new File(dirs.getBakDataRoot().concat("/naviItem.json"));
			naviItemFile.createNewFile();
			EFile.write(naviItemFile, EJson.toJson(naviItems));
		}
		//文章
		if(able.isEnContent()){
			Integer years = Integer.parseInt(bak.getDataYear());
			List<Record> cons = contentSer.getBetweenYears(domain, years, years);
			File conFile = new File(dirs.getBakDataRoot().concat("/content.json"));
			conFile.createNewFile();
			EFile.write(conFile, EJson.toJson(cons));
		}
	}
	
	/**
	 * <p>
	 * 网站备份：模板文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @throws IOException
	 */
	private void backup_temp(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnTemp()) return;
		File srcFolder = new File(dirs.getSrcTempRoot());
		File distFolder = new File(dirs.getBakTempRoot());
		EFile.copyDir(srcFolder, distFolder);
	}
	
	/**
	 * <p>
	 * 网站备份：资源文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @throws IOException
	 */
	private void backup_res(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnRes()) return;
		File srcFolder = new File(dirs.getSrcResRoot());
		File distFolder = new File(dirs.getBakResRoot());
		EFile.copyDir(srcFolder, distFolder);
	}
	
	/**
	 * <p>
	 * 网站备份：上传文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @throws IOException
	 */
	private void backup_upload(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnUpload()) return;
		String years = EStr.FileSep + bak.getDataYear();
		File srcFolder = new File(dirs.getSrcFileRoot() + years);
		if(!srcFolder.exists()) return;
		File distFolder = new File(dirs.getBakFileRoot() + years);
		EFile.copyDir(srcFolder, distFolder);
	}
	
	/**
	 * 删除备份（批量）
	 */
	@Override
	public void removeBak(String ids) {
		 List<Backup> baks = findByIds(ids);
		 baks.forEach(bak -> removeBakItem(bak));
	}
	
	/**
	 * <p>
	 * 删除备份（单条）
	 * </p>
	 * 
	 * @param bak
	 */
	private void removeBakItem(Backup bak){
		BakDirs dirs = new BakDirs(bak.getDomain(), bak.getBakDirName());
		String bakDir = dirs.getBakRoot();
		EFile.deleteDir(bakDir);

		String zipPath = bakDir.concat(".zip");
		File zipFile = new File(zipPath);
		if(zipFile.exists()) zipFile.delete();
		
		deleteById(bak.getId());
	}
	
	/**
	 * 压缩备份文件夹并下载
	 */
	@Override
	public File zip(String id) {
		Backup bak = findById(id);
		BakDirs dirs = new BakDirs(bak.getDomain(), bak.getBakDirName());
		String bakRoot = dirs.getBakRoot();
		String zipPath = bakRoot.concat(".zip");
		File zipFile = new File(zipPath);
		if(zipFile.exists()) return zipFile;
		ZipKit.me.zip(bakRoot, null);
		return new File(zipPath);
	}
	
	/**
	 * 上传备份文件
	 */
	@Override
	public void upload(File file) throws IOException {
		String parentPath = file.getParent();
		String fileNameAll = file.getName();
		String fileName = fileNameAll.split("\\.")[0];
		EFile.unzip(file.getPath());
		
		File parentDir = new File(parentPath.concat(EStr.FileSep).concat(fileName));
		File[] files = parentDir.listFiles();
		if(files == null || files.length > 1) {
			file.delete();
			EFile.deleteDir(parentDir.getPath());
			throw new RuntimeException("备份文件结构错误！");
		}
		
		File bakDir = files[0];
		String bakDirStr = bakDir.getPath().concat(EStr.FileSep);
		
		File bakFile = new File(bakDirStr.concat("backup.json"));
		if(!bakFile.exists()){
			file.delete();
			EFile.deleteDir(parentDir.getPath());
			throw new RuntimeException("未发现备份描述文件！");
		}
		
		String bakJson = EFile.read(bakFile);
		Backup bak = JSON.parseObject(bakJson, Backup.class);
		
		Backup bakExist = getByCaption(bak.getCaption());
		if(bakExist != null){
			file.delete();
			EFile.deleteDir(parentDir.getPath());
			throw new RuntimeException("备份记录已经存在，请不要重复上传！");
		}

		BakDirs dirs = new BakDirs($domain(), bak.getBakDirName());
		String bakRoot = dirs.getBakRoot();
		File bakRootFolder = new File(bakRoot);
		if(bakRootFolder.exists()){
			file.delete();
			EFile.deleteDir(parentDir.getPath());
			throw new RuntimeException("备份目录已经存在，请不要重复上传！");
		}else{
			bakRootFolder.mkdirs();
		}
		
		EFile.copyDir(bakDir, bakRootFolder);
		EFile.copyDir(file, new File(bakRoot.concat(".zip")));
		
		EFile.deleteDir(parentDir.getPath());
		file.delete();
		
		bak.remove("id", "domain", "cdate", "mdate");
		bak.setState(1);
		save(bak);
	}
	
	@Override
	public void restore(EMap paras) {
		Backup bak = findById(paras.getStr("id"));
		
		//设置还原状态
		update(bak.setState(0));

		//开启还原线程
		new Thread(new Runnable() {
			@Override
			public void run() {
				restoreEvent(bak, paras);
			}
		}).start();
	}
	
	/**
	 * <p>
	 * 执行网站还原（线程）
	 * </p>
	 * 
	 * @param bak 备份记录
	 * @param chnRep 是否覆盖栏目
	 * @param conRep 是否覆盖文章
	 * @param tagRep 是否覆盖标签
	 * @param navRep 是否覆盖导航
	 */
	private void restoreEvent(Backup bak, EMap paras){
		try {
			BakDirs dirs = new BakDirs(bak.getDomain(), bak.getBakDirName());
			dirs.initDirs();
			BakAble able = new BakAble(paras);
			restore_data(bak, dirs, able);			//网站数据
			restore_temp(bak, dirs, able);			//网站模板
			restore_res(bak, dirs, able);			//网站资源文件
			restore_upload(bak, dirs, able);		//网站上传文件
			update(bak.setState(1));				//更新还原状态
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
			ELog.error(ExceptionUtil.getMessage(e));
		}
	}
	
	/**
	 * <p>
	 * 还原网站数据
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param able
	 */
	private void restore_data(Backup bak, BakDirs dirs, BakAble able){
		if(!able.isEnData()) return;
		EMap chnIdAll = EMap.create();									//栏目ID汇总（原有、新增）用于在还原文章、导航时判断对应栏目是否存在
		EMap chnIdChgs = EMap.create();									//备份栏目的ID变化记录（原ID -> 新ID）
		restore_data_info(bak, dirs, able);								//基础信息
		restore_data_channel(bak, dirs, able, chnIdAll, chnIdChgs);		//栏目
		restore_data_content(bak, dirs, able, chnIdAll, chnIdChgs);		//文章
		restore_data_nav(bak, dirs, able, chnIdAll, chnIdChgs);			//导航
		restore_data_tag(bak, dirs, able);								//标签
	}
	
	/**
	 * <p>
	 * 还原网站数据：基础信息
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 */
	private void restore_data_info(Backup bak, BakDirs dirs, BakAble able){
		if(!able.isEnInfo()) return;
		File infoFile = new File(dirs.getBakDataRoot().concat("/info.json"));
		if(!infoFile.exists()) return;
		String infoStr = EFile.read(infoFile);
		if(EStr.isEmpty(infoStr)) return;

		String domain = bak.getDomain();
		Info info = JSON.parseObject(infoStr, Info.class);
		info.keep("caption", "subCaption", "welcome");	//基础信息仅还原标题、副标题、欢迎词三项数据
		Info infoExist = infoSer.getInfo(domain);
		infoExist._setAttrs(info);
		infoSer.modifyInfo(infoExist);
	}
	
	/**
	 * <p>
	 * 还原网站数据：栏目
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param chnRep
	 * @param chnIdChgs
	 */
	private void restore_data_channel(Backup bak, BakDirs dirs, BakAble able, EMap chnIdAll, EMap chnIdChgs){
		//获取现有栏目列表
		String domain = bak.getDomain();
		List<Record> chnsExist = chnSer.getBackChannels(domain);
		
		//为了便于查找现有栏目，准备两个字典表
		EMap idMap = EMap.create(), codeMap = EMap.create();
		for(int i = 0, len = chnsExist.size(); i < len; i++){
			Record chn = chnsExist.get(i);
			chnIdAll.set(chn.getStr("id"), true);
			idMap.set(chn.getStr("id"), i);
			codeMap.set(chn.getStr("code"), i);
		}
		
		//解析备份文件
		File channelFile = new File(dirs.getBakDataRoot().concat("/channel.json"));
		if(!channelFile.exists()) return;
		String chnStr = EFile.read(channelFile);
		if(EStr.isEmpty(chnStr)) return;
		List<Channel> chnsBak = JSON.parseArray(chnStr, Channel.class);
		
		//提前遍历一次备份栏目，处理别名相同的情况
		//适用于跨站还原文章时，未选择还原栏目，可以根据栏目别名智能匹配的特性
		chnsBak.forEach(chnBak -> {
			String bakId = chnBak.getId();
			String bakCode = chnBak.getCode();
			if(!idMap.containsKey(bakId) && codeMap.containsKey(bakCode)){
				Record chn = chnsExist.get(codeMap.getAs(bakCode));
				chnIdChgs.set(bakId, chn.getStr("id"));
			}
		});
		
		//不还原栏目，直接跳出
		if(!able.isEnChannel()) return;
		
		//准备两个列表，分别存储新增加的栏目和覆盖修改的栏目
		List<Record> chnAdd = Lists.newArrayList();
		List<Record> chnChg = Lists.newArrayList();
		
		//栏目覆盖的字段列表
		String[] keeps = {"pid", "code", "icon", "pic", "caption", "subCaption", "content", "channelTemp", "contentTemp", "sort"};

		//遍历备份栏目列表
		chnsBak.forEach(chnBak -> {
			String bakId = chnBak.getId();
			String bakPid = chnBak.getPid();
			String bakCode = chnBak.getCode();
			chnBak.keep(keeps);
			//如果备份栏目的上级栏目ID变更，则同步子栏目的父栏目ID
			if(EStr.notEmpty(bakPid) && chnIdChgs.containsKey(bakPid)){
				bakPid = chnIdChgs.getStr(bakPid);
				chnBak.setPid(bakPid);
			}
			//如果栏目ID或别名已存在，则覆盖现有栏目
			if(idMap.containsKey(bakId) || codeMap.containsKey(bakCode)){
				if(!able.isEnChannelRep()) return;
				Record chn;
				if(idMap.containsKey(bakId)){ //栏目ID相同，仅覆盖即可
					chn = chnsExist.get(idMap.getAs(bakId));
				}else{ //栏目ID不同，但别名相同，则视为相同栏目，可继续使用已有栏目ID，但需要将备份栏目ID手动调整为已有栏目ID，否则备份栏目的树型关系将出现错误
					chn = chnsExist.get(codeMap.getAs(bakCode));
				}
				chn.setColumns(chnBak);
				chnChg.add(chn);
			}else{ //如果栏目ID及别名都不存在，则添加新栏目
				String nid = EUid.generator();
				chnIdAll.set(nid, false);
				chnIdChgs.set(bakId, nid);
				Record chn = new Record();
				chn.set("id", nid).set("domain", domain).set("subDomain", "").set("used", 1);
				chn.setColumns(chnBak);
				chnAdd.add(chn);
			}
		});
		
		//执行事务
		dbo().tx(() -> {
			int[] empty = new int[]{1};
			int[] ret1 = chnChg.size() > 0 ? dbo().batchUpdate("site_channel", chnChg, chnChg.size()) : empty;
			int[] ret2 = chnAdd.size() > 0 ? dbo().batchSave("site_channel", chnAdd, chnAdd.size()) : empty;
			return EArr.vali(ret1, ret2);
		});
		
		//重新生成栏目树型（path）
		chnSer.genChannelPath(domain);
	}

	/**
	 * <p>
	 * 还原网站数据：文章
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param conRep
	 * @param chnIdChgs
	 */
	private void restore_data_content(Backup bak, BakDirs dirs, BakAble able, EMap chnIdAll, EMap chnIdChgs){
		if(!able.isEnContent()) return;
		File contentlFile = new File(dirs.getBakDataRoot().concat("/content.json"));
		if(!contentlFile.exists()) return;
		String conStr = EFile.read(contentlFile);
		if(EStr.isEmpty(conStr)) return;
		
		//获取现有文章列表
		String domain = bak.getDomain();
		Integer years = Integer.parseInt(bak.getDataYear());
		List<Record> consExist = contentSer.getBetweenYears(domain, years, years);

		//为了便于查找现有文章，准备字典表
		EMap idMap = EMap.create();
		for(int i = 0, len = consExist.size(); i < len; i++){
			Record con = consExist.get(i);
			idMap.set(con.getStr("id"), i);
		}
		
		//准备两个列表，分别存储新增加的文章和覆盖修改的文章
		List<Record> conAdd = Lists.newArrayList();
		List<Record> conChg = Lists.newArrayList();
		
		//文章覆盖的字段列表
		String[] keeps = {"channelId", "channelCaption", "tagCodes", "tagCaps", "pic", "thum", "caption", "subCaption", "summary", "author", "content", "captionColor", "link", "cdate", "mdate", "sort"};
		
		//遍历备份栏目列表
		List<Content> consBak = JSON.parseArray(conStr, Content.class);
		consBak.forEach(conBak -> {
			String bakId = conBak.getId();
			String bakChnId = conBak.getChannelId();
			//文章所属栏目为空时直接跳过
			if(EStr.isEmpty(bakChnId)) return;
			//如果文章所属栏目ID发生变更，则同步为新的栏目ID
			if(chnIdChgs.containsKey(bakChnId)){
				bakChnId = chnIdChgs.getStr(bakChnId);
				conBak.setChannelId(bakChnId);
			}
			//文章所属栏目不存在时直接跳过
			if(!chnIdAll.containsKey(bakChnId)) return;
			conBak.keep(keeps);
			//如果文章ID已存在，则覆盖现有文章
			if(idMap.containsKey(bakId)){
				if(!able.isEnContentRep()) return;
				Record con = consExist.get(idMap.getAs(bakId));
				con.setColumns(conBak).remove("channelCode");
				conChg.add(con);
			}else{ //如果文章ID不存在，则添加新文章
				String nid = EUid.generator();
				Record con = new Record();
				con.set("id", nid).set("domain", domain).set("subDomain", "").set("used", 1);
				con.setColumns(conBak).remove("channelCode");
				conAdd.add(con);
			}
		});
		
		//执行事务
		dbo().tx(() -> {
			int[] empty = new int[]{1};
			int[] ret1 = conChg.size() > 0 ? dbo().batchUpdate("site_content", conChg, conChg.size()) : empty;
			int[] ret2 = conAdd.size() > 0 ? dbo().batchSave("site_content", conAdd, conAdd.size()) : empty;
			return EArr.vali(ret1, ret2);
		});
	}

	/**
	 * <p>
	 * 还原网站数据：导航
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param naviRep
	 * @param chnIdChgs
	 */
	private void restore_data_nav(Backup bak, BakDirs dirs, BakAble able, EMap chnIdAll, EMap chnIdChgs){
		if(!able.isEnNav()) return;
		File naviFile = new File(dirs.getBakDataRoot().concat("/navi.json"));
		File itemFile = new File(dirs.getBakDataRoot().concat("/naviItem.json"));
		if(!naviFile.exists()) return;
		String naviStr = EFile.read(naviFile);
		String itemStr = EFile.read(itemFile);
		if(EStr.isEmpty(naviStr)) return;
		
		//获取现有导航列表
		String domain = bak.getDomain();
		List<Record> navisExist = naviSer.getBackNavis(domain);
		List<Record> itemsExist = naviItemSer.getBackNavis(domain);

		//为了便于查找现有导航，准备字典表
		EMap naviIdMap = EMap.create(), naviCodeMap = EMap.create(), itemMap = EMap.create();
		for(int i = 0, len = navisExist.size(); i < len; i++){
			Record navi = navisExist.get(i);
			naviIdMap.set(navi.getStr("id"), i);
			naviCodeMap.set(navi.getStr("code"), i);
		}
		for(int i = 0, len = itemsExist.size(); i < len; i++){
			Record item = itemsExist.get(i);
			itemMap.set(item.getStr("id"), i);
		}
		
		//准备两个列表，分别存储新增加的导航和覆盖修改的导航
		List<Record> naviAdd = Lists.newArrayList();
		List<Record> naviChg = Lists.newArrayList();
		List<Record> itemAdd = Lists.newArrayList();
		List<Record> itemChg = Lists.newArrayList();
		
		//备份导航的ID变化记录（原ID -> 新ID）
		EMap naviIdChgs = EMap.create();
		
		//覆盖的字段列表
		String[] naviKeeps = {"code", "caption"};
		String[] itemKeeps = {"naviId", "channelId", "caption", "icon", "href", "target", "sort"};
		
		//遍历备份列表
		List<Navi> navisBak = JSON.parseArray(naviStr, Navi.class);
		navisBak.forEach(naviBak -> {
			String bakId = naviBak.getId();
			String bakCode = naviBak.getCode();
			naviBak.keep(naviKeeps);
			//如果导航ID或Code已存在，则覆盖现有导航
			if(naviIdMap.containsKey(bakId) || naviCodeMap.containsKey(bakCode)){
				if(!able.isEnNavRep()) return;
				Record navi;
				if(naviIdMap.containsKey(bakId)){ //导航ID相同，仅覆盖即可
					navi = navisExist.get(naviIdMap.getAs(bakId));
				}else{ //导航ID不同，但别名相同，则视为相同导航，可继续使用已有导航ID，但需要将备份导航ID手动调整为已有导航ID，否则备份明细中将无法匹配到所在导航
					navi = navisExist.get(naviCodeMap.getAs(bakCode));
					naviIdChgs.set(bakId, navi.getStr("id"));
				}
				navi.setColumns(naviBak);
				naviChg.add(navi);
			}else{ //如果导航ID或Code都不存在，则添加新导航
				String nid = EUid.generator();
				naviIdChgs.set(bakId, nid);
				Record navi = new Record();
				navi.set("id", nid).set("domain", domain).set("subDomain", "");
				navi.setColumns(naviBak);
				naviAdd.add(navi);
			}
		});
		
		//遍历备份明细列表
		List<NaviItem> itemsBak = JSON.parseArray(itemStr, NaviItem.class);
		itemsBak.forEach(itemBak -> {
			String bakId = itemBak.getId();
			String bakNaviId = itemBak.getNaviId();
			String bakChnId = itemBak.getChannelId();
			if(naviIdChgs.containsKey(bakNaviId)) itemBak.setNaviId(naviIdChgs.getStr(bakNaviId));
			if(EStr.notEmpty(bakChnId)){
				if(!chnIdAll.containsKey(bakChnId)) return;
				if(chnIdChgs.containsKey(bakChnId)) itemBak.setChannelId(chnIdChgs.getStr(bakChnId));
			}
			itemBak.keep(itemKeeps);
			//如果明细ID，则覆盖现有明细
			if(itemMap.containsKey(bakId)){
				if(!able.isEnNavRep()) return;
				Record item = itemsExist.get(itemMap.getAs(bakId));
				item.setColumns(itemBak);
				itemChg.add(item);
			}else{ //如果明细ID不存在，则添加新明细
				String nid = EUid.generator();
				Record item = new Record();
				item.set("id", nid).set("domain", domain).set("subDomain", "");
				item.setColumns(itemBak);
				itemAdd.add(item);
			}
		});
		
		//执行事务
		dbo().tx(() -> {
			int[] empty = new int[]{1};
			int[] ret1 = naviChg.size() > 0 ? dbo().batchUpdate("site_navi", naviChg, naviChg.size()) : empty;
			int[] ret2 = naviAdd.size() > 0 ? dbo().batchSave("site_navi", naviAdd, naviAdd.size()) : empty;
			int[] ret3 = itemChg.size() > 0 ? dbo().batchUpdate("site_navi_item", itemChg, itemChg.size()) : empty;
			int[] ret4 = itemAdd.size() > 0 ? dbo().batchSave("site_navi_item", itemAdd, itemAdd.size()) : empty;
			return EArr.vali(ret1, ret2, ret3, ret4);
		});
	}

	/**
	 * <p>
	 * 还原网站数据：标签
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param tagRep
	 */
	private void restore_data_tag(Backup bak, BakDirs dirs, BakAble able){
		if(!able.isEnTag()) return;
		File tagFile = new File(dirs.getBakDataRoot().concat("/tag.json"));
		if(!tagFile.exists()) return;
		String tagStr = EFile.read(tagFile);
		if(EStr.isEmpty(tagStr)) return;
		
		//获取现有标签列表
		String domain = bak.getDomain();
		List<Tags> tagsExist = tagSer.getTags(domain);

		//为了便于查找现有标签，准备字典表
		EMap codeMap = EMap.create();
		for(int i = 0, len = tagsExist.size(); i < len; i++){
			Tags tag = tagsExist.get(i);
			codeMap.set(tag.getCode(), i);
		}
		
		//准备两个列表，分别存储新增加的标签和覆盖修改的标签
		List<Tags> tagAdd = Lists.newArrayList();
		List<Tags> tagChg = Lists.newArrayList();
		
		//标签覆盖的字段列表
		String[] keeps = {"code", "pub", "sort", "caption", "summary"};

		//遍历备份标签列表
		List<Tags> tagsBak = JSON.parseArray(tagStr, Tags.class);
		tagsBak.forEach(tagBak -> {
			String bakCode = tagBak.getCode();
			tagBak.keep(keeps);
			//如果标签Code已存在，则覆盖现有标签
			if(codeMap.containsKey(bakCode)){
				if(!able.isEnTagRep()) return;
				Tags tag = tagsExist.get(codeMap.getAs(bakCode));
				tag._setAttrs(tagBak);
				tagChg.add(tag);
			}else{ //如果标签Code不存在，则添加新标签
				String nid = EUid.generator();
				Tags tag = new Tags();
				tag.set("id", nid).set("domain", domain).set("used", 1);
				tag._setAttrs(tagBak);
				tagAdd.add(tag);
			}
		});
		
		//执行事务
		dbo().tx(() -> {
			int[] empty = new int[]{1};
			int[] ret1 = tagChg.size() > 0 ? dbo().batchUpdate(tagChg, tagChg.size()) : empty;
			int[] ret2 = tagAdd.size() > 0 ? dbo().batchSave(tagAdd, tagAdd.size()) : empty;
			return EArr.vali(ret1, ret2);
		});
	}
	
	/**
	 * <p>
	 * 网站还原：模板文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param able
	 * @throws IOException
	 */
	private void restore_temp(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnTemp()) return;
		File srcFolder = new File(dirs.getBakTempRoot());
		File distFolder = new File(dirs.getSrcTempRoot());
		EFile.copyDir(srcFolder, distFolder);
	}
	
	/**
	 * <p>
	 * 网站还原：资源文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param able
	 * @throws IOException
	 */
	private void restore_res(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnRes()) return;
		File srcFolder = new File(dirs.getBakResRoot());
		File distFolder = new File(dirs.getSrcResRoot());
		EFile.copyDir(srcFolder, distFolder);
	}
	
	/**
	 * <p>
	 * 网站还原：上传文件
	 * </p>
	 * 
	 * @param bak
	 * @param dirs
	 * @param able
	 * @throws IOException
	 */
	private void restore_upload(Backup bak, BakDirs dirs, BakAble able) throws IOException{
		if(!able.isEnUpload()) return;
		String years = EStr.FileSep + bak.getDataYear();
		File srcFolder = new File(dirs.getBakFileRoot() + years);
		if(!srcFolder.exists()) return;
		File distFolder = new File(dirs.getSrcFileRoot() + years);
		EFile.copyDir(srcFolder, distFolder);
	}
	
}