package itez.plat.site.controller;

import java.util.List;

import com.google.inject.Inject;
import com.jfinal.plugin.activerecord.Page;

import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.controller.EController;
import itez.kit.EClean;
import itez.kit.EStr;
import itez.kit.restful.Result;
import itez.plat.site.ModuleConfig;
import itez.plat.site.model.Channel;
import itez.plat.site.model.Content;
import itez.plat.site.service.ChannelService;
import itez.plat.site.service.ContentService;

/**
 * <p>
 * 网站数据控制器（栏目）
 * URL：/site/data/channel/[Action]/[ChannelCode]?[PARAMS...]
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年12月1日 下午2:14:55
 */
@ControllerDefine(key = "/site/data/channel", summary = "网站数据控制器（栏目）", view = "/")
public class SiteDataChannelController extends EController {

	@Inject
	ChannelService chnSer;
	
	@Inject
	ContentService contSer;

	/**
	 * <p>
	 * 返回首页栏目详情
	 * </p>
	 *
	 */
	public void index(){
		Channel chn = chnSer.findByCode(ModuleConfig.INDEX_CODE);
		if(null == chn){
			renderJson(Result.fail("首页栏目不存在！"));
			return;
		}
		renderJson(Result.success().setData(chn._getAttrs()));
	}
	
	/**
	 * <p>
	 * 获取指定栏目详情
	 * </p>
	 *
	 */
	public void by(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		renderJson(Result.success().setData(chn._getAttrs()));
	}
	
	/**
	 * <p>
	 * 获取指定栏目地址
	 * </p>
	 *
	 */
	public void url(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		renderJson(Result.success("url", chn.url()));
	}
	
	/**
	 * <p>
	 * 返回父栏目详情
	 * </p>
	 *
	 */
	public void parent(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		Channel par = chn.parent();
		renderJson(Result.success().setData(par._getAttrs()));
	}
	
	/**
	 * <p>
	 * 返回子栏目列表
	 * </p>
	 *
	 */
	public void children(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		List<Channel> list = chn.children();
		renderJson(Result.success("list", list));
	}
	
	/**
	 * <p>
	 * 返回第一篇文章
	 * </p>
	 *
	 */
	public void first(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		Content cont = chn.first();
		renderJson(cont == null ? Result.fail("没有文章！") : Result.success().setData(cont._getAttrs()));
	}
	
	/**
	 * <p>
	 * 返回最后一篇文章
	 * </p>
	 *
	 */
	public void last(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		Content cont = chn.last();
		renderJson(cont == null ? Result.fail("没有文章！") : Result.success().setData(cont._getAttrs()));
	}
	
	/**
	 * <p>
	 * 返回文章列表（支持多个栏目代码，之间用逗号分隔）
	 * </p>
	 * 
	 * @param count
	 * @param desc
	 * @param children
	 */
	public void list(Integer count, Boolean desc, Boolean children){
		String codes = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(codes)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		if(EStr.isEmpty(desc)) desc = true;
		if(EStr.isEmpty(children)) children = false;
		List<Content> list = new Channel().list(codes, count, desc, children);
		renderJson(Result.success("list", list));
	}
	
	/**
	 * <p>
	 * 分页返回文章列表（支持多个栏目代码，之间用逗号分隔）
	 * </p>
	 * 
	 * @param pageNum
	 * @param pageSize
	 * @param desc
	 * @param children
	 */
	public void page(Integer pageNum, Integer pageSize, Boolean desc, Boolean children){
		String codes = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(codes)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		if(EStr.isEmpty(pageNum)) pageNum = 1;
		if(EStr.isEmpty(pageSize)) pageSize = 20;
		if(EStr.isEmpty(desc)) desc = true;
		if(EStr.isEmpty(children)) children = false;
		Page<Content> list = new Channel().page(codes, pageNum, pageSize, desc, children);
		renderJson(Result.success("list", list));
	}
	
	/**
	 * <p>
	 * 按关键字查找文章（支持多个栏目代码，之间用逗号分隔）
	 * </p>
	 * 
	 * @param keyword
	 * @param desc
	 * @param children
	 */
	public void search(String keyword, Boolean desc, Boolean children){
		String codes = EClean.clean(getPara(0), EClean.text);
		keyword = EClean.clean(keyword, EClean.text);
		if(EStr.isEmpty(codes)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		if(EStr.isEmpty(keyword)){
			renderJson(Result.fail("未接收到关键字！"));
			return;
		}
		if(EStr.isEmpty(desc)) desc = true;
		if(EStr.isEmpty(children)) children = false;
		List<Content> list = new Channel().search(codes, keyword, desc, children);
		renderJson(Result.success("list", list));
	}
	
	/**
	 * <p>
	 * 按关键字查找文章（支持多个栏目代码，之间用逗号分隔）
	 * </p>
	 * 
	 * @param keyword
	 * @param desc
	 * @param children
	 * @param pageNum
	 * @param pageSize
	 */
	public void searchPage(String keyword, Boolean desc, Boolean children, Integer pageNum, Integer pageSize){
		String codes = EClean.clean(getPara(0), EClean.text);
		keyword = EClean.clean(keyword, EClean.text);
		if(EStr.isEmpty(codes)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		if(EStr.isEmpty(keyword)){
			renderJson(Result.fail("未接收到关键字！"));
			return;
		}
		if(EStr.isEmpty(pageNum)) pageNum = 1;
		if(EStr.isEmpty(pageSize)) pageSize = 20;
		if(EStr.isEmpty(desc)) desc = true;
		if(EStr.isEmpty(children)) children = false;
		Page<Content> list = new Channel().searchPage(codes, keyword, desc, children, pageNum, pageSize);
		renderJson(Result.success("list", list));
	}
	
	/**
	 * <p>
	 * 返回栏目内的置顶文章
	 * </p>
	 */
	public void tops(){
		String code = EClean.clean(getPara(0), EClean.text);
		if(EStr.isEmpty(code)){
			renderJson(Result.fail("未接收到栏目代码！"));
			return;
		}
		Channel chn = chnSer.findByCode(code);
		if(null == chn){
			renderJson(Result.fail("栏目不存在！"));
			return;
		}
		List<Content> list = contSer.getTops(chn.getId());
		renderJson(Result.success("list", list));
	}
	
}
