package itez.plat.site.controller;

import java.io.File;
import java.util.List;
import java.util.stream.Collectors;

import com.google.inject.Inject;
import com.jfinal.render.ContentType;

import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.restful.Result;
import itez.plat.main.model.Counter;
import itez.plat.main.util.ConfigKit;
import itez.plat.site.bean.SiteRender;
import itez.plat.site.bean.SiteTempEngine;
import itez.plat.site.model.Channel;
import itez.plat.site.model.Content;
import itez.plat.site.model.Info;
import itez.plat.site.model.Navi;
import itez.plat.site.model.Tags;
import itez.plat.site.service.ChannelService;
import itez.plat.site.service.ContentService;
import itez.plat.site.service.InfoService;
import itez.plat.site.service.SiteTempService;
import itez.plat.wrapper.controller.EControllerSite;
import itez.plat.site.bean.SiteTempEngine.GEN;
import itez.plat.site.bean.SiteTempEngine.Type;

/**
 * <p>
 * 网站控制器超类
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年8月23日 上午10:30:34
 */
public abstract class SiteController extends EControllerSite {
	
	@Inject
	protected InfoService infoSer;
	
	@Inject
	protected ChannelService channelSer;
	
	@Inject
	protected ContentService contentSer;
	
	@Inject
	protected SiteTempService tempSer;
	
	/**
	 * 空白提示页面
	 */
	protected void renderEmptySite(){
		render("site/temp/emptySite.html");
	}

	/**
	 * 渲染页面
	 * 
	 * @param type
	 * @param site
	 * @param channel
	 * @param content
	 * @param view
	 */
	protected void renderSite(Type type, Info site, Channel channel, Content content, String view){
		String staticPagePath = "";			//静态页面路径
		File staticPageFile = null;			//静态页面文件
		Integer genModeInt = getParaToInt("mode", site.getGenMode());	//可通过参数指定生成模式（0：纯动态，1：自动静态，2：手动静态）
		GEN genMode = GEN.values()[genModeInt];							//生成模式
		boolean genAble = false;										//是否需要重新生成
		boolean genMast = EStr.notEmpty(getPara("_GEN_MODE_"));			//是否强制生成（仅自动模式有效，纯静态模式下总是强制生成）
		
		//网站整体黑白化
		setAttr("GLOBAL_GRAY", ConfigKit.getBool("GlobalGray"));
		
		//如果非动态渲染，则需要初始化静态页面信息
		if(genMode != GEN.DYNAMIC){
			staticPagePath = SiteTempEngine.formatStaticPath(type, channel, content);			
			staticPageFile = new File(SiteTempEngine.getWebRoot() + staticPagePath);
			if(genMode == GEN.STATIC){ //纯静态模式
				//保存当前生成的页面类型，用于生成静态页面的站内链接
				attr().setItem(SiteTempEngine.SITE_PAGE_TYPE, type.name());
				genAble = true;
			}else{ //自动模式，则需要判断是否需要重新生成静态页面
				if(genMast){ //手动生成
					genAble = true;
				}else if(!staticPageFile.exists()){ //静态页面不存在，直接生成
					genAble = true;
				}else{ //判断静态页面是否已超期
					genAble = SiteTempEngine.valiExpir(staticPageFile, site.getGenTimeout());
				}
			}
		}
		
		//仅自动生成模式，且无需重新生成静态页的情况下，才无需传递页面参数
		if(genMode != GEN.AUTO || genAble){
			setAttr("Type", type.name());
			setAttr("Site", site);
			setAttr("Channel", channel);
			setAttr("Content", content);
			setAttr("Navi", new Navi());
			setAttr("Tag", new Tags());
			setAttr("Request", paramPack().getParas().cleanAll());
			setAttr("webres", attr().getFileStoreUrl().concat("/res"));
		}
		
		try {
			String viewPath = SiteTempEngine.formatViewPath(view);
			SiteRender render = new SiteRender(viewPath);
			if(genMode == GEN.DYNAMIC){ //动态模式
				initCounter(type, site, channel, content);
				render(render);
			}else{ //自动或静态模式
				if(genAble){ //需要生成页面
					render.setContext(request(), response());
					render.generator(staticPageFile);
					SiteTempEngine.clearCache(staticPagePath);
				}
				if(genMode == GEN.AUTO){ //自动模式，直接渲染
					render.setView(staticPagePath);
					render(render);
				}else{ //静态生成模式，返回生成结果
					if(genMast){ //生成模式
						renderJson(Result.success());
					}else{ //浏览模式
						redirect(staticPagePath);
					}
				}
			}
		} catch (Exception e) {
			if(genMode == GEN.STATIC){ //纯静态模式下，使用JSON格式返回异常信息
				renderJson(Result.fail(e.getMessage()));
			}else{
				renderText("页面渲染出现异常，请稍后再试！");
			}
			if(EProp.DevMode) e.printStackTrace();
		}
	}
	
	/**
	 * <p>
	 * 统一页面处理（计数等）
	 * http://domain/siteHandle/INDEX-{channelId}
	 * http://domain/siteHandle/CHANNEL-{channelId}
	 * http://domain/siteHandle/CONTENT-{channelId}-{contentId}
	 * </p>
	 *
	 */
	public void siteHandle(){
		String typeName = getClean(0);
		String channelId = getClean(1);
		String contentId = getClean(2);
		Type type = Type.valueOf(typeName);
		Info site = infoSer.getInfo();
		Channel channel = EStr.isEmpty(channelId) ? new Channel() : channelSer.findById(channelId);
		Content content = EStr.isEmpty(contentId) ? new Content() : contentSer.findById(contentId);
		setAttr("Type", type.name());
		setAttr("Site", site);
		setAttr("Channel", channel);
		setAttr("Content", content);
		initCounter(type, site, channel, content);
		SiteRender render = new SiteRender("/siteTemp/common/siteHandle.js");
		render.setContentType(ContentType.JAVASCRIPT);
		render(render);
	}
	
	/**
	 * <p>
	 * 初始化统计对象
	 * 1、动态模式下：
	 *     每次页面被访问时自动调用该方法，实现计数累加，再利用 Site/Channel/Content 的 counter() 方法返回当前计数统计值。
	 * 2、其他模式下：
	 *     需要在模板中手动引用 siteHandle 实现计数累加，例如：<script src="#(attr.getDomainUrl())/siteHandle/#(Type)-#(Channel.id)-#(Content.id??)"></script>
	 *     如果页面基于标准模板base，则已实现自动引用，无需手动引用，只需要将页面内需要显示计数的元素增加以下 class 即可：SITE-CNT-GLOBAL / SITE-CNT-CHANNEL / SITE-CNT-CONTENT
	 * </p>
	 * 
	 * @param navs
	 * @param contentId
	 * @param site
	 * @param channel
	 * @param content
	 */
	private void initCounter(Type type, Info site, Channel channel, Content content){
		if(site.getUseCountor() != 1) return;
		String moduleCode = itez.plat.site.ModuleConfig.MODULE_CODE;
		Counter cnt = new Counter();
		cnt.setModuleCode(moduleCode);
		String keys = null;
		
		if(type == Type.INDEX){
			Channel indexChannel = channelSer.getIndexChannel();
			keys = indexChannel.getId();
		}else if(type == Type.CHANNEL || type == Type.CONTENT){
			List<Channel> navs = channel.location();
			keys = navs.stream().map(c -> c.getId()).collect(Collectors.joining(","));
			if(type == Type.CONTENT) keys = keys + "," + content.getId();
		}
		
		if(null != keys){
			cnt.setKey(keys);
			cnt.access();
		}
	}
	
}
