package itez.plat.site.controller;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.util.List;

import javax.imageio.ImageIO;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import com.google.inject.Inject;
import com.jfinal.plugin.activerecord.Page;
import com.jfinal.plugin.activerecord.Record;

import itez.core.wrapper.controller.ControllerDefine;
import itez.kit.EHttp;
import itez.kit.EJson;
import itez.kit.EPara;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.pic.EPicKit;
import itez.kit.restful.EMap;
import itez.kit.restful.Result;
import itez.plat.main.util.ConfigKit;
import itez.plat.site.model.Channel;
import itez.plat.site.model.CollectorItem;
import itez.plat.site.model.CollectorSrc;
import itez.plat.site.model.Content;
import itez.plat.site.model.Info;
import itez.plat.site.model.Tags;
import itez.plat.site.service.ChannelService;
import itez.plat.site.service.CollectorSrcService;
import itez.plat.site.service.CollectorTaskService;
import itez.plat.site.service.ContentService;
import itez.plat.site.service.InfoService;
import itez.plat.site.service.TagsService;
import itez.plat.wrapper.controller.EControllerMgr;

@ControllerDefine(key = "/site/content", summary = "网站文章管理", view = "/")
public class SiteContentController extends EControllerMgr {

	@Inject
	InfoService infoSer;
	
	@Inject
	ChannelService channelSer;
	
	@Inject
	ContentService contentSer;
	
	@Inject
	CollectorSrcService srcSer;

	@Inject
	CollectorTaskService taskSer;
	
	@Inject
	TagsService tagSer;
	
	public void index(String pid){
		Info info = infoSer.getInfo();
		List<Tags> tags = tagSer.getTagsEnable();
		List<Channel> chns = channelSer.getAllChannels();
		List<Record> conbtChns = channelSer.getConbtChannels();
		List<CollectorSrc> srcs = srcSer.getSrcList(null);
		setAttr("info", EJson.toJson(info));
		setAttr("pid", EStr.ifEmpty(pid, channelSer.getIndexChannel().getId()));
		setAttr("channels", EJson.toJson(chns));
		setAttr("conbtChns", EJson.toJson(conbtChns));
		setAttr("srcs", EJson.toJsonFilter(srcs, "id", "caption"));
		setAttr("tags", EJson.toJsonFilter(tags, "id", "code", "caption"));
		setAttr("SiteUpdateStop", ConfigKit.getBool("SiteUpdateStop"));
		render("content.html");
	}
	
	public void getContentList(String pid, Integer page){
		Channel chn = channelSer.findById(pid);
		if(EStr.isEmpty(page)) page = 1;
		Page<Content> contents = contentSer.getPageByChannelId(pid, page, 20, true, true);
		renderJson(Result.success("contents", contents).set("chn", chn));
	}
	
	/**
	 * 检索同名文章
	 * @param title
	 */
	public void getSameContent(String caption){
		List<Content> conts = contentSer.getSameContent(caption);
		renderJson(Result.success("conts", conts));
	}
	
	public void formEvent(){
		boolean SiteUpdateStop = ConfigKit.getBool("SiteUpdateStop");
		if(SiteUpdateStop){
			renderJson(Result.fail("管理员已暂停网站更新功能！"));
			return;
		}
		EPara paras = paramPack(false);
		Content content = paras.getModel(Content.class);
				
		//文章封面及缩略图
		File picFile = paras.getFileObj("pic");
		String picUrl = "", thumUrl = "";
		if(picFile != null){
			String picPath = picFile.getAbsolutePath();
			BufferedImage thum = EPicKit.use().compress(picPath, 0.8f, 800, 600);
			picUrl = getUpload(picFile);
			thumUrl = getUpload(thum, EPicKit.getFormatName(picFile));
		}else {
			boolean collPic = true; //是否自动从正文抓取封面图
			if(EStr.notEmpty(content.getId())){
				Content exist = contentSer.findById(content.getId());
				if(EStr.notEmpty(exist.getPic()) && EStr.notEmpty(exist.getThum())) collPic = false; //已经设置封面图，无需重新抓取
			}
			if(collPic){
				EMap pics = getTopPic(content.getContent());
				picUrl = pics.getStr("picUrl");
				thumUrl = pics.getStr("thumUrl");
			}
		}
		if(EStr.notEmpty(picUrl)) content.setPic(picUrl);
		if(EStr.notEmpty(thumUrl)) content.setThum(thumUrl);
		
		if(EStr.isEmpty(content.getId())){
			contentSer.save(content);
		}else{
			contentSer.update(content);
		}
		
		//更新静态页面
		contentSer.genStaticPage(content.getId());

		//投稿抄送
		if(EStr.notEmpty(paras.get("conbt"))){
			boolean conbt = paras.getBoolean("conbt");
			String[] conbtTo = paras.getValues("conbtTo");
			if(conbt && null != conbtTo && conbtTo.length > 0){
				contentSer.conbt(content, conbtTo);
			}
		}
		
		renderJson(Result.success("content", contentSer.findById(content.getId())));
	}
	
	/**
	 * <p>
	 * 获得正文中的第一张图片（当未上传封面图片时，自动使用正文第一张图片作为封面）
	 * </p>
	 * 
	 * @param content
	 * @return
	 */
	private EMap getTopPic(String content){
		String picUrl = "", thumUrl = "";
		if(EStr.notEmpty(content)){
			Document doc = Jsoup.parse(content);
			Elements imgs = doc.select("img");
			if(imgs.size() > 0){
				Element img = imgs.get(0);
		        String imgUrl = img.attr("src");
	        	picUrl = imgUrl;
	        	if(imgUrl.startsWith("//")) imgUrl = EStr.join(attr().getScheme(), ":", imgUrl);
	        	else if(imgUrl.startsWith("/")) imgUrl = EStr.join(attr().getWeb(), imgUrl);
	            try {
		            byte[] imgByte = EHttp.me.getByte(imgUrl);
		            String imgType = ERegex.findFirst(imgUrl, "\\.(jpg|png|gif|bmp)");
		            if(EStr.isEmpty(imgType)) imgType = "jpg";
		            String TempPath = EProp.FileUploadTemp;
	    	        ByteArrayInputStream imgIn = new ByteArrayInputStream(imgByte);
	    	        BufferedImage imgBuff = ImageIO.read(imgIn);
	    	        File picFile = new File(TempPath.concat("thu.").concat(imgType));
	    	        ImageIO.write(imgBuff, imgType, picFile);
	    	        imgIn.close();
	    			String picPath = picFile.getAbsolutePath();
	    	        BufferedImage thum = EPicKit.use().compress(picPath, 0.8f, 800, 600);
	    	        thumUrl = getUpload(thum, imgType);
	    		} catch (Exception e) {
	    			thumUrl = imgUrl;
	    		}
			}
		}
		return EMap.by("picUrl", picUrl).set("thumUrl", thumUrl);
	}
	
	/**
	 * <p>
	 * 删除文章
	 * </p>
	 * 
	 * @param ids
	 */
	public void removeContent(String ids){
		contentSer.removeStaticPage(ids);
		contentSer.deleteByIds(ids);
		renderJson(Result.success());
	}
	
	/**
	 * <p>
	 * 文章置顶（添加	/取消）
	 * </p>
	 * 
	 * @param pid
	 * @param ids
	 * @param val
	 */
	public void setTops(String pid, String ids, Boolean val){
		List<Content> conts = contentSer.findByIds(ids);
		for(Content cont : conts){
			cont.setIsTop(val);
			contentSer.update(cont);
		}
		Page<Content> contents = contentSer.getPageByChannelId(pid, 1, 20, true);
		renderJson(Result.success("contents", contents));
	}
	
	/**
	 * <p>
	 * 文章移动
	 * </p>
	 * 
	 * @param ids
	 * @param pid
	 */
	public void move(String ids, String pid){
		Channel chn = channelSer.findById(pid);
		List<Content> conts = contentSer.findByIds(ids);
		for(Content cont : conts){
			cont.setChannelId(pid).setChannelCaption(chn.getCaption());
			contentSer.update(cont);
		}
		renderJson(Result.success());
	}
	
	/**
	 * <p>
	 * 删除当前静态页
	 * </p>
	 * 
	 * @param ids
	 */
	public void removeStaticPage(String ids){
		contentSer.removeStaticPage(ids);
		renderJson(Result.success());
	}
	
	/**
	 * <p>
	 * 生成静态页
	 * </p>
	 * 
	 * @param ids
	 */
	public void genStaticPage(String ids){
		try {
			contentSer.genStaticPage(ids);
			renderJson(Result.success());
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 文章采集
	 * </p>
	 * 
	 * @param pid
	 * @param srcId
	 * @param url
	 */
	public void collContent(String pid, String srcId, String url){
		if(EStr.hasEmpty(pid, srcId, url)){
			renderJson(Result.fail("缺少参数！"));
			return;
		}
		Channel chn = channelSer.findById(pid);
		CollectorSrc src = srcSer.findById(srcId);
		try {
			CollectorItem collItem = new CollectorItem().setUrl(url);
			Content cont = taskSer.collItemDetail(attr().getDomain(), chn, src, true, collItem, true);
			cont.remove("id");
			renderJson(Result.success("content", cont));
		} catch (Exception e) {
			renderJson(Result.fail(e.getCause().getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 草稿箱
	 * </p>
	 *
	 */
	public void draft(){
		Info info = infoSer.getInfo();
		setAttr("info", EJson.toJson(info));
		render("draft.html");
	}
	
	public void getDraftData(){
		List<Record> conts = contentSer.getDraft();
		renderJson(Result.success("contents", conts));
	}
	
}
