/**
 * Copyright(C) 2021 上游科技 . All Rights Reserved
 */
package itez.plat.site.controller;

import java.util.Date;
import java.util.List;

import com.google.inject.Inject;

import itez.core.wrapper.controller.ControllerDefine;
import itez.kit.EDate;
import itez.kit.EJson;
import itez.kit.ELog;
import itez.kit.EPara;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.exception.ExceptionKit;
import itez.kit.log.ELogBase;
import itez.kit.restful.Result;
import itez.plat.site.model.CollectorSrc;
import itez.plat.site.model.CollectorTask;
import itez.plat.site.service.ChannelService;
import itez.plat.site.service.CollectorSrcService;
import itez.plat.site.service.CollectorTaskService;
import itez.plat.site.service.impl.CollectorCommon;
import itez.plat.wrapper.controller.EControllerMgr;

/**
 * <p>
 * 采集任务管理
 * </p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年11月26日 下午10:36:19
 */
@ControllerDefine(key = "/site/collector/task", summary = "采集任务管理", view = "/collector")
public class CollectorTaskController extends EControllerMgr {

	@Inject
	ChannelService chnSer;
	
	@Inject
	CollectorSrcService srcSer;

	@Inject
	CollectorTaskService taskSer;
	
	private static ELogBase log = ELog.log(CollectorTaskController.class);
	
	public void index(){
		String channelCode = getPara(0);
		List<CollectorTask> tasks = taskSer.getTasks(channelCode);
		List<CollectorSrc> srcs = srcSer.getSrcList(CollectorCommon.SRC_TYPE.channel.name());
		setAttr("channel", EJson.toJson(chnSer.findByCode(channelCode).keep("code", "caption")));
		setAttr("srcs", EJson.toJsonFilter(srcs, "id", "caption"));
		setAttr("tasks", EJson.toJson(tasks));
		render("task.html");
	}
	
	public void event(){
		EPara paras = paramPack();
		try {
			CollectorTask task = paras.getModel(CollectorTask.class);
			CollectorTask exist = taskSer.getTask(task.getChannelCode(), task.getSrcId(), task.getUrl());
			if(EStr.isEmpty(task.getId())){
				if(exist != null){
					renderJson(Result.fail("该采集任务已经存在！"));
					return;
				}
				task.setListState(0).setListSize(0).setItemState(0).setItemSize(0);
				taskSer.save(task);
			}else{
				if(exist != null && !exist.getId().equals(task.getId())){
					renderJson(Result.fail("该采集任务已经存在！"));
					return;
				}
				taskSer.update(task);
			}
			List<CollectorTask> tasks = taskSer.getTasks(task.getChannelCode());
			renderJson(Result.success("tasks", tasks));
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	public void remove(String id){
		CollectorTask task = taskSer.findById(id);
		taskSer.deleteById(id);
		List<CollectorTask> tasks = taskSer.getTasks(task.getChannelCode());
		renderJson(Result.success("tasks", tasks));
	}
	
	public void collList(String id) throws InterruptedException{
		CollectorTask task = taskSer.findById(id);
		if(task.getItemState() == 2){
			renderJson(Result.fail("请等待文章采集完成后再进行列表采集"));
			return;
		}
		if(task.getListState() == 2){
			renderJson(Result.fail("列表采集正在进行中，请不要重复点击"));
			return;
		}
		try {
			Date dtStart = EDate.getDate();
			task.setListState(2).update();
			int size = taskSer.collList(task);
			Date dtEnd = EDate.getDate();
			task.setListState(1).setListStart(dtStart).setListEnd(dtEnd);
			task.setListSize(task.getListSize() + size).update();
			List<CollectorTask> tasks = taskSer.getTasks(task.getChannelCode());
			renderJson(Result.success("tasks", tasks));
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
			log.error(ExceptionKit.getExceptionMore(e));
			task.setListState(0).update();
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	public void collItem(String id){
		CollectorTask task = taskSer.findById(id);
		if(task.getListState() != 1){
			renderJson(Result.fail("请等待列表采集完成后再进行文章采集"));
			return;
		}
		if(task.getItemState() == 2){
			renderJson(Result.fail("文章采集正在进行中，请不要重复点击"));
			return;
		}
		try {
			Date dtStart = EDate.getDate();
			task.setItemState(2).update();
			int size = taskSer.collItem(task, false, true);
			Date dtEnd = EDate.getDate();
			task.setItemState(1).setItemStart(dtStart).setItemEnd(dtEnd);
			task.setItemSize(task.getItemSize() + size).update();
			List<CollectorTask> tasks = taskSer.getTasks(task.getChannelCode());
			renderJson(Result.success("tasks", tasks));
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
			log.error(ExceptionKit.getExceptionMore(e));
			task.setItemState(0).update();
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	public void delItem(String id){
		CollectorTask task = taskSer.findById(id);
		try {
			taskSer.delItem(id, true);
			List<CollectorTask> tasks = taskSer.getTasks(task.getChannelCode());
			renderJson(Result.success("tasks", tasks));
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
}
