package itez.plat.site.bean;

import java.io.File;
import java.util.Date;

import com.jfinal.kit.PathKit;
import com.jfinal.template.Engine;
import com.jfinal.template.source.FileSourceFactory;

import itez.core.runtime.EContext;
import itez.kit.EDate;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.El;
import itez.plat.site.ModuleConfig;
import itez.plat.site.model.Channel;
import itez.plat.site.model.Content;

/**
 * <p>
 * 网站模板引擎管理类
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年8月19日 上午10:27:35
 */
public abstract class SiteTempEngine {

	//页面类型（首页、栏目页、内容页）
	public static enum Type { INDEX, CHANNEL, CONTENT }
	//生成模式（纯动态、自动静态、手动静态）
	public static enum GEN { DYNAMIC, AUTO, STATIC }
	//模型中存储页面类型的键名
	public static final String SITE_PAGE_TYPE = "SITE_PAGE_TYPE";

	private static Engine engine;
	private static String webRoot = PathKit.getWebRootPath();
	
	static {
		engine = Engine.create("SITE_DYNAMIC");
		engine.setDevMode(EProp.DevMode);
		engine.setSourceFactory(new FileSourceFactory());
		engine.setBaseTemplatePath(webRoot);
		engine.addSharedFunction("/siteTemp/common/layout_base.html");
		El.addSharedObject(engine);
	}
	
	/**
	 * <p>
	 * 返回模板引擎
	 * </p>
	 * 
	 * @return
	 */
	public static Engine getEngine(){
		return engine;
	}
	
	/**
	 * <p>
	 * 返回主目录
	 * </p>
	 * 
	 * @return
	 */
	public static String getWebRoot(){
		return webRoot;
	}
	
	/**
	 * <p>
	 * 格式化模板文件路径
	 * </p>
	 * 
	 * @param view
	 * @return
	 */
	public static String formatViewPath(String view){
		String domain = EContext.getAttr().getDomain();
		return EStr.join("/siteTemp/", domain, "/", view);
	}
	
	/**
	 * <p>
	 * 清除模板缓存
	 * </p>
	 * 
	 * @param views
	 */
	public static void clearTempCache(String...views){
		if(views == null || views.length == 0) return;
		for(int i = 0, len = views.length; i < len; i++){
			views[i] = formatViewPath(views[i]);
		}
		clearCache(views);
	}
	
	/**
	 * <p>
	 * 清除模板缓存
	 * </p>
	 * 
	 * @param views
	 */
	public static void clearCache(String...views){
		if(views == null || views.length == 0) return;
		for(String view : views){
			try {
				engine.removeTemplateCache(view);
				ELog.info("网站模板缓存清理成功：{}", view);
			} catch (Exception e) {
				ELog.error("网站模板缓存清理失败，模板文件不存在：{}", view);
			}
		}
	}
	
	/**
	 * <p>
	 * 获取静态页面路径
	 * </p>
	 * 
	 * @param type
	 * @param channel
	 * @param content
	 * @return
	 */
	public static String formatStaticPath(Type type, Channel channel, Content content){
		String domain = EContext.getAttr().getDomain();
		String staticPagePath = EStr.join("/sitePage/", domain, "/");
		Integer page = EContext.getAttr().getPage();
		if(type == Type.INDEX || (type == Type.CHANNEL && channel.getCode().equals(ModuleConfig.INDEX_CODE))) staticPagePath += "index.html";
		else if(type == Type.CHANNEL) staticPagePath = EStr.join(staticPagePath, channel.getCode(), "/", page, ".html");
		else if(type == Type.CONTENT) staticPagePath = EStr.join(staticPagePath, channel.getCode(), "/", content.getId(), ".html");
		return staticPagePath;
	}
	
	/**
	 * <p>
	 * 返回当前租户静态文件主目录路径
	 * </p>
	 * 
	 * @return
	 */
	public static String getStaticRootPath(){
		String domain = EContext.getAttr().getDomain();
		return EStr.join(webRoot, EStr.FileSep, "sitePage", EStr.FileSep, domain);
	}
	
	/**
	 * <p>
	 * 返回当前租户静态文件主目录网址
	 * </p>
	 * 
	 * @return
	 */
	public static String getStaticRootUrl(){
		String webUrl = EContext.getAttr().getWeb();
		String domain = EContext.getAttr().getDomain();
		return EStr.join(webUrl, "/", "sitePage", "/", domain, "/");
	}
	
	/**
	 * <p>
	 * 判断静态页面是否超期
	 * </p>
	 * 
	 * @param file
	 * @param hours
	 * @return
	 */
	public static boolean valiExpir(File file, int hours){
		Date tm = new Date(file.lastModified());
		return EDate.diff(tm, EDate.getDate()).toHours() > hours;
	}
	
}
