package itez.plat.main.service.impl;

import java.util.Date;
import java.util.Map;
import java.util.Set;

import com.beust.jcommander.internal.Sets;
import com.google.common.collect.Maps;
import com.google.inject.Inject;
import com.google.inject.Singleton;

import itez.core.launch.JW;
import itez.core.runtime.EContext;
import itez.core.runtime.service.Define;
import itez.core.runtime.service.EModelService;
import itez.core.wrapper.dbo.model.Query;
import itez.core.wrapper.dbo.model.Querys;
import itez.kit.EClean;
import itez.kit.EDate;
import itez.kit.EJson;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.SM4Kit;
import itez.kit.log.ELogBase;
import itez.kit.restful.EMap;
import itez.kit.restful.Result;
import itez.plat.main.model.PolicyAccount;
import itez.plat.main.model.PolicyPwd;
import itez.plat.main.model.User;
import itez.plat.main.model.UserLockedInfo;
import itez.plat.main.service.PolicyAccountService;
import itez.plat.main.service.PolicyLockService;
import itez.plat.main.service.PolicyPwdService;
import itez.plat.main.service.UserLoginService;
import itez.plat.main.service.UserService;

@Define
@Singleton
public class UserLoginServiceImpl extends EModelService<User> implements UserLoginService {

	@Inject
	UserService userSer;
	
	@Inject
	PolicyLockService lockSer;
	
	@Inject
	PolicyPwdService pwdSer;
	
	@Inject
	PolicyAccountService accSer;

	private static final String TOKEN_KEY_DATA = "data";
	private static final String TOKEN_KEY_TIME = "timestamp";
	
	//锁定清单
	private static final Map<Integer, UserLockedInfo> LOCKED_LIST = Maps.newConcurrentMap();
	//用户索引
	private static final Map<String, Set<Integer>> LOCKED_INDEX = Maps.newConcurrentMap();
	
	private static final ELogBase log = ELog.log(UserLoginService.class);

	@Override
	public User pwdLogin(User inp, String loginPass, String policyAccCode) {
		
		/**
		 * 账号方案
		 */
		if(EStr.isEmpty(policyAccCode)) throw new RuntimeException("未发现账号方案代码！");
		PolicyAccount acc = accSer.getPolicy(policyAccCode);
		if(acc == null) throw new RuntimeException("账号方案不存在！");
		String policyLockCode = acc.getPwdLockPolicy();
		
		/**
		 * 解析登录凭据
		 */
		cleanUserXss(inp);
		String loginName = inp.getLoginName();
		String idNum = inp.getIdNum();
		String num = inp.getNum();
		String phone = inp.getPhone();
		String email = inp.getEmail();
		if(EStr.allEmpty(loginName, idNum, num, phone, email)) throw new RuntimeException("用户名不允许为空！");
		
		loginPass = EClean.clean(loginPass, EClean.text);
		if(EStr.isEmpty(loginPass)) throw new RuntimeException("密码不允许为空！");
		
		//获取登录凭据的HashCode，作为清单主键
		int hash = inp.hashCode();
		
		/**
		 * 账号锁定校验
		 */
		UserLockedInfo lockInfo = LOCKED_LIST.get(hash);
		if(null != lockInfo && !lockInfo.retry()){ //已被锁定，不允许登录
			if(lockInfo.isForever()) throw new RuntimeException("由于连续登录失败，该账号已被永久锁定，请联系管理员解除锁定！");
			else throw new RuntimeException("由于连续登录失败，该账号已被锁定，请等待 " + lockInfo.surMinutes() + " 分钟后再继续尝试！");
		}
				
		/**
		 * 数据库校验
		 */
		Querys qs1 = Querys.or(Query.eq("domain", $domain())).add(Query.eq("level", 2));
		Querys qs2 = Querys.or();
		if(EStr.notEmpty(loginName)) qs2.add(Query.eq("loginName", loginName));
		if(EStr.notEmpty(idNum)) qs2.add(Query.eq("idNum", idNum));
		if(EStr.notEmpty(num)) qs2.add(Query.eq("num", num));
		if(EStr.notEmpty(phone)) qs2.add(Query.eq("phone", phone));
		if(EStr.notEmpty(email)) qs2.add(Query.eq("email", email));
		Querys qs3 = Querys.and(qs1).add(qs2);
		if(acc.getRegMember() == 1) qs3.add(Query.eq("member", 1));
		User user = selectFirst(qs3, "locked, used desc", false);
		if(null == user) throw new RuntimeException("用户名无效！");
		
		/**
		 * 密码错误
		 */
		if(!user.checkPass(loginPass)){
			if(null == lockInfo){
				lockInfo = new UserLockedInfo(user.getId(), lockSer.getPolicy(policyLockCode));
				addLockList(hash, lockInfo);
				log.info("登录密码错误(1)：[uid]{} [loginName]{} [idNum]{} [phone]{} [email]{}", user.getId(), loginName, idNum, phone, email);
				throw new RuntimeException("密码错误！");
			}else{
				lockInfo.failAgain();
				log.info("登录密码错误({})：[uid]{} [loginName]{} [idNum]{} [phone]{} [email]{}", lockInfo.getFailCnt(), user.getId(), loginName, idNum, phone, email);
				if(lockInfo.isLocked()){
					if(lockInfo.isForever()) throw new RuntimeException("密码错误！该账号已被永久锁定！");
					else throw new RuntimeException("密码错误！该账号已被锁定，请等待 " + lockInfo.surMinutes() + " 分钟后再继续尝试！");
				}else{
					int cnt = lockInfo.getModel().getRetryCnt() - lockInfo.getFailCnt();
					throw new RuntimeException("密码错误！您还可以尝试 " + cnt + " 次，否则账号将被锁定！");
				}
			}
		}

		/**
		 * 数据库账号状态校验
		 */
		if(user.getLocked()) throw new RuntimeException("该账号已被永久锁定，请联系管理员解除锁定！");
		if(user.getUsed() == 0) throw new RuntimeException("该账号已被注销！");
		
		/**
		 * 登录成功，从锁定清单中移除
		 */
		removeLockList(hash);
		log.info("登录校验成功：[uid]{} [loginName]{} [idNum]{} [num]{} [phone]{} [email]{}", user.getId(), loginName, idNum, num, phone, email);
		
		return user;
	}
	
	@Override
	public UserLockedInfo getLockInfo(User inp) {
		cleanUserXss(inp);
		int hash = inp.hashCode();
		return LOCKED_LIST.get(hash);
	}
	
	@Override
	public void cleanUserXss(User user){
		user.setLoginName(EStr.ifEmpty(EClean.clean(user.getLoginName(), EClean.text), ""));
		user.setIdNum(EStr.ifEmpty(EClean.clean(user.getIdNum(), EClean.text), ""));
		user.setNum(EStr.ifEmpty(EClean.clean(user.getNum(), EClean.text), ""));
		user.setPhone(EStr.ifEmpty(EClean.clean(user.getPhone(), EClean.text), ""));
		user.setEmail(EStr.ifEmpty(EClean.clean(user.getEmail(), EClean.text), ""));
	}

	/**
	 * 加入锁定清单
	 * @param hash
	 * @param lockInfo
	 */
	private void addLockList(int hash, UserLockedInfo lockInfo){
		LOCKED_LIST.put(hash, lockInfo);
		String uid = lockInfo.getUid();
		Set<Integer> hs = LOCKED_INDEX.get(uid);
		if(null == hs) hs = Sets.newHashSet();
		hs.add(hash);
		LOCKED_INDEX.put(uid, hs);
	}

	/**
	 * 移除锁定清单
	 * @param hash
	 */
	private void removeLockList(int hash){
		LOCKED_LIST.remove(hash);
	}

	@Override
	public void removeLockList(String uid){
		Set<Integer> hs = LOCKED_INDEX.get(uid);
		if(null == hs) return;
		hs.forEach(hash -> removeLockList(hash));
		LOCKED_INDEX.remove(uid);
	}

	@Override
	public Result pwdVali(String loginPass, String policyPwdCode) {
		PolicyPwd policy = pwdSer.getPolicy(policyPwdCode);
		if(loginPass.length() < policy.getMins()) return Result.fail("密码长度不能小于 " + policy.getMins() + " 个字符!");
		if(loginPass.length() > policy.getMaxs()) return Result.fail("密码长度不能大于 " + policy.getMaxs() + " 个字符!");
		if(policy.getHasNum() > 0 && !ERegex.has(loginPass, "[0-9]")) return Result.fail("密码中必须包含数字!");
		if(policy.getHasLesChar() > 0 && !ERegex.has(loginPass, "[a-z]")) return Result.fail("密码中必须包含小写字母!");
		if(policy.getHasCapChar() > 0 && !ERegex.has(loginPass, "[A-Z]")) return Result.fail("密码中必须包含大写字母!");
		if(policy.getHasSpeSign() > 0 && !ERegex.has(loginPass, "[^0-9a-zA-Z]")) return Result.fail("密码中必须包含特殊字符!");
		return Result.success();
	}
	
	@Override
	public String ssoRouter(String userId) {
		User user = userSer.findById(userId);
		String domain = user.getDomain();
		String token = tokenEncode(userId);
		log.info("生成单点登录Token：{} [user]{}", token, user.getCaption());
		
		String domainTemp = EProp.DomainTemplate;
		domainTemp = domainTemp.replace("${domain}", domain);
		String url = EStr.join(domainTemp, "/plat/ssoLogin?token=", token);
		if(!url.startsWith("http")) url = EStr.join(EContext.getAttr().getScheme(), ":", url);
		return url;
	}
	
	@Override
	public User ssoLogin(String token) {
		String uid = tokenDecode(token);
		User user = userSer.findById(uid);
		log.info("单点登录验证通过。[token]{} [user]{}", token, user.getCaption());
		return user;
	}
	
	@Override
	public String tokenEncode(String data) {
		return tokenEncode(data, null);
	}
	
	@Override
	public String tokenEncode(String data, String key) {
		EMap map = EMap.by(TOKEN_KEY_DATA, data).set(TOKEN_KEY_TIME, EDate.getTime());
		String json = map.toJson();
		return SM4Kit.encrypt(json, EStr.ifEmpty(key, JW.TokenSecret));
	}
	
	@Override
	public String tokenDecode(String code) {
		return tokenDecode(code, null);
	}
	
	@Override
	public String tokenDecode(String code, String key) {
		if(EStr.isEmpty(code)) throw new RuntimeException("未发现Token！");
		String json = SM4Kit.decrypt(code, EStr.ifEmpty(key, JW.TokenSecret));
		if(EStr.isEmpty(json)) throw new RuntimeException("Token无效！");
		EMap map = EJson.parse(json, EMap.class);
		String data = map.getStr(TOKEN_KEY_DATA);
		long times = map.getLong(TOKEN_KEY_TIME);
		Date expireDt = EDate.addMinute(new Date(times), 30); //有效期：30分钟
		if(EDate.isExpire(expireDt)) throw new RuntimeException("等待时间过长，Token已超期，请重新操作！");
		return data;
	}

}
