package itez.plat.main.service;

import java.io.File;
import java.util.List;

import com.jfinal.plugin.activerecord.Record;

import itez.core.runtime.service.IModelService;
import itez.core.wrapper.dbo.model.Querys;
import itez.kit.poi.XlsWriter;
import itez.plat.main.model.Dept;
import itez.plat.main.model.User;

/**
 * 成员管理
 * 
 * @author netwild
 *
 */
public interface UserService extends IModelService<User>{

	/**
	 * 按机构查找用户列表
	 * @param deptId 机构ID
	 * @return 用户列表
	 */
	public List<User> getInsUsers(String deptId);
	
	/**
	 * 按机构和类型查找用户列表
	 * @param deptId 机构ID
	 * @param type 用户类型
	 * @return 用户列表
	 */
	public List<User> getInsUsers(String deptId, String type);
	
	/**
	 * 查找指定机构已离职的用户列表
	 * @param deptId 机构ID
	 * @return 用户列表
	 */
	public List<User> getLeavedUsers(String deptId);
	
	

	///////////////////////////  成员查找相关（仅面向内部用户：member=1）   /////////////////////////////////////
	
	
	/**
	 * 按关键字（姓名、身份证号、电话号、邮箱、工号/学号）查找成员
	 * @param keyword 关键字
	 * @return 用户列表
	 */
	public List<User> searchInsUser(String keyword);
	
	/**
	 * 按关键字查找成员，并且在指定机构内
	 * @param keyword 关键字
	 * @param deptId 机构ID
	 * @return 用户列表
	 */
	public List<User> searchInsUser(String keyword, String deptId);
	
	/**
	 * 按关键字查找成员，并且在指定机构内、指定类型
	 * @param keyword 关键字
	 * @param deptId 机构ID
	 * @param type 用户类型
	 * @return 用户列表
	 */
	public List<User> searchInsUser(String keyword, String deptId, String type);
	
	/**
	 * 自定义条件查找（默认条件：member=1 AND used=1 AND level != 2）
	 * @param qs 查询条件
	 * @return 用户列表
	 */
	public List<User> searchInsUser(Querys qs);

	/**
	 * <p>
	 * 根据关键字模糊查找用户（当前租户范围内）
	 * </p>
	 * 
	 * @param deptId 部门ID，为空时查询租户内全部用户
	 * @param type 用户类型，为空时查询全部类型
	 * @param member 内部成员/外部成员，为空时查询全部成员
	 * @param fieldName 查询属性名称，可选项：caption（姓名）、idNum（身份证号码）、phone（电话号）、email（电邮）、loginName（登录账号）
	 * @param keyword 关键字，支持模糊匹配
	 * @return 用户列表
	 */
	public List<Record> searchByField(String deptId, String type, Integer member, String fieldName, String keyword);
		
	/**
	 * <p>
	 * 在指定租户内根据登录名称查找用户（仅用于绑定第三方账号时进行本地账号检索）
	 * </p>
	 * 
	 * @param domain 子域
	 * @param loginName 登录名
	 * @return 用户对象
	 */
	public User loginByDomain(String domain, String loginName);
	
	
	
	
	///////////////////////////  用户机构相关   /////////////////////////////////////
	
	

	/**
	 * <p>
	 * 返回用户的默认机构ID
	 * </p>
	 * 
	 * @param user 用户对象
	 * @return 用户所在机构ID（考虑管理员、管辖机构等情况），同User.getRootDeptId()
	 */
	public String getDeptId(User user);
	
	/**
	 * <p>
	 * 返回用户的默认机构
	 * </p>
	 * 
	 * @param user 用户对象
	 * @return 用户所在机构（考虑管理员、管辖机构等情况），同User.getDept()
	 */
	public Dept getDept(User user);
	
	/**
	 * <p>
	 * 返回用户所在的下级机构树型（默认解析逻辑机构、包含全部类型）
	 * </p>
	 * 
	 * @param user 用户对象
	 * @return 机构列表
	 */
	public List<Dept> getDeptTree(User user);
	
	/**
	 * <p>
	 * 返回用户所在的下级机构树型
	 * </p>
	 * 
	 * @param user 用户对象
	 * @param includeLogic 是否解析逻辑机构
	 * @return 机构列表
	 */
	public List<Dept> getDeptTree(User user, boolean includeLogic);
	
	
	
	

	
	///////////////////////////  用户编辑相关   /////////////////////////////////////
	
	
	
	
	/**
	 * 添加新用户
	 * @param user 用户对象
	 * @param loginPass 登录密码
	 */
	public void add(User user, String loginPass);
	
	/**
	 * 修改用户
	 * @param user 用户对象
	 */
	public void modify(User user);
	
	/**
	 * 修改用户，同时指定密码
	 * @param user 用户对象
	 * @param loginPass 新密码
	 */
	public void modify(User user, String loginPass);
	
	/**
	 * 删除用户
	 * @param ids 要删除的用户ID，多个ID时中间用逗号分割
	 */
	public void remove(String ids);
	
	/**
	 * 还原用户
	 * @param ids 要还原的用户ID，多个ID时中间用逗号分割
	 */
	public void reback(String ids);
	
	/**
	 * 移动用户到指定机构
	 * @param ids 要移动的用户ID，多个ID时中间用逗号分割
	 * @param pidTo 目标机构ID
	 */
	public void move(String ids, String pidTo);
	
	/**
	 * 设置管辖机构
	 * @param ids 要设置管辖的用户ID，多个ID时中间用逗号分割
	 * @param pidTo 目标机构ID
	 */
	public void setMgrDept(String ids, String pidTo);
	
	/**
	 * <p>
	 * 修改指定登录用户的密码为随机密码
	 * </p>
	 * 
	 * @param loginName 登录名
	 * @param len 密码长度
	 * @return 新生成的随机密码
	 */
	public String chgPassRandom(String loginName, int len);

	
	
	

	
	///////////////////////////  用户导入导出   /////////////////////////////////////
	
	
	
	
	
	
	/**
	 * <p>
	 * 导出用户
	 * </p>
	 * 
	 * @param dept 所在机构
	 * @return 电子表格对象
	 */
	public XlsWriter export(Dept dept);
	
	/**
	 * <p>
	 * 导出用户
	 * </p>
	 * 
	 * @param dept 所在机构
	 * @param type 用户类型
	 * @return 电子表格对象
	 */
	public XlsWriter export(Dept dept, String type);
	
	/**
	 * <p>
	 * 导入用户
	 * </p>
	 * 
	 * @param deptId 所在机构ID
	 * @param file 导入文件
	 * @return 导入条数
	 */
	public int imports(String deptId, File file);
	
	/**
	 * <p>
	 * 导入用户
	 * </p>
	 * 
	 * @param deptId 所在机构ID
	 * @param file 导入文件
	 * @param type 用户类型
	 * @param multiDept 是否同时向多个机构导入
	 * @return 导入条数
	 */
	public int imports(String deptId, File file, String type, boolean multiDept);
	

}