package itez.plat.main.model;

import java.util.Date;

import org.apache.commons.lang3.ArrayUtils;

import itez.core.launch.JW;
import itez.core.runtime.EContext;
import itez.core.runtime.service.Ioc;
import itez.kit.EDate;
import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.log.ELogBase;
import itez.plat.main.service.UserService;

/**
 * 账号锁定详情
 * @author netwild
 *
 */
public class UserLockedInfo {

	private String uid;				//UserId
	private int failCnt;			//密码错误次数（超过最高次数将被临时锁定）
	private boolean locked;			//是否已被锁定
	private int lockCnt;			//已连续锁定次数（超过最高次数将被永久锁定）
	private boolean forever;		//是否已被永久锁定
	private Date times;				//最后一次锁定时间
	private PolicyLock model;		//锁定规则

	private static UserService userSer = Ioc.get(UserService.class);
	private static final ELogBase log = ELog.log(UserLockedInfo.class);
	
	/**
	 * 第一次密码错误时，实例化锁定信息
	 * @param uid
	 * @param model
	 */
	public UserLockedInfo(String uid, PolicyLock model) {
		this.uid = uid;
		this.failCnt = 1;
		this.locked = false;
		this.lockCnt = 0;
		this.forever = false;
		this.times = EDate.getDate();
		this.model = model;
	}
	
	/**
	 * 重复尝试
	 */
	public boolean retry(){
		if(!locked){ //还未锁定，直接返回允许登录
			return true;
		}
		if(forever){ //永久锁定，直接返回无法登录
			return false;
		}
		if(EDate.isExpire(getExpire())){ //已经超过锁定时限，允许重新尝试登录
			failCnt = 0;
			locked = false;
			return true;
		}
		times = EDate.getDate(); //更新锁定时间，重新计算解锁时间
		return false;
	}
	
	/**
	 * 返回解锁时间
	 * @return
	 */
	public Date getExpire(){
		return EDate.addMinute(this.times, model.getLockLen());
	}
	
	/**
	 * 返回剩余解锁分钟数
	 * @return
	 */
	public long surMinutes(){
		return EDate.diff(EDate.getDate(), getExpire()).toMinutes();
	}
	
	/**
	 * 再次输入错误
	 */
	public void failAgain(){
		failCnt += 1;
		if(failCnt == model.getRetryCnt()){ //达到错误次数限制，临时锁定该账号
			locked = true;
			times = EDate.getDate();	//更新锁定时间
			lockCnt += 1;				//统计锁定次数
			log.info("账号临时锁定{}：[domain]{} [session]{} [uid]{}", lockCnt, EContext.getAttr().getDomain(), EContext.getSession().getId(), uid);
		}
		if(lockCnt == model.getLockCnt()){ //达到锁定次数限制，永久锁定该账号
			forever = true;
			if(EStr.notEmpty(uid)){
				User user = userSer.findById(uid);
				if(!ArrayUtils.contains(JW.IGNORE_SUP_NAME, user.getLoginName())){
					userSer.modify(user.setLocked(true).setLockTime(times));
				}
			}
			log.info("账号永久锁定：[domain]{} [session]{} [uid]{}", EContext.getAttr().getDomain(), EContext.getSession().getId(), uid);
		}
	}

	public String getUid() {
		return uid;
	}

	public int getFailCnt() {
		return failCnt;
	}

	public boolean isLocked() {
		return locked;
	}

	public int getLockCnt() {
		return lockCnt;
	}

	public boolean isForever() {
		return forever;
	}

	public Date getTimes() {
		return times;
	}

	public PolicyLock getModel() {
		return model;
	}
	
}
