package itez.plat.main.controller;

import java.io.File;
import java.util.List;

import javax.sql.DataSource;

import com.beust.jcommander.internal.Lists;
import com.google.inject.Inject;
import com.jfinal.plugin.activerecord.Record;
import com.jfinal.plugin.activerecord.generator.TableMeta;

import itez.core.runtime.auth.AuthCode;
import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.dbo.DbManager;
import itez.core.wrapper.dbo.generator.EMetaBuilder;
import itez.kit.ECommand;
import itez.kit.EJson;
import itez.kit.EPara;
import itez.kit.EStr;
import itez.kit.fileup.EFileKit;
import itez.kit.fileup.FileItem;
import itez.kit.restful.Result;
import itez.kit.terminal.TerminalUtil;
import itez.plat.main.ModuleConfig;
import itez.plat.main.model.Module;
import itez.plat.main.model.TerminalEnvs;
import itez.plat.main.service.ModuleService;
import itez.plat.main.service.TerminalEnvsService;
import itez.plat.task.SupPass2Term;
import itez.plat.wrapper.controller.EControllerSup;

/**
 * <p>
 * 超管工具箱（本地访问终端）
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月14日 上午8:40:25
 */
@ControllerDefine(key = "/sup", summary = "超管工具箱", view = "/sup")
public class SupController extends EControllerSup {

	@Inject
	TerminalEnvsService envSer;
	
	@Inject
	ModuleService moduleSer;
	
	/**
	 * <p>
	 * 手动向云端签到
	 * </p>
	 *
	 */
	public void checkin(){
		try {
			TerminalUtil.checkIn();
			renderJson(Result.success());
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 更新超管密码
	 * </p>
	 *
	 */
	public void chgPass(){
		try {
			new SupPass2Term("itez").run();
			renderJson(Result.success());
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 配置终端环境
	 * </p>
	 *
	 */
	public void env(){
		TerminalEnvs env = envSer.selectFirst();
		setAttr("env", EJson.toJson(env));
		render("env/index.html");
	}
	
	/**
	 * <p>
	 * 配置终端环境提交
	 * </p>
	 *
	 */
	public void envEvent(){
		EPara paras = paramPack();
		TerminalEnvs env = paras.getModel(TerminalEnvs.class);
		env.setJavaHome(env.getJavaHome().replace("\\", "/"));
		env.setDbHome(env.getDbHome().replace("\\", "/"));
		env.setWebRoot(env.getWebRoot().replace("\\", "/"));
		env.setDbBakRoot(env.getDbBakRoot().replace("\\", "/"));
		env.setLogRootJWinner(env.getLogRootJWinner().replace("\\", "/"));
		env.setTomcatHome1(env.getTomcatHome1().replace("\\", "/"));
		if(EStr.notEmpty(env.getTomcatHome2())) env.setTomcatHome2(env.getTomcatHome2().replace("\\", "/"));
		if(EStr.notEmpty(env.getTomcatHome3())) env.setTomcatHome3(env.getTomcatHome3().replace("\\", "/"));
		envSer.saveOrUpdate(env);
		renderJson(Result.success());
	}
	
	/**
	 * <p>
	 * 终端Shell命令
	 * </p>
	 *
	 */
	public void cmd(){
		setAttr("os", ECommand.getInstance().getOs().getCaption());
		render("cmd/index.html");
	}
	
	/**
	 * <p>
	 * 执行脚本命令
	 * </p>
	 * 
	 * @param cmd 命令内容
	 * @param envs 环境变量，可选。（格式：param1=var1,param2=var2...）
	 * @param path 目录位置，可选。
	 */
	public void doCmd(String cmd, String envs, String path){
		Result ret = CommandKit.cmd(cmd, envs, path);
		renderJson(ret);
	}
	
	/**
	 * <p>
	 * 重启Tomcat时需要的相关信息
	 * </p>
	 *
	 */
	public void doRestartBefore(){
		TerminalEnvs env = envSer.selectFirst();
		if(env == null){
			renderJson(Result.fail("请先设置 <code>终端环境配置 > Tomcat安装路径</code> 相关参数"));
			return;
		}
		renderJson(Result.success("env", env));
	}
	
	/**
	 * <p>
	 * 重启Tomcat
	 * </p>
	 *
	 */
	public void doRestart(String tomcatHome, String mode){
		Result ret = CommandKit.restartTomcat(tomcatHome, mode);
		renderJson(ret);
	}
	
	/**
	 * 备份数据
	 */
	public void dump(String moduleCode){
		TerminalEnvs env = envSer.selectFirst();
		if(env == null){
			renderErrMsg(AuthCode.LOGIC, "请先设置 <code>终端环境配置 > 数据库备份目录</code> 参数");
			return;
		}
		List<Module> modules = moduleSer.getByComp();
		if(EStr.isEmpty(moduleCode)) moduleCode = modules.get(0).getCode();
		setAttr("dbBakRoot", env.getDbBakRoot());
		setAttr("moduleCode", moduleCode);
		setAttr("modules", EJson.toJsonFilter(modules, "code", "caption"));
		render("dump/index.html");
	}
	
	/**
	 * 返回指定模块的table列表
	 * @param moduleCode
	 */
	public void getTableList(String moduleCode){
		DataSource ds = DbManager.me.getDataSource(moduleCode);
		String prefixes = moduleCode.equals(ModuleConfig.MODULE_CODE) ? "base" : moduleCode;
		EMetaBuilder builder = new EMetaBuilder(ds, prefixes.concat("_"));
		List<TableMeta> tableMetas = builder.build();
		List<Record> tables = Lists.newArrayList();
		tableMetas.forEach(meta -> {
			Record rec = new Record();
			rec.set("name", meta.name).set("remarks", meta.remarks);
			tables.add(rec);
		});
		renderJson(Result.success("tables", tables));
	}
	
	/**
	 * <p>
	 * 备份数据库
	 * </p>
	 * 
	 * @param moduleCode
	 */
	public void doDump(String moduleCode, String tabs){
		TerminalEnvs env = envSer.selectFirst();
		if(env == null){
			renderJson(Result.fail("请先配置终端环境中的“数据库备份主目录”"));
			return;
		}
		Result ret = CommandKit.dump(moduleCode, env.getDbHome() + "/bin", tabs, env.getDbBakRoot());
		renderJson(ret);
	}
	
	/**
	 * 查看备份文件列表
	 */
	public void listDump(String moduleCode){
		TerminalEnvs env = envSer.selectFirst();
		String bakRoot = env.getDbBakRoot();
		List<FileItem> files = EFileKit.localList(bakRoot, "", "/" + moduleCode, false);
		renderJson(Result.success("files", files));
	}
	
	/**
	 * 下载备份文件
	 * @param fileName
	 */
	public void downDump(String moduleCode, String fileName){
		TerminalEnvs env = envSer.selectFirst();
		String bakRoot = env.getDbBakRoot();
		String dumpPath = EStr.join(bakRoot, EStr.FileSep, moduleCode, EStr.FileSep, fileName);
		File file = new File(dumpPath);
		renderFile(file);
	}
	
	/**
	 * <p>
	 * 日志浏览器
	 * </p>
	 * 
	 */
	public void log(String sub){
		TerminalEnvs env = envSer.selectFirst();
		if(env == null){
			renderErrHelp(AuthCode.LOGIC, "请先配置终端环境（JWinner日志目录）", attr().getCtrl() + "/env");
			return;
		}
		if(EStr.isEmpty(sub)) sub = "";
		String path = env.getLogRootJWinner() + sub;
		List<FileItem> files = EFileKit.localList(path, null, "", true);
		setAttr("sub", sub);
		setAttr("files", EJson.toJson(files));
		render("log/index.html");
	}
	
	/**
	 * <p>
	 * 日志详情
	 * </p>
	 * 
	 * @param sub
	 * @param fileName
	 */
	public void logDetail(String sub, String fileName){
		TerminalEnvs env = envSer.selectFirst();
		String path = EStr.join(env.getLogRootJWinner(), sub, "/", fileName);
		File file = EFileKit.localFile(path);
		FileItem fileItem = new FileItem(file).loadContent();
		setAttr("sub", sub);
		setAttr("fileName", fileName);
		setAttr("file", fileItem);
		render("log/detail.html");
	}
	
}
