package itez.plat.main.controller;

import java.util.List;

import com.google.inject.Inject;

import itez.core.runtime.modules.ModuleManager;
import itez.core.util.ECacheKit;
import itez.core.wrapper.controller.ControllerDefine;
import itez.kit.EJson;
import itez.kit.EPara;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.restful.Result;
import itez.plat.main.model.Comp;
import itez.plat.main.model.Module;
import itez.plat.main.service.CompService;
import itez.plat.main.service.MenuService;
import itez.plat.main.service.ModuleService;
import itez.plat.main.service.PermService;
import itez.plat.main.service.RoleService;
import itez.plat.wrapper.controller.EControllerSup;

@ControllerDefine(key = "/sup/comp", summary = "租户管理", view = "/sup/comp")
public class SupCompController extends EControllerSup {

	@Inject
	CompService compSer;
	
	@Inject
	ModuleService moduleSer;
	
	@Inject
	MenuService menuSer;

	@Inject
	PermService permSer;

	@Inject
	RoleService roleSer;
	
	public void index(){
		List<Module> modules = moduleSer.selectAll();
		String moduleMain = itez.plat.main.ModuleConfig.MODULE_CODE;             //主模块
		String moduleDef = ModuleManager.me.getDefaultModule().getModuleCode();  //默认模块
		if(!moduleMain.equals(moduleDef)) moduleMain = moduleMain.concat(",").concat(moduleDef);
		setAttr("modules", EJson.toJson(modules));
		setAttr("moduleMain", moduleMain);
		setAttr("moduleDef", moduleDef);
		setAttr("domainDef", EProp.DomainDefault);
		setAttr("domainTemp", EProp.DomainTemplate);
		render("index.html");
	}
	
	public void getData(){
		List<Comp> comps = compSer.getAllComp();
		renderJson(Result.success("comps", comps));
	}
	
	public void formEvent(){
		EPara paras = paramPack();
		Comp comp = paras.getModel(Comp.class);
		Comp exist = null;			//原租户
		String origModules = null;  //原启用模块
		if(EStr.isEmpty(comp.getId())){
			exist = compSer.getByDomain(comp.getDomain());
			if(exist != null){
				renderJson(Result.fail("该子域已经存在，请更换其他子域。"));
				return;
			}
			comp.setPid(attr().getCompDef().getId());
		}else{
			exist = compSer.findById(comp.getId());
			origModules = exist.getModules();
		}
		String codeMain = itez.plat.main.ModuleConfig.MODULE_CODE;
		String[] modules = paras.getValues("modules");
		if(modules == null || EStr.findInArr(modules, codeMain) == -1){
			renderJson(Result.fail("授权模块中，“基础平台” 为必选模块。"));
			return;
		}
		if(EStr.findInArr(modules, comp.getIndex()) == -1){
			renderJson(Result.fail("默认模块必须包含在授权模块内。"));
			return;
		}
		
		//自动附加默认模块的根路径
		Module defModule = moduleSer.findByCode(comp.getIndex());
		String defModulePath = String.join(":", defModule.getCode(), defModule.getPath());
		comp.setIndex(defModulePath);
		
		compSer.saveOrUpdate(comp);
		initComp(comp, origModules); //初始化租户环境数据
		renderJson(Result.success());
	}
	
	/**
	 * 删除租户
	 * @param id
	 */
	public void remove(String id){
		compSer.disable(id);
		renderJson(Result.success());
	}
	
	/**
	 * 初始化租户环境数据
	 * @param comp
	 * @param origModules
	 */
	private void initComp(Comp comp, String origModules){
		ECacheKit.removeAll(ECacheKit.appendTenant(comp.getDomain(), "MODULE_COMP"));
		initModulePerms(comp, origModules);
		initModuleRoles(comp, origModules);
		initModuleMenus(comp, origModules);
	}

	/**
	 * 初始化新租户的权限
	 * @param comp
	 * @param origModules
	 */
	private void initModulePerms(Comp comp, String origModules){
		ECacheKit.removeAllBase(ECacheKit.appendTenant(comp.getDomain(), "PERM_ALL"));
		ECacheKit.removeAllBase(ECacheKit.appendTenant(comp.getDomain(), "PERM_BY_CODES"));
		String moduleCodeStr = comp.getModules();
		String[] moduleCodeArr = moduleCodeStr.split(",");
		for(String moduleCode : moduleCodeArr){
			permSer.importJson(comp.getDomain(), moduleCode);
		}
	}

	/**
	 * 初始化新租户的角色
	 * @param comp
	 * @param origModules
	 */
	private void initModuleRoles(Comp comp, String origModules){
		ECacheKit.removeAllBase(ECacheKit.appendTenant(comp.getDomain(), "ROLE_ALL"));
		ECacheKit.removeAllBase(ECacheKit.appendTenant(comp.getDomain(), "ROLE_BY_CODES"));
		String moduleCodeStr = comp.getModules();
		String[] moduleCodeArr = moduleCodeStr.split(",");
		for(String moduleCode : moduleCodeArr){
			roleSer.importJson(comp.getDomain(), moduleCode);
		}
	}
	
	/**
	 * 初始化或更新租户的菜单
	 * @param comp
	 * @param origModules
	 */
	private void initModuleMenus(Comp comp, String origModules){
		String mtype = "menu";
		String moduleCodeStr = comp.getModules();
		String[] moduleCodeArr = moduleCodeStr.split(",");
		
		//删除不再授权的模块菜单
		if(EStr.notEmpty(origModules)){
			String[] origModuleCodeArr = origModules.split(",");
			for(int i=0, len=origModuleCodeArr.length; i<len; i++){
				String origModuleCode = origModuleCodeArr[i].trim();
				if(EStr.findInArr(moduleCodeArr, origModuleCode) == -1) menuSer.delMenuByModule(comp.getDomain(), mtype, origModuleCode);
			}
		}
		
		//初始化新授权的模块菜单
		for(int i=0, len=moduleCodeArr.length; i<len; i++){
			String moduleCode = moduleCodeArr[i].trim();
			menuSer.init(comp.getDomain(), mtype, moduleCode);
		}
	}
	
	/**
	 * 租户数据初始化（手动同步）
	 * @param type 数据类型（perm/role/menu）
	 * @param domain 租户子域
	 * @param code 模块代码
	 */
	public void inits(String type, String domain, String code){
		try {
			if(type.equals("perm")){ //权限
				permSer.importJson(domain, code);
			}else if(type.equals("role")){ //角色
				roleSer.importJson(domain, code);
			}else if(type.equals("menu")){ //菜单
				menuSer.delMenuByModule(domain, "menu", code);
				menuSer.init(domain, "menu", code);
			}
			renderJson(Result.success());
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
}
