/**
 * 
 */
package itez.plat.main.controller;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;

import com.google.inject.Inject;
import com.jfinal.upload.UploadFile;

import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.controller.EController;
import itez.core.wrapper.dbo.exec.DbExec;
import itez.core.wrapper.dbo.exec.DbResult;
import itez.kit.EStr;
import itez.kit.restful.EMap;
import itez.kit.restful.Result;
import itez.kit.terminal.TerminalUtil;
import itez.plat.main.model.TerminalEnvs;
import itez.plat.main.service.TerminalEnvsService;

/**
 * <p>
 * 终端接口控制器（远程访问终端）
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月19日 下午10:01:57
 */
@ControllerDefine(key = "/sup/api", summary = "终端接口控制器", view = "/")
public class SupApiController extends EController {

	@Inject
	TerminalEnvsService envSer;

	/**
	 * <p>
	 * 接收并执行SQL更新
	 * </p>
	 * 
	 * <p>
	 * 参数格式：/plat/sup/api/dosql?module=<模块别名>&sql=<SQL内容>
	 * </p>
	 *
	 */
	public void dosql(){
		EMap params = getParaData();
		try {
			if(EStr.isEmpty(params.getStr("module")) || EStr.isEmpty(params.getStr("sql"))){
				renderJson(Result.fail("参数为空"));
				return;
			}
			//接收请求时，先解密，再校验签名
			//无加密参数时省略TerminalUtil.decrypt(params, keys);
			if(!TerminalUtil.valiSign(params, new String[]{ "module", "sql" })){
				renderJson(Result.fail("校验签名失败：" + params.getStr("module") + "，" + params.getStr("sql")));
				return;
			}
			DbResult res = DbExec.execute(params.getStr("module"), params.getStr("sql"));
			renderJson(Result.success("result", res));
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 重启指定的服务器负载
	 * </p>
	 * 
	 * <p>
	 * 参数格式：/plat/sup/api/dorestart?index=<负载序号>&mode=<启动类型（cmd/service）>
	 * </p>
	 *
	 */
	public void dorestart(){
		EMap params = getParaData();
		try {
			if(EStr.isEmpty(params.getStr("index")) || EStr.isEmpty(params.getStr("mode"))){
				renderJson(Result.fail("参数为空"));
				return;
			}
			//接收请求时，先解密，再校验签名
			//无加密参数时省略TerminalUtil.decrypt(params, keys);
			if(!TerminalUtil.valiSign(params, new String[]{ "index", "mode" })){
				renderJson(Result.fail("校验签名失败：" + params.getStr("index") + "，" + params.getStr("mode")));
				return;
			}
			TerminalEnvs env = envSer.selectFirst();
			if(env == null){
				renderJson(Result.fail("请先设置 <code>终端环境配置 > Tomcat安装路径</code> 相关参数"));
				return;
			}
			Result ret = CommandKit.restartTomcat(env.getStr("tomcatHome" + params.getStr("index")), params.getStr("mode"));
			renderJson(ret);
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
	/**
	 * <p>
	 * 接收并覆盖JAR包
	 * </p>
	 * 
	 * <p>
	 * 参数格式：/plat/sup/api/dojar?jar=File
	 * </p>
	 *
	 */
	public void dojar(){
		UploadFile jarUp = getFile("jar");
		if(jarUp == null){
			renderJson(Result.fail("未接收到JAR包"));
			return;
		}
		File jar = jarUp.getFile();
		if(jar == null){
			renderJson(Result.fail("JAR文件是空的"));
			return;
		}
		try {
			TerminalEnvs env = envSer.selectFirst();
			if(env == null){
				renderJson(Result.fail("请先设置 <code>终端环境配置 > Tomcat安装路径</code> 相关参数"));
				return;
			}
			String webRoot = env.getWebRoot();
			if(EStr.isEmpty(webRoot)){
				renderJson(Result.fail("未配置WebRoot的目录位置"));
				return;
			}
			String jarRoot = webRoot + "/WEB-INF/lib";
			File jarFile = new File(jarRoot + '/' + jar.getName());
			if(jarFile.exists()) jarFile.delete();
			Files.move(jar.toPath(), jarFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
			renderJson(Result.success());
		} catch (Exception e) {
			renderJson(Result.fail(e.getMessage()));
		}
	}
	
}
