package itez.plat.main.controller;

import java.util.List;

import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.domain.AppTokenExchangeSubElement;
import com.alipay.api.request.AlipayOpenAuthTokenAppRequest;
import com.alipay.api.response.AlipayOpenAuthTokenAppResponse;
import com.jfinal.kit.Prop;
import com.jfinal.kit.PropKit;

import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.controller.EController;
import itez.kit.EBase64;
import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.EWeb;
import itez.kit.exception.ExceptionKit;
import itez.kit.log.ELogBase;
import itez.kit.restful.EMap;

/**
 * <p>
 * 支付宝服务商授权控制器（云端）
 * 文档：https://opendocs.alipay.com/isv/10467/xldcyq
 * </p>
 * 
 * <p>Copyright(C) 2017-2022 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2022年3月2日 上午11:24:28
 */
@ControllerDefine(key = "/payment/isv/ali/cloud", summary = "支付宝服务商授权控制器", view = "/payment")
public class IsvAliCloudController extends EController {
	
	//加载第三方应用授权配置文件
	private final static Prop prop = PropKit.use("isv_ali.properties");
	private final static String AUTH_URL = "https://openauth.alipay.com/oauth2/appToAppBatchAuth.htm?app_id=%s&application_type=%s&redirect_uri=%s&state=%s";
	
	private static ELogBase log = ELog.log(IsvAliCloudController.class);
	
	/**
	 * <p>
	 * 发起授权（跳转到支付宝授权页面，带二维码，客户确认后会跳转到授权回调入口）
	 * 支付宝授权请求模板：https://openauth.alipay.com/oauth2/appToAppBatchAuth.htm?app_id=第三方应用APPID&application_type=限制的商家应用类型&redirect_uri=UrlEncode(redirect_uri)&state=自定义参数值
	 * </p>
	 * 
	 * @param state 本地回调地址，应提前进行UrlEncoder
	 */
	public void auth(String state){
		String th_app_id = prop.get("APP_ID");
		String application_type = "WEBAPP";
		String redirect_uri = prop.get("RETURN_URL");
		
		log.info("开始发起支付宝授权申请：第三方应用APPID（{}），本地回传地址（{}）", th_app_id, state);
		
		redirect_uri = EWeb.UrlEncoder(redirect_uri);
		state = EBase64.encode(state);
		String url = String.format(AUTH_URL, th_app_id, application_type, redirect_uri, state);
		
		redirect(url);
	}
	
	/**
	 * <p>
	 * 授权回调（该入口应配置到支付宝开放平台第三方应用的授权回调地址中）
	 * </p>
	 * 
	 * @param app_id 客户选择授权的应用AppId
	 * @param app_auth_code 授权码，可利用该授权码换取授权令牌：app_auth_token
	 * @param state 发起授权时传入的本地回调地址，最终会将授权令牌回传到该地址
	 */
	public void index(String app_id, String app_auth_code, String state){
		
		String msg = "授权错误！"; //错误描述

		log.info("客户完成支付宝授权，接收到参数：app_id（{}），app_auth_code（{}），state（{}）", app_id, app_auth_code, state);
				
		String th_app_id = prop.get("APP_ID");
		String th_pri_key = prop.get("APP_PRI_KEY");
		String th_pub_key = prop.get("ALI_PUB_KEY");
		String FORMAT = "json";
		String CHARSET = EStr.Encoding;
		String SIGN_TYPE = "RSA2";
		AlipayClient client = new DefaultAlipayClient("https://openapi.alipay.com/gateway.do", th_app_id, th_pri_key, FORMAT, CHARSET, th_pub_key, SIGN_TYPE);
		AlipayOpenAuthTokenAppRequest req = new AlipayOpenAuthTokenAppRequest();
		EMap querys = EMap.by("grant_type", "authorization_code").set("code", app_auth_code);
		String biz = querys.toJson();

		log.info("准备请求授权令牌：bizContent（{}）", biz);
		
		req.setBizContent(biz);
		try {
			AlipayOpenAuthTokenAppResponse res = client.execute(req);
			if(res.isSuccess()){
				List<AppTokenExchangeSubElement> appTokens = res.getTokens();
				if(appTokens.size() == 0){
					msg = "Tokens列表为空";
					log.info("授权令牌获取失败：{}", msg);
				}else{
					AppTokenExchangeSubElement appToken = appTokens.get(0);
					String uid = appToken.getUserId();
					String appid = appToken.getAuthAppId();
					String token = appToken.getAppAuthToken();
					log.info("授权令牌获取成功：uid（{}），appid（{}），token（{}）", uid, appid, token);
					
					state = EBase64.decode(state);
					state = EWeb.UrlDecoder(state);
					String url = EStr.join(state, "?uid=", uid, "&appid=", appid, "&token=", token);
					log.info("授权回调地址：{}", url);
					redirect(url);
					return;
				}
			}else{
				msg = res.getMsg();
				log.info("授权令牌获取失败：code（{}），msg（{}）", res.getCode(), msg);
			}
		} catch (Exception e) {
			msg = ExceptionKit.getExceptionMore(e);
			log.info("授权令牌获取时出现错误：{}", msg);
		}
		
		renderHtml(msg);
	}
	
}
