package itez.plat.main.controller;

import java.io.File;
import java.util.List;

import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.jfinal.plugin.activerecord.Record;

import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.auth.AuthRequire;
import itez.core.runtime.service.common.IUser;
import itez.core.runtime.service.common.IUserService;
import itez.core.wrapper.controller.ControllerDefine;
import itez.kit.EJson;
import itez.kit.ELog;
import itez.kit.EPara;
import itez.kit.EStr;
import itez.kit.log.ELogBase;
import itez.kit.poi.XlsWriter;
import itez.kit.restful.Result;
import itez.plat.main.ModuleConfig;
import itez.plat.main.model.Dept;
import itez.plat.main.model.Module;
import itez.plat.main.model.User;
import itez.plat.main.service.DeptService;
import itez.plat.main.service.ModuleService;
import itez.plat.main.service.RoleService;
import itez.plat.main.service.UserLoginService;
import itez.plat.main.service.UserService;
import itez.plat.wrapper.controller.EControllerMgr;
import itez.tp.ApiCommon.USR_SOURCE;

@AuthRequire.Role(value = ModuleConfig.RoleBaseAdmin)
@ControllerDefine(key = "/insUser", summary = "成员管理", view = "/")
public class InsUserController extends EControllerMgr {

	@Inject
	UserService us;

	@Inject
	IUserService userSer;
	
	@Inject
	UserLoginService loginSer;
	
	@Inject
	DeptService ds;
	
	@Inject
	RoleService rs;
	
	@Inject
	ModuleService ms;
	
	private static final ELogBase log = ELog.log(InsUserController.class);
	
	public void index(String type, String pid){
		User user = (User)session().getUser();
		String moduleIds = user.getModuleIds();
		List<Module> modules = Lists.newArrayList();
		String roleIds = user.getRoleIds();
		List<Record> roles = Lists.newArrayList();
		if(EStr.notEmpty(moduleIds)) modules = ms.findByCodes(moduleIds);
		if(EStr.notEmpty(roleIds)) roles = rs.getByCodeInModule(moduleIds, roleIds);
		if(EStr.isEmpty(pid)) pid = user.getRootDeptId();
		setAttr("current", user.getLoginName());
		setAttr("type", type);
		setAttr("pid", pid);
		setAttr("modules", EJson.toJsonFilter(modules, "code", "caption"));
		setAttr("roles", EJson.toJsonFilter(roles, "code", "caption", "moduleId", "moduleCaption"));
		setAttr("level", user.getLevel());
		render("insUser.html");
	}
	
	/**
	 * 返回用户数据（成员管理、用户选择框共用）
	 * @param type 用户类型（为空时所有）
	 * @param pid 所在机构ID（为空时不返回用户数据）
	 * @param root 是否从根节点返回机构数据（true：根节点，false：当前用户所在机构）
	 * @param includeTree 是否同时返回机构树型
	 */
	@AuthRequire.Logined
	public void getData(String type, String pid, Boolean root, Boolean includeTree){
		Result ret = Result.success();
		if(includeTree){
			List<Dept> deptTree;
			if(EStr.notEmpty(root) && root){
				deptTree = ds.getDeptTree();
			}else{
				User user = (User)session().getUser();
				deptTree = us.getDeptTree(user);
			}
			ret.set("tree", EJson.toJsonFilter(deptTree, "id", "code", "caption", "path", "logic"));
		}
		if(EStr.notEmpty(pid)){
			List<User> users = us.getInsUsers(pid, type);
			ret.set("users", users);
		}
		renderJson(ret);
	}
	
	public void formEvent(){
		String id = getPara("id");
		String pid = getPara("deptId");
		String type = getPara("type");
		String caption = getPara("caption");
		String loginName = getPara("loginName");
		String loginPass = getPara("loginPass");
		String idNum = getPara("idNum");
		String num = getPara("num");
		String phone = getPara("phone");
		String email = getPara("email");
		String sex = getPara("sex");
		Integer level = getInt("level", 0);
		String moduleIds = EStr.parseArray(getParaValues("moduleIds"));
		
		IUser exist = userSer.findByLogin(loginName);
		if(exist != null && (EStr.isEmpty(id) || !exist.getId().equals(id))){
			renderJson(Result.fail("登录账号已存在！"));
			return;
		}
		if(EStr.notEmpty(idNum)){
			exist = userSer.findByIdNum(idNum);
			if(exist != null && (EStr.isEmpty(id) || !exist.getId().equals(id))){
				renderJson(Result.fail("身份证号已存在！"));
				return;
			}
		}
		if(EStr.notEmpty(phone)){
			exist = userSer.findByPhone(phone);
			if(exist != null && (EStr.isEmpty(id) || !exist.getId().equals(id))){
				renderJson(Result.fail("手机号已存在！"));
				return;
			}
		}
		if(EStr.notEmpty(email)){
			exist = userSer.findByEmail(email);
			if(exist != null && (EStr.isEmpty(id) || !exist.getId().equals(id))){
				renderJson(Result.fail("Email已存在！"));
				return;
			}
		}
		
		User user = new User();
		user.setCaption(caption);
		user.setLoginName(loginName);
		user.setType(type);
		user.setDeptId(pid);
		user.setIdNum(idNum);
		user.setNum(num);
		user.setPhone(phone);
		user.setEmail(email);
		user.setSex(sex);
		user.setLevel(level);
		user.setModuleIds(moduleIds);
		
		if(EStr.isEmpty(id)){
			user.setDomain(attr().getDomain());
			user.setMember(1);
			user.setRegType(USR_SOURCE.MGR.name());
			us.add(user, loginPass);
		}else{
			user.setId(id);
			us.modify(user, loginPass);
		}
		
		renderJson(Result.success());
	}
	
	/**
	 * 设置用户授权范围（模块、角色）
	 */
	public void roleEvent(){
		String id = getPara("id");
		String moduleIds = EStr.parseArray(getParaValues("moduleIds"));
		String roleIds = EStr.parseArray(getParaValues("roleIds"));
		
		User user = us.findById(id);
		user.setModuleIds(moduleIds);
		user.setRoleIds(roleIds);
		us.modify(user);
		
		renderJson(Result.success());
	}
	
	/**
	 * 删除用户
	 * @param ids
	 */
	public void remove(String ids){
		us.remove(ids);
		log.info("删除成员：{}", ids);
		renderJson(Result.success());
	}
	
	/**
	 * 账号解锁
	 * @param ids
	 */
	public void unlock(String ids){
		User user = us.findById(ids);
		us.modify(user.setLocked(false));
		loginSer.removeLockList(ids);

		log.info("解锁成员：[id]{} [caption]{}", ids, user.getCaption());
		renderJson(Result.success());
	}
	
	/**
	 * 移动到指定部门
	 * @param ids
	 * @param pid
	 */
	public void move(String ids, String pid){
		us.move(ids, pid);
		log.info("移动成员：[id]{} [pid]{}", ids, pid);
		renderJson(Result.success());
	}
	
	/**
	 * 管辖到指定部门
	 * @param ids
	 * @param pid
	 */
	public void toMgr(String ids, String pid){
		us.setMgrDept(ids, pid);
		log.info("设置成员管辖机构：[id]{} [pid]{}", ids, pid);
		renderJson(Result.success());
	}
	
	/**
	 * 导出用户
	 * @param deptId
	 * @param type
	 */
	public void exports(String deptId, String type){
		Dept dept;
		if(EStr.notEmpty(deptId)){
			dept = ds.findById(deptId);
		}else{
			dept = us.getDept((User)session().getUser());
		}
		XlsWriter xls = us.export(dept, type);
		render(xls);
		log.info("导出成员信息：[dept]{} [type]{}", deptId, type);
	}
	
	/**
	 * 开始导入
	 * @param deptId
	 * @param type
	 */
	public void imports(String deptId, String type){
		Dept dept = ds.findById(deptId);
		setAttr("dept", dept);
		setAttr("type", type);
		render("insUserImports.html");
	}
	
	/**
	 * 导入执行
	 */
	public void importsEvent(){
		EPara paras = paramPack(false);
		String deptId = paras.get("deptId");
		String type = paras.get("type");
		boolean multiDept = paras.get("multiDept", "").equals("ok");
		File file = paras.getFileObj("file");
		try {
			int cnt = us.imports(deptId, file, type, multiDept);
			setFlashMsg("导入成功，共导入 " + cnt + " 条成员信息。");
			log.info("导入成员信息：[cnt]{} [dept]{} [type]{} [multiDept]{}", cnt, deptId, type, multiDept);
			redirect(attr().getCtrl() + "?pid=" + deptId + "&type=" + type);
		} catch (Exception e) {
			setFlashMsg("导入失败，" + e.getMessage());
			redirect(attr().getCtrl() + "/imports?deptId=" + deptId + "&type=" + type);
		}
	}
	
	/**
	 * 用户查找（成员管理、用户选择框共用）
	 * @param type
	 * @param keyword
	 */
	public void search(String type, String keyword){
		Result ret = Result.success();
		if(EStr.notEmpty(keyword)){
			List<User> users = us.searchInsUser(keyword, null, type);
			if(users.size() > 0){
				users.forEach(u -> {
					Dept dept = u.getDept();
					if(dept != null) u.put("deptCaption", dept.getCaption());
				});
				ret.set("users", users);
			}
		}
		renderJson(ret);
	}
	
	/**
	 * <p>
	 * 账号管理
	 * </p>
	 *
	 */
	public void findUser(){
		render("findUser.html");
	}
	
	/**
	 * <p>
	 * 账号管理：查找用户
	 * </p>
	 *
	 */
	public void findUserEvent(String fieldName, String keyword){
		if(EStr.isEmpty(fieldName) || EStr.isEmpty(keyword)){
			renderJson(Result.fail("请填写查找条件"));
			return;
		}
		List<Record> users = us.searchByField(null, null, null, fieldName, keyword);
		renderJson(Result.success("users", users));
	}
	
	/**
	 * <p>
	 * 账号管理：重置密码
	 * </p>
	 * 
	 * @param id
	 */
	public void resetPass(){
		String id = getPara(0);
		if(EStr.isEmpty(id)){
			renderErrMsg(AuthCode.LOGIC, "未接收到有效的用户ID");
			return;
		}
		User user = us.findById(id);
		if(user == null || user.getUsed() == 0){
			renderErrMsg(AuthCode.LOGIC, "用户不存在");
			return;
		}
		setAttr("id", id);
		setAttr("caption", user.getCaption());
		render("resetPass.html");
	}
	
	/**
	 * <p>
	 * 账号管理：重置密码提交
	 * </p>
	 *
	 */
	public void resetPassEvent(String id, String pass, String passRep){
		if(EStr.isEmpty(pass)){
			renderJson(Result.fail("请填写新密码"));
			return;
		}else if(!pass.equals(passRep)){
			renderJson(Result.fail("两次输入的新密码不相同"));
			return;
		}
		User user = us.findById(id);
		us.modify(user, pass);
		log.info("重置成员密码：[id]{} [caption]{}", id, user.getCaption());
		renderJson(Result.success());
	}
	
}
