package itez.plat.main.controller;

import java.io.File;

import com.google.inject.Inject;
import com.jfinal.aop.Clear;
import com.jfinal.upload.UploadFile;

import itez.core.launch.JW;
import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.auth.AuthRequire;
import itez.core.runtime.service.common.IUserService;
import itez.core.wrapper.controller.ControllerDefine;
import itez.core.wrapper.render.ECaptchaRender;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERet;
import itez.kit.EStr;
import itez.kit.log.ELogBase;
import itez.plat.main.ModuleConfig;
import itez.plat.main.model.Dept;
import itez.plat.main.model.FileStore;
import itez.plat.main.model.User;
import itez.plat.main.service.CompService;
import itez.plat.main.service.DeptService;
import itez.plat.main.service.FileStoreService;
import itez.plat.main.service.MenuService;
import itez.plat.main.service.PolicyPwdService;
import itez.plat.main.service.UserLoginService;
import itez.plat.main.service.UserService;
import itez.plat.wrapper.controller.EControllerDef;

/**
 * 平台默认控制器
 * 只实现最通用的几个入口Action
 * 
 * @author netwild
 *
 */
@ControllerDefine(key = "/", summary = "平台首页", view = "/")
public class IndexController extends EControllerDef {

	@Inject
	CompService compSer;
	
	@Inject
	IUserService userService;

	@Inject
	UserService userSer;
	
	@Inject
	UserLoginService loginSer;
	
	@Inject
	PolicyPwdService pwdSer;
	
	@Inject
	MenuService menuSer;

	@Inject
	DeptService deptSer;
	
	@Inject
	FileStoreService fileStoreSer;
	
	private final static String SESS_KEY_POLICY = "ACC_POLICY_CODE";	//SESSION键名：当前使用的方案代码
	
	private static final ELogBase log = ELog.log(IndexController.class);
	
	/**
	 * 平台首页
	 */
	public void index(){
		render(EProp.TempIndexPath);
	}

	/**
	 * 验证码
	 */
	@Clear
	public void valicode(){
		render(new ECaptchaRender());
	}
	
	/**
	 * 二维码
	 * @param code
	 * @param width
	 * @param height
	 */
	@Clear
	public void qrcode(String code, Integer width, Integer height){
		if(EStr.isEmpty(code)){
			renderNull();
			return;
		}
		if(EStr.isEmpty(width)) width = 200;
		if(EStr.isEmpty(height)) height = 200;
		renderQrCode(code, width, height);
	}
	
	/**
	 * 文件下载
	 */
	public void download(){
		String id = getClean(0);
		if(EStr.isEmpty(id)){
			renderErrMsg(AuthCode.LOGIC, "未发现文件ID");
			return;
		}
		FileStore fileStore = fileStoreSer.findById(id);
		if(null == fileStore){
			renderErrMsg(AuthCode.LOGIC, "文件不存在！");
			return;
		}
		File file = new File(fileStore.getFilePath());
		if(!file.exists()){
			renderErrMsg(AuthCode.LOGIC, "文件不存在，可能已被清理！");
			return;
		}
		fileStoreSer.downAdd(id);
		renderFile(file, EStr.ifEmpty(fileStore.getDisplayName(), fileStore.getFileName()));
	}
	
	/**
	 * 平台登录页
	 */
	@Clear
	public void login(){
		String policy = EStr.ifEmpty(session().getAttr(SESS_KEY_POLICY), JW.MainPolicyCode);
		redirect(attr().getSrc().concat("/account/signin/").concat(policy));
	}
	
	/**
	 * 注销身份
	 */
	public void logout(){
		String policy = EStr.ifEmpty(session().getAttr(SESS_KEY_POLICY), JW.MainPolicyCode);
		redirect(attr().getSrc().concat("/account/signout/").concat(policy));
	}
	
	/**
	 * 账号管理
	 */
	public void account(){
		String policy = EStr.ifEmpty(session().getAttr(SESS_KEY_POLICY), JW.MainPolicyCode);
		redirect(attr().getSrc().concat("/account/").concat(policy));
	}
	
	/**
	 * 错误页
	 */
	@Clear
	public void error(){
		AuthCode authCode = AuthCode.valueOf(getPara("authCode"));
		String errMsg = getPara("errMsg");
		String helpUrl = getPara("helpUrl");
		if(EStr.isEmpty(errMsg)) renderErr(authCode);
		else if(EStr.isEmpty(helpUrl)) renderErrMsg(authCode, errMsg);
		else renderErrHelp(authCode, errMsg, helpUrl);
		return;
	}
	
	/**
	 * <p>
	 * 登录初始化页面，实现租户和机构用户的自定义首页功能
	 * </p>
	 *
	 */
	public void initLogin(){
		String url = attr().getIndexUrl();
		String deptId = session().getUser().getDeptId();
		if(EStr.notEmpty(deptId)){
			Dept dept = deptSer.findById(deptId);
			if(dept != null && EStr.notEmpty(dept.getIndex())) url = attr().getSrc().concat(dept.getIndex());
		}
		redirect(url);
	}
	
	/**
	 * <p>
	 * 管理员工具箱
	 * </p>
	 *
	 */
	@AuthRequire.Role(value = ModuleConfig.RoleBaseAdmin)
	public void suptools(){
		render("sup/index.html");
	}
	
	/**
	 * 自定义编辑器上传
	 */
	@Clear
	public void ueditorUpload(){
		String action = getPara("action", "config");
		/**
		 * 百度编辑器加载出按钮图标前 会将所有控件的路径 先通过config.json
		 * 文件加载出来(包括上传图片路径，视频路径等路径都是通过config.json 文件读取的)
		 * 所以某些控件点击不了 是因为 config.json文件没有找到 或者是文件里面的路径有问题
		 */
		if("config".equals(action)){
			render("/main/asset/js/ueditorConfig.json");
		}else{
			UploadFile upfile = getFile("upfile");
			String moduleCode = getPara("moduleCode", "editor");
			if(upfile != null){
				String fileName = upfile.getFileName();
				String url = getUpload(upfile.getFile(), moduleCode);
				ERet ret = ERet.create();
				ret.set("state", "SUCCESS");
				ret.set("url", url);
				ret.set("title", fileName);
				ret.set("original", upfile.getOriginalFileName());
				ret.set("type", fileName.split("\\.")[1]);
				ret.set("size", upfile.getFile().length());
				renderJson(ret);
			}
		}
	}
	
	/**
	 * <p>
	 * 子域单点登录
	 * </p>
	 * 
	 * @param token
	 */
	@Clear
	public void ssoLogin(String token){
		try {
			User user = loginSer.ssoLogin(token);
			session().setLogined(true);
			session().setUid(user.getId());
			menuSer.sessMenuClear();
			redirect(user.getIndex());
			log.info("单点登录成功：[uid]{} [name]{}", user.getId(), user.getCaption());
		} catch (Exception e) {
			renderErrMsg(AuthCode.LOGIC, e.getMessage());
			log.info("单点登录验证失败：{} [token]{}", e.getMessage(), token);
		}
	}

}
