package itez.plat.main.controller;

import java.io.File;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.collect.Maps;

import itez.core.runtime.modules.ModuleManager;
import itez.core.wrapper.dbo.DbProp;
import itez.core.wrapper.dbo.dialect.EDialect.DbType;
import itez.kit.ECommand;
import itez.kit.EStr;
import itez.kit.ECommand.OS;
import itez.kit.log.ELogBase;
import itez.kit.EDate;
import itez.kit.ELog;
import itez.kit.restful.Result;

/**
 * <p>
 * 命令工具类
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月15日 下午11:00:29
 */
abstract class CommandKit {

	static final ELogBase log = ELog.log(CommandKit.class);
	static Map<ECommand.OS, String> dumpTemp = Maps.newHashMap();
	
	static {
		dumpTemp.put(OS.WIN,   "mysqldump -h%s -P%s -u%s -p%s --single-transaction --quick --lock-tables=false %s %s > %s/%s.sql");
		dumpTemp.put(OS.LINUX, "mysqldump -h%s -P%s -u%s -p%s --single-transaction --quick --lock-tables=false %s %s | gzip > %s/%s.sql.gz");
	}
	
	/**
	 * <p>
	 * 执行Shell命令
	 * </p>
	 * 
	 * @param isCmd
	 * @param cmd
	 * @param envs
	 * @param path
	 */
	static Result cmd(String cmd, String envs, String path){
		if(EStr.isEmpty(cmd)) return Result.fail("命令内容为空！");
		String[] envp = EStr.isEmpty(envs) ? null : envs.split(",");
		File dir = EStr.isEmpty(path) ? null : new File(path);
		ECommand ins = ECommand.getInstance();
		Result ret = ins.cmd(cmd, envp, dir);
		return ret;
	}
	
	/**
	 * <p>
	 * 重启Tomcat
	 * </p>
	 * 
	 * @param tomcatHome Tomcat安装目录
	 * @param mode 启动模式（cmd：命令；service：服务）
	 * @return
	 */
	static Result restartTomcat(String tomcatHome, String mode){
		if(EStr.isEmpty(tomcatHome)) return Result.fail("未发现Tomcat安装目录");
		tomcatHome += "/bin";
		if(EStr.isEmpty(mode)) mode = "cmd";
		String script = mode.equals("cmd") ? "restart" : "restartService";
		String ext = ECommand.getInstance().getOs() == OS.WIN ? ".bat" : ".sh";
		String shPath = EStr.join(tomcatHome, "/", script, ext);
		File shFile = new File(shPath);
		if(!shFile.exists()) return Result.fail("未发现Tomcat重启脚本：" + script + ext);
		if(ECommand.getInstance().getOs() == OS.LINUX) script = "./" + script;
		Result ret = cmd(script + ext, null, tomcatHome);
		return ret;
	}

	/**
	 * 备份数据库
	 * @param moduleCode 模块代码
	 * @param cmdPath 备份命令位置
	 * @param tabs 表格列表
	 * @param bakPath 备份位置
	 * @return
	 */
	static Result dump(String moduleCode, String cmdPath, String tabs, String bakPath){
		OS Os = ECommand.getInstance().getOs();
		String cmdTemp = dumpTemp.get(Os);
		if(cmdTemp == null) return Result.fail("未知的操作系统类型");
		if(EStr.isEmpty(bakPath)) return Result.fail("备份目录不能为空");
		bakPath = String.join("/", bakPath, moduleCode);
		bakPath = bakPath.replaceAll("\\\\", "/");
		File bakFolder = new File(bakPath);
		if(!bakFolder.exists()) bakFolder.mkdirs();
		
		DbProp dbProp = ModuleManager.me.getModuleDbProp(moduleCode);
		DbType dbType = dbProp.getType();
		if(dbType != DbType.mysql) return Result.fail("目前终端备份数据库功能仅支持MySQL");

		String url = dbProp.getJdbcUrl();
		
		Pattern pattern = Pattern.compile("^jdbc\\:mysql\\:\\/\\/([0-9a-zA-Z.-]+):(\\d+)/([^?]+)");
		Matcher matcher = pattern.matcher(url);
		matcher.find();
		int cnt = matcher.groupCount();
		if(cnt != 3) return Result.fail("解析jdbcUrl错误，未发现符合要求的host/port/dababase内容");
		
		String h = matcher.group(1);
		String P = matcher.group(2);
		String d = matcher.group(3);
		String u = dbProp.getUserName();
		String p = formatMetaChar(dbProp.getPassWord());
		
        //-h -P -u -p -d path file
		String uuid = EDate.format(EDate.getDate(), "yyyyMMddHHmmss");
        String bakName = String.join("_", d, uuid);
		String cmdStr = String.format(cmdTemp, h, P, u, p, d, tabs, bakPath, bakName);
		//log.info("备份数据库：{}", cmdStr);
		Result ret = cmd(cmdStr, null, cmdPath);
		String fileName = bakName + (Os == OS.LINUX ? ".sql.gz" : ".sql");
		return ret.set("bakFile", fileName);
	}
	
	/**
	 * <p>
	 * 对特殊字符（元字符）进行转义
	 * </p>
	 * 
	 * @param str
	 * @return
	 */
	static String formatMetaChar(String str){
		Pattern pattern = Pattern.compile("([\\`\\~\\!\\@\\#\\$\\%\\^\\&\\*\\(\\)\\_\\-\\+\\=])");
		Matcher matcher = pattern.matcher(str);
		return matcher.replaceAll("\\\\$1");
	}
	
}
