/**
 * TerminalCheckIn.java
 */
package itez.kit.terminal;

import itez.kit.ECode;
import itez.kit.EHttp;
import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.log.ELogBase;
import itez.kit.restful.EMap;
import itez.kit.restful.Result;
import itez.kit.rsa.RsaKit;

/**
 * <p>
 * 终端通信工具类（终端 --> 云端）
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月9日 下午4:37:38
 */
public abstract class TerminalUtil {

	private static ELogBase log = ELog.log(TerminalUtil.class);
	
	/**
	 * <p>
	 * 签到
	 * </p>
	 *
	 */
	public static void checkIn(){
		if(EStr.isEmpty(TerminalCommon.TOKEN)) throw new RuntimeException("本地Token未设置！");
		String url = EStr.join(TerminalCommon.SERVER_ROOT, "/mgr/api/term/checkin?domain=", TerminalCommon.DOMAIN, "&token=", TerminalCommon.TOKEN);
		String json = EHttp.me.get(url);
		if(EStr.isEmpty(json)){
			log.error("签到失败");
			throw new RuntimeException("签到失败");
		}
		Result ret = Result.parseJson(json);
		if(null == ret){
			log.error("签到结果为空");
			throw new RuntimeException("签到结果为空");
		}
		if(ret.getState()){
			TerminalCommon.KEYS[0] = ret.getData().getStr("key");
			log.info("签到成功");
		}else{
			log.error("签到失败，原因：{}", ret.getMsg());
			throw new RuntimeException(ret.getMsg());
		}
	}
	
	/**
	 * <p>
	 * 重置密码
	 * </p>
	 * 
	 * @param pass
	 */
	public static void chgpass(String pass){
		EMap params = EMap.by("domain", TerminalCommon.DOMAIN).set("pass", pass);
		
		//发送请求时，先生成签名，再加密
		genSign(params, new String[]{ "domain", "pass" });
		encrypt(params, new String[]{ "pass" });
		
		String url = "/mgr/api/term/chgpass";
		Result ret = getData(url, params);
		if(ret == null){
			log.error("重置密码未返回有效结果");
		}else if(!ret.getState()){
			log.error("重置密码失败：" + ret.getMsg());
		}else{
			log.info("重置密码成功！");
		}
	}
	
	/**
	 * <p>
	 * 向服务端发送请求
	 * </p>
	 * 
	 * @param url 请求资源路径，不含域名
	 * @param paras 含有签名（sign）的请求参数集合
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static String getDataStr(String url, EMap params){
		if(EStr.isEmpty(TerminalCommon.KEYS[0])){
			log.error("本地私钥不存在");
			throw new RuntimeException("本地私钥不存在");
		}
		try {
			if(!url.startsWith("/")) url = "/" + url;
			return EHttp.me.postForm(TerminalCommon.SERVER_ROOT + url, params);
		} catch (Exception e) {
			log.error("发送请求时发生错误：" + e.getMessage());
			throw new RuntimeException(e.getMessage());
		}
	}
	
	/**
	 * <p>
	 * 向服务端发送请求
	 * </p>
	 * 
	 * @param url 请求资源路径，不含域名
	 * @param paras 含有签名（sign）的请求参数集合
	 * @return
	 */
	public static Result getData(String url, EMap params){
		String json = getDataStr(url, params);
		if(EStr.isEmpty(json)) return null;
		return Result.parseJson(json);
	}
	
	/**
	 * <p>
	 * 生成签名
	 * 注意：如果某参数同时进行加密与签名，需要先进行 [签名] 再进行 [加密]。
	 * </p>
	 * 
	 * @param params 发送请求的参数集合，生成签名后会添加sign参数
	 * @param keys 参与签名的参数名称数组，注意顺序
	 * @return
	 */
	public static EMap genSign(EMap params, String[] keys){
		if(EStr.isEmpty(TerminalCommon.TOKEN)){
			log.error("本地TOKEN不存在");
			throw new RuntimeException("本地TOKEN不存在");
		}
		StringBuffer sb = new StringBuffer();
		for(String key : keys){
			sb.append(params.getStr(key));
		}
		try {
			String sign = ECode.md5(sb.toString() + TerminalCommon.TOKEN);
			params.set("sign", sign);
		} catch (Exception e) {
			log.error("生成签名异常：{}", e.getMessage());
			throw new RuntimeException(e.getMessage());
		}
		return params;
	}
	
	/**
	 * <p>
	 * 对请求参数进行加密处理
	 * 注意：如果某参数同时进行加密与签名，需要先进行 [签名] 再进行 [加密]。
	 * </p>
	 * 
	 * @param value
	 * @return
	 */
	public static EMap encrypt(EMap params, String[] keys){
		if(EStr.isEmpty(TerminalCommon.KEYS[0])){
			log.error("本地私钥不存在");
			throw new RuntimeException("本地私钥不存在");
		}
		try {
			for(String key : keys){
				String val = params.getStr(key);
				if(EStr.isEmpty(val)) continue;
				byte[] code = RsaKit.encryptByPublicKey(val.getBytes(EStr.UTF_8), TerminalCommon.KEYS[0]);
				params.set(key, RsaKit.encryptBASE64(code));
			}
		} catch (Exception e) {
			log.error("加密异常：{}", e.getMessage());
			throw new RuntimeException(e.getMessage());
		}
		return params;
	}
	
	/**
	 * <p>
	 * 对请求参数进行解密处理
	 * 注意：如果请求方同时对某参数进行了加密与签名，则校验时应先 [解密] 再进行 [签名校验]
	 * </p>
	 * 
	 * @param value
	 * @return
	 */
	public static EMap decrypt(EMap params, String[] keys){
		if(EStr.isEmpty(TerminalCommon.KEYS[0])){
			log.error("本地私钥不存在");
			throw new RuntimeException("本地私钥不存在");
		}
		try {
			for(String key : keys){
				String val = params.getStr(key);
				if(EStr.isEmpty(val)) continue;
				byte[] code = RsaKit.decryptByPublicKey(val, TerminalCommon.KEYS[0]);
				params.set(key, new String(code, EStr.UTF_8));
			}
		} catch (Exception e) {
			log.error("解密异常：{}", e.getMessage());
			throw new RuntimeException(e.getMessage());
		}
		return params;
	}

	/**
	 * <p>
	 * 签名校验
	 * 注意：如果请求方同时对某参数进行了加密与签名，则校验时应先 [解密] 再进行 [签名校验]
	 * </p>
	 * 
	 * @param params 返回的响应集合，其中包含sign签名字段
	 * @param keys 参与签名的字段名称数组，注意顺序
	 * @return
	 */
	public static boolean valiSign(EMap params, String[] keys){
		if(EStr.isEmpty(TerminalCommon.TOKEN)){
			log.error("本地TOKEN不存在");
			throw new RuntimeException("本地TOKEN不存在");
		}
		String signRes = params.getStr("sign");
		if(EStr.isEmpty(signRes)){
			log.error("未找到签名");
			throw new RuntimeException("未找到签名");
		}
		StringBuffer sb = new StringBuffer();
		for(String key : keys){
			sb.append(params.getStr(key));
		}
		try {
			String sign = ECode.md5(sb.toString() + TerminalCommon.TOKEN);
			if(signRes.equals(sign)) return true;
		} catch (Exception e) {
			log.error("签名校验异常：{}", e.getMessage());
			throw new RuntimeException(e.getMessage());
		}
		return false;
	}
	
}
