/**
 * RsaKit.java
 */
package itez.kit.rsa;

import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import javax.crypto.Cipher;

import org.apache.commons.codec.binary.Base64;

import itez.kit.EProp;
import itez.kit.EStr;

/**
 * <p>
 * RSA加密工具类
 * 
 * 使用1024初始化KeyPairGenerator，RSA加密后的密文的长度为1024位，即128个字节
 * 此时明文的最大长度不能超过117个字节
 * 超过117个字节需要使用2048的keysize来初始化KeyPairGenerator
 * 超过245个字节则需要使用更高位数的keysize。
 * RSA的keysize位数越高，其产生密钥对及加密、解密的速度越慢，这是基于大素数非对称加密算法的缺陷。 
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年3月16日 上午11:22:23
 */
public class RsaKit {
	
	public static final String KEY_ALGORITHM = "RSA";
	private static RsaKeys defKeys = null;
	
	/**
	 * <p>
	 * 获取默认公钥/私钥对
	 * 该公钥/私钥对可共用，减少生成时间，在对安全性要求不特别高的情况下使用
	 * </p>
	 * 
	 * @return
	 * @throws Exception
	 */
	public static RsaKeys getDefKeys() throws Exception{
		if(defKeys == null) defKeys = generatorKey();
		return defKeys;
	}
	
	/**
	 * <p>
	 * 使用默认私钥对数据进行解密
	 * </p>
	 * 
	 * @param data
	 * @return
	 */
	public static String decryptByDefPrivateKey(String data){
		if(EStr.isEmpty(data)) return null;
		try {
			byte[] byt = decryptByPrivateKey(data, getDefKeys().getPrivateKey());
			return new String(byt, EStr.UTF_8);
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
			return null;
		}
	}
	
	/**
	 * <p>
	 * 生成公钥/私钥对
	 * 注：该操作耗时较高
	 * </p>
	 * 
	 * @return
	 * @throws Exception
	 */
	public static RsaKeys generatorKey() throws Exception {
        return generatorKey(1024);
    }
	
	/**
	 * <p>
	 * 生成公钥/私钥对
	 * 注：该操作耗时较高
	 * </p>
	 * 
	 * @param size 私钥长度（默认1024位，即128个字节）
	 * @return
	 * @throws Exception
	 */
	public static RsaKeys generatorKey(int keysize) throws Exception {
        KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        keyPairGen.initialize(keysize);
        KeyPair keyPair = keyPairGen.generateKeyPair();
        return new RsaKeys(keyPair);
    }
	
	/**
	 * <p>
	 * 用公钥加密
	 * </p>
	 * 
	 * @param data 待加密数据（字节码）
	 * @param key 公钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
	public static byte[] encryptByPublicKey(byte[] data, String key) throws Exception {
        // 先对公钥进行Base64解码
        byte[] keyBytes = decryptBASE64(key);
        // 取得公钥对象
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key publicKey = keyFactory.generatePublic(x509KeySpec);
        // 对数据加密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, publicKey);
        return cipher.doFinal(data);
    }
	
	/**
	 * <p>
	 * 用公钥加密
	 * </p>
	 * 
	 * @param data 待加密数据（Base64编码字符串）
	 * @param key 公钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
	public static byte[] encryptByPublicKey(String data, String key) throws Exception {
        return encryptByPublicKey(decryptBASE64(data), key);
    }
	
	/**
	 * <p>
	 * 用公钥解密
	 * </p>
	 * 
	 * @param data 待解密数据（字节码）
	 * @param key 公钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
	public static byte[] decryptByPublicKey(byte[] data, String key) throws Exception {
        // 先对公钥进行Base64解码
        byte[] keyBytes = decryptBASE64(key);
        // 取得公钥对象
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key publicKey = keyFactory.generatePublic(x509KeySpec);
        // 对数据解密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, publicKey);
        return cipher.doFinal(data);
    }
	
	/**
	 * <p>
	 * 用公钥解密
	 * </p>
	 * 
	 * @param data 待解密数据（Base64编码字符串）
	 * @param key 公钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
	public static byte[] decryptByPublicKey(String data, String key) throws Exception {
        return decryptByPublicKey(decryptBASE64(data), key);
    }
    
	/**
	 * <p>
	 * 用私钥加密
	 * </p>
	 * 
	 * @param data 待加密数据（字节码）
	 * @param key 私钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
    public static byte[] encryptByPrivateKey(byte[] data, String key) throws Exception {
        // 先对私钥进行Base64解码
        byte[] keyBytes = decryptBASE64(key);
        // 取得私钥对象
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 对数据加密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, privateKey);
        return cipher.doFinal(data);
    }

    /**
     * <p>
     * 用私钥加密
     * </p>
     * 
     * @param data 待加密数据（Base64编码字符串）
     * @param key 私钥（Base64编码字符串）
     * @return
     * @throws Exception
     */
    public static byte[] encryptByPrivateKey(String data, String key) throws Exception {
        return encryptByPrivateKey(decryptBASE64(data), key);
    }
	
	/**
	 * <p>
	 * 用私钥解密
	 * </p>
	 * 
	 * @param data 待解密数据（字节码）
	 * @param key 私钥（Base64编码字符串）
	 * @return
	 * @throws Exception
	 */
    public static byte[] decryptByPrivateKey(byte[] data, String key) throws Exception{
        // 先对私钥进行Base64解码
        byte[] keyBytes = decryptBASE64(key);
        // 取得私钥对象
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
        // 对数据解密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, privateKey);
        return cipher.doFinal(data);
    }

    /**
     * <p>
     * 用私钥解密
     * </p>
     * 
     * @param data 待解密数据（Base64编码字符串）
     * @param key 私钥（Base64编码字符串）
     * @return
     * @throws Exception
     */
    public static byte[] decryptByPrivateKey(String data, String key) throws Exception {
        return decryptByPrivateKey(decryptBASE64(data), key);
    }

    /**
     * <p>
     * Base64编码
     * </p>
     * 
     * @param bytes 源数据（字节码）
     * @return
     */
    public static String encryptBASE64(byte[] bytes) {
        return Base64.encodeBase64String(bytes);
    }
    
    /**
     * <p>
     * Base64解码
     * </p>
     * 
     * @param key 源数据（Base64编码字符串）
     * @return
     */
    public static byte[] decryptBASE64(String key) {
        return Base64.decodeBase64(key);
    }
    
    public static void main(String[] args) throws Exception {
    	String sign = "UgFwBMna/n8X/waJrcumcm+GlHnd/6PYKsskPLQrTJqSAEsbEZRKnBl9OQTVb/cvl69bMjDAmRIv0nkuL2bQ5NI8/ij4AU2ZqZ31Z5sbyHGxlzh1NjuFuYvPtWaYTt3zHBhLTemRX/efMiTvwK8shkqvMvYcKFcYeLEFyH39D/E=";
    	String priKey = "MIICdQIBADANBgkqhkiG9w0BAQEFAASCAl8wggJbAgEAAoGBAKf8AZVNXi6lHkgco2WglIwc5uVXmRlPxgJmIthkpPqkvUaG2L7ZMq1Fl4/2bYhHBFZKR8xRNeR8ubPMk4HbWYg5YoZoPXs78l33aQby3zqXHnujoHIaxagK3C8/wk0e9gGDd4RP7qxwviKBir5eAk/XfPPI4P8zOGKA8yMKJaSVAgMBAAECgYBg1cnH2mCnfPG22iNrCyV2m5aSsGuGBtF7QotluKbOAbdGd6dsZOJstghHqeO8QkwCvjEjW9E4VJnsRsrleZ/jEDDrSyigycGOo1DnlgBB1hfoy642qhHz0nTbn/r99pjFeFFeLCr5BG+DXqSJlInjhcSFH51kmolDbkHE+HeNxQJBAPL9khFCesnSExhwNA9/y5BSgt2Per++/OZAeZ0MQoohOUyXcH96rLn3tj14Bpb1LrOIMLH3BIk1EipCB6QeyUsCQQCw+mbBmTpCF/7jxcydL3j1SeVGQKe4Gbeg/Q58pQwAGd2Sjx/+jQynYfPnsyvVeYkXKYlHx+LoNM7LI+3XF32fAkAKwJW6FcsBKz6VEkH7fZgwZrZLv5Ga8oSWW7wF6wj2u8mhTgI4YeFKyV9TEQCZpXfPkfM1rMySlbqcDBfydzpJAkAKKASXnaSDQH9HMQXCC4Rkhz4aHwHBBfDM01jufEZJBzL5Ay+x2wRb4rotlNGyKYxdgOu469awvr8TV1eB3X4/AkAMx6LYR8oRWVMizHQjm6sjqGCGWmOYx1Ujbt7MCBLvktW754qC1wNedZIgq7CscYlboUZdOv1l+7dbeKFpUxN8";
    	byte[] code = decryptByPrivateKey(sign, priKey);
    	System.out.println(new String(code, EStr.UTF_8));
	}
	
}
