/**
 * Result.java
 */
package itez.kit.restful;

import java.io.Serializable;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import itez.kit.EStr;

/**
 * <p>
 * RestFul返回值封装
 * </p>
 * 
 * <p>Copyright(C) 2017-2020 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2020年4月22日 上午10:12:47
 */
public class Result implements Serializable {

	private static final long serialVersionUID = 4165163611445221484L;
	
	protected boolean state;	//状态：true（成功），false（失败）
	protected int code;			//状态码
	protected String desc;		//状态描述
	protected String msg;		//详细信息
	protected EMap data;		//返回数据
	
	public Result(boolean state){
		this.state = state;
		this.code = state ? ResultCode.SUCCESS.getCode() : ResultCode.ERR_LOGIC.getCode();
        this.desc = state ? ResultCode.SUCCESS.getDesc() : ResultCode.ERR_LOGIC.getDesc();
        this.msg = this.desc;
		this.data = EMap.create();
	}
    
    /**
     * <p>
     * 根据JSON反序列化
     * </p>
     * 
     * @param json
     * @return
     */
    public static Result parseJson(String json){
    	if(EStr.isEmpty(json)) return null;
    	JSONObject obj = JSON.parseObject(json);
    	Boolean state = EStr.ifNull(obj.getBoolean("state"), false);
    	Integer code = obj.getIntValue("code");
    	String desc = obj.getString("desc");
    	String msg = obj.getString("msg");
    	JSONObject data = obj.getJSONObject("data");
    	Result ret = new Result(state);
    	if(null != code) ret.code = code;
    	if(null != desc) ret.desc = desc;
    	if(null != msg) ret.msg = msg;
    	if(null != data) ret.data = EMap.parseJsonObject(data);
    	return ret;
    }
    
    /**
     * <p>
     * 序列化为JSON字符串
     * </p>
     * 
     * @return
     */
    public String toJson(){
    	return JSON.toJSONString(this);
    }
    
    /**
     * <p>
     * 成功时
     * </p>
     * 
     * @param data 返回数据
     * @return
     */
    public static Result success(){
        return new Result(true);
    }
    
    /**
     * <p>
     * 成功时
     * </p>
     * 
     * @param data 返回数据
     * @return
     */
    public static Result success(Object key, Object value){
        return new Result(true).set(key, value);
    }
    
    /**
     * <p>
     * 失败时
     * </p>
     * 
     * @param resultCode 状态码
     * @return
     */
    public static Result fail(){
        return new Result(false);
    }
    
    /**
     * <p>
     * 失败时
     * </p>
     * 
     * @param resultCode 状态码
     * @return
     */
    public static Result fail(ResultCode resultCode){
        return new Result(false).setCode(resultCode.getCode());
    }
    
    /**
     * <p>
     * 失败时
     * </p>
     * 
     * @param resultCode 状态码
     * @param msg 详细描述
     * @return
     */
    public static Result fail(String msg){
        return new Result(false).setMsg(msg);
    }
    
    /**
     * <p>
     * 失败时
     * </p>
     * 
     * @param resultCode 状态码
     * @param msg 详细描述
     * @return
     */
    public static Result fail(ResultCode resultCode, String msg){
        return new Result(false).setCode(resultCode.getCode()).setMsg(msg);
    }
    
    /**
     * <p>
     * 添加数据
     * </p>
     * 
     * @param key
     * @param value
     * @return
     */
    public Result set(Object key, Object value){
    	this.data.set(key, value);
    	return this;
    }

	public boolean getState() {
		return state;
	}

	public Result setCode(int code) {
		this.code = code;
		return this;
	}

	public int getCode() {
		return code;
	}

	public Result setDesc(String desc) {
		this.desc = desc;
		return this;
	}

	public String getDesc() {
		return desc;
	}

	public Result setMsg(String msg) {
		this.msg = msg;
		return this;
	}

	public String getMsg() {
		return msg;
	}

	public EMap getData() {
		return data;
	}
	
	@SuppressWarnings("rawtypes")
	public Result setData(Map data){
		this.data.set(data);
		return this;
	}
	
}
