/**
 * 
 */
package itez.kit.restful;

import java.util.HashMap;
import java.util.Map;

import org.jsoup.safety.Whitelist;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import itez.core.util.inject.Injector;
import itez.kit.EClean;
import itez.kit.EJson;
import itez.kit.EStr;
import itez.kit.rsa.RsaKit;

/**
 * <p>
 * 键值对
 * </p>
 * 
 * <p>Copyright(C) 2017-2020 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2020年4月20日 下午4:35:03
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class EMap extends HashMap {

	private static final long serialVersionUID = -536264630368648754L;

    //*******************************************  创建实例   ***********************************************/

	/**
	 * 构造函数
	 */
	public EMap() {
	}
	
	/**
	 * <p>
	 * 创建新实例
	 * </p>
	 * 
	 * @return JMap
	 */
	public static EMap create(){
		return new EMap();
	}
	
	/**
	 * <p>
	 * 创建新实例，同时初始化一对数据
	 * </p>
	 * 
	 * @param key 键名
	 * @param value 值
	 * @return JMap
	 */
	public static EMap by(Object key, Object value){
		return new EMap().set(key, value);
	}

    //*******************************************  添加数据   ***********************************************/

	/**
	 * <p>
	 * 添加数据
	 * </p>
	 * 
	 * @param key 键名
	 * @param value 值
	 * @return JMap
	 */
	public EMap set(Object key, Object value) {
		super.put(key, value);
		return this;
	}
	
	/**
	 * <p>
	 * 添加数据
	 * </p>
	 * 
	 * @param map Map
	 * @return JMap
	 */
	public EMap set(Map map) {
		if(null != map) super.putAll(map);
		return this;
	}
	
	/**
	 * <p>
	 * 添加数据
	 * </p>
	 * 
	 * @param kv JMap
	 * @return JMap
	 */
	public EMap set(EMap kv) {
		super.putAll(kv);
		return this;
	}

	/**
	 * <p>
	 * 添加非Null数据
	 * </p>
	 * 
	 * @param key 键名
	 * @param value 值
	 * @return JMap
	 */
	public EMap setIfNotNull(Object key, Object value) {
		if (EStr.notNull(value)) {
			set(key, value);
		}
		return this;
	}

	/**
	 * <p>
	 * 添加非空数据
	 * </p>
	 * 
	 * @param key 键名
	 * @param value 值
	 * @return JMap
	 */
	public EMap setIfNotEmpty(Object key, String value) {
		if (EStr.notEmpty(value)) {
			set(key, value);
		}
		return this;
	}

    //*******************************************  返回数据   ***********************************************/

	/**
	 * <p>
	 * 返回指定泛型键值
	 * </p>
	 * 
	 * @param key 键名
	 * @return Object
	 */
	public <T> T getAs(Object key) {
		return (T)get(key);
	}
	
	/**
	 * <p>
	 * 返回字符串
	 * </p>
	 * 
	 * @param key 键名
	 * @return String
	 */
	public String getStr(Object key) {
		Object s = get(key);
		return s != null ? s.toString() : null;
	}
	
	/**
	 * <p>
	 * 返回整型
	 * </p>
	 * 
	 * @param key 键名
	 * @return Integer
	 */
	public Integer getInt(Object key) {
		Number n = (Number)get(key);
		return n != null ? n.intValue() : null;
	}
	
	/**
	 * <p>
	 * 返回长整型
	 * </p>
	 * 
	 * @param key 键名
	 * @return Long
	 */
	public Long getLong(Object key) {
		Number n = (Number)get(key);
		return n != null ? n.longValue() : null;
	}
	
	/**
	 * <p>
	 * 返回数值型
	 * </p>
	 * 
	 * @param key 键名
	 * @return Number
	 */
	public Number getNumber(Object key) {
		return (Number)get(key);
	}
	
	/**
	 * <p>
	 * 返回布尔型
	 * </p>
	 * 
	 * @param key 键名
	 * @return Boolean
	 */
	public Boolean getBoolean(Object key) {
		return (Boolean)get(key);
	}

    //*******************************************  删除数据   ***********************************************/

	/**
	 * <p>
	 * 删除键值
	 * </p>
	 * 
	 * @param key 键名
	 * @return JMap
	 */
	public EMap delete(Object...keys) {
		for(Object key : keys) super.remove(key);
		return this;
	}
	
	//*******************************************  根据JSONObject初始化   ***********************************************/
	
	public static EMap parseJsonObject(JSONObject obj){
    	if(null == obj) return null;
    	EMap map = EMap.create();
    	for(Object key : obj.keySet()){
    		Object keyvalue = obj.get(key);
    		map.set(key, keyvalue);
    	}
    	return map;
	}
	
	//*******************************************  反序列化JSON   ***********************************************/
	
	public static EMap parseJson(String json){
    	if(EStr.isEmpty(json)) return null;
    	EMap map = EMap.create();
    	JSONObject obj = JSON.parseObject(json);
    	for(Object key : obj.keySet()){
    		Object keyvalue = obj.get(key);
    		map.set(key, keyvalue);
    	}
    	return map;
	}
	
	//*******************************************  生成JSON   ***********************************************/
	
	public String toJson(){
		return EJson.toJson(this);
	}
	
	//*******************************************  XSS过滤   ***********************************************/

	public EMap cleanAll(){
		return this.cleanAll(null);
	}
	public EMap cleanAll(Whitelist whitelist){
		if(whitelist == null) whitelist = EClean.text;
		for(Object key: keySet()){
			String val = String.valueOf(get(key));
			val = EClean.clean(val, whitelist);
			set(key, val);
		}
		return this;
	}
	public EMap clean(String...keys){
		return this.clean(null, keys);
	}
	public EMap clean(Whitelist whitelist, String...keys){
		if(whitelist == null) whitelist = EClean.text;
		for(String key: keys){
			Object o = get(key);
			if(null != o){
				String val = String.valueOf(o);
				val = EClean.clean(val, whitelist);
				set(key, val);
			}
		}
		return this;
	}
	
	//*******************************************  解密   ***********************************************/

	public EMap decryptAll(){
		for(Object key: keySet()){
			String val = String.valueOf(get(key));
			val = RsaKit.decryptByDefPrivateKey(val);
			set(key, val);
		}
		return this;
	}
	public EMap decrypt(String...keys){
		for(String key: keys){
			Object o = get(key);
			if(null != o){
				String val = String.valueOf(o);
				val = RsaKit.decryptByDefPrivateKey(val);
				set(key, val);
			}
		}
		return this;
	}
	
	//*******************************************  生成请求参数串   ***********************************************/
	
	public String toRequestParams(){
		StringBuffer sb = new StringBuffer("");
		for(Object key: keySet()){
			sb.append("&");
			sb.append(key);
			sb.append("=");
			sb.append(get(key));
		}
		String params = sb.toString();
		return EStr.isEmpty(params) ? "" : params.substring(1);
	}
	
	//*******************************************  转换为对象   ***********************************************/

	public <T> T getModel(Class<T> beanClass){
		return getModel(beanClass, false);
	}
	public <T> T getModel(Class<T> beanClass, boolean skipConvertError){
		return (T)Injector.injectBean(beanClass, "", this, skipConvertError);
	}
	
}
