/**
 * Copyright(C) 2022 上游科技 . All Rights Reserved
 */
package itez.kit.queue;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import itez.kit.ELog;
import itez.kit.log.ELogBase;

/**
 * <p>
 * 线程工厂
 * </p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2022年3月30日 上午10:51:51
 */
public class QueFactory {

	private final ELogBase log = ELog.log(getClass());
	
	public static QueFactory me = new QueFactory();
	
	// 等待队列
	private BlockingQueue<Runnable> queue;
	// 消费者线程池
	private ThreadPoolExecutor threadPool;

	// 核心线程数（添加新任务时，如果超过核心线程数，则尝试添加到等待队列）
	private int corePoolSize;
	// 最大线程数（添加新任务时，如果等待队列已满，并且还未超过最大线程数，则尝试创建新线程；否则抛出异常）
	private int maximumPoolSize;
	// 线程的空闲存活时长（秒），当设置allowCoreThreadTimeOut(true)时，核心线程和非核心线程都有效；否则仅对非核心线程有效；当allowCoreThreadTimeOut(false)且核心线程数和最大数相同时，该参数无效
	private long keepAliveTime;
	// 队列最大长度（如果队列长度设成Integer.MAX_VALUE不限制，则maximumPoolSize参数无效）
	private int queueCapacity;
	
	// CPU核心数
	private int cpuCoreCnt;
	
	/**
	 * 线程工厂构造方法
	 */
	private QueFactory(){
		cpuCoreCnt = Runtime.getRuntime().availableProcessors();	
		log.info("获取CPU核心数：" + cpuCoreCnt);
		
		corePoolSize = cpuCoreCnt;		// 核心线程数
		maximumPoolSize = cpuCoreCnt;	// 最大线程数
		keepAliveTime = 30;				// 非核心线程的空闲存活时长（分钟）
		queueCapacity = 9999;			// 队列最大长度
		queue = new LinkedBlockingQueue<Runnable>(queueCapacity);
		
		createThreadPool(corePoolSize, maximumPoolSize, keepAliveTime);
	}
	
	/**
	 * <p>
	 * 创建新的线程池
	 * </p>
	 * 
	 * @param corePoolSize
	 * @param maximumPoolSize
	 * @param keepAliveTime
	 */
	private void createThreadPool(int corePoolSize, int maximumPoolSize, long keepAliveTime){
		log.info("开始初始化线程池！");
		threadPool = new ThreadPoolExecutor(corePoolSize, maximumPoolSize, keepAliveTime, TimeUnit.MINUTES, queue);
		threadPool.allowCoreThreadTimeOut(true);	//允许回收核心线程
		log.info("线程池初始化完毕：corePoolSize（{}），maximumPoolSize（{}），queueCapacity（{}），keepAliveTime（{}分钟）", corePoolSize, maximumPoolSize, queueCapacity, keepAliveTime);
	}
	
	/**
	 * <p>
	 * 添加任务
	 * 注：如果线程池已关闭，会自动创建新的线程池
	 * </p>
	 * 
	 * @param task
	 */
	public void addTask(QueTask task){
		log.debug("添加新任务，等待队列长度：{}", queue.size());
		if(threadPool.isShutdown()) createThreadPool(corePoolSize, maximumPoolSize, keepAliveTime);
		threadPool.execute(task);
	}
	
	/**
	 * <p>
	 * 手动关闭线程池
	 * 注：线程池空闲一段时间后会自动关闭，无需手动关闭
	 * </p>
	 *
	 */
	public void shutdown(){
		threadPool.shutdown();
		try {
			threadPool.awaitTermination(30, TimeUnit.SECONDS);
			log.info("线程池已关闭");
		}catch(Exception e){
			threadPool.shutdownNow();
			log.info("线程池已强制关闭");
		}
	}
	
}
