package itez.kit.poi;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;
import javax.servlet.ServletOutputStream;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;

import com.beust.jcommander.internal.Lists;
import com.google.common.collect.Maps;
import com.jfinal.plugin.activerecord.Record;
import com.jfinal.render.Render;

import itez.kit.EStr;
import itez.kit.El;
import itez.kit.restful.EMap;

public class XlsWriter extends Render {

	private String fileName;
	private Excel excel;
	private Workbook workbook;
	private int headerRows = 0;
	
	public XlsWriter(String fileName) {
		this.fileName = fileName;
		excel = new Excel();
	}
	
	public XlsWriter(String fileName, Excel excel) {
		this.fileName = fileName;
		this.excel = excel;
	}
	
	public static XlsWriter create(String fileName){
		return new XlsWriter(fileName);
	}
	
	public XlsWriter putSheet(String caption, List<Record> data, XlsHeader header){
		return putSheet(caption, data, header, true);
	}
	
	public XlsWriter putSheet(String caption, List<Record> data, XlsHeader header, boolean hasSortColumn){
		ESheet sheet = new ESheet();
		headerRows = initHeader(sheet, header, hasSortColumn);
		int rowIndex = 0;
		int ind = headerRows;
		int count = data.size();
		for(; rowIndex < count; rowIndex++){
			addRow(sheet, data.get(rowIndex), header, ind, 0, hasSortColumn);
			ind += 1;
		}
		sheet.setCaption(caption);
		sheet.setIndex(excel.getSheetCount());
		sheet.setCellCount(header.getItems().size() + (hasSortColumn ? 1 : 0));
		sheet.setRowCount(ind);
		excel.putSheet(sheet);
		return this;
	}
	
	private int initHeader(ESheet sheet, XlsHeader header, boolean hasSortColumn){
		int headerRows = 0;
		if(header.getShowCode()){
			addRow(sheet, null, header, headerRows, 1, hasSortColumn);
			headerRows += 1;
		}
		if(header.getShowCaption()){
			addRow(sheet, null, header, headerRows, 2, hasSortColumn);
			headerRows += 1;
		}
		return headerRows;
	}
	
	private void addRow(ESheet sheet, Record rec, XlsHeader header, int rowIndex, int isHeader, boolean hasSortColumn){
		List<ECell> cells = Lists.newArrayList();
		int ind= 0;
		if(hasSortColumn){
			Map<String, Object> values = Maps.newHashMap();
			String caption = isHeader == 1 ? "ON" : isHeader == 2 ? "序号" : rowIndex - headerRows + 1 + "";
			CellType type = isHeader > 0 ? CellType.STRING : CellType.NUMERIC;
			values.put(type.name(), caption);
			ECell cell = new ECell();
			cell.setRowIndex(rowIndex).setCellIndex(ind++).setType(type.name()).setValue(values);
			cells.add(cell);
		}
		for(XlsHeaderItem item : header.getItems()){
			ECell cell = new ECell();
			Map<String, Object> values = Maps.newHashMap();
			CellType type = isHeader > 0 ? CellType.STRING : item.getCellType();
			Object defValue = item.getDefValue();
			Object value = null;
			if(isHeader == 1){
				value = item.getCode();
			}else if(isHeader == 2){
				value = item.getCaption();
			}else{
				if(type == CellType.NUMERIC) value = rec.getDouble(item.getCode());
				else value = rec.getStr(item.getCode());
				if(value == null && defValue != null) value = defValue;
			}
			values.put(type.name(), value);
			cell.setRowIndex(rowIndex).setCellIndex(ind++).setType(type.name()).setValue(values);
			cells.add(cell);
		}
		ERow row = new ERow().setIndex(rowIndex).setHeader(isHeader > 0).setCells(cells);
		sheet.putRow(row);
	}
	
	public XlsWriter generator(){
		workbook = new HSSFWorkbook();
		Font fontTitle = workbook.createFont();
		fontTitle.setColor(HSSFColor.HSSFColorPredefined.DARK_BLUE.getIndex());
		fontTitle.setBold(true);
		CellStyle styleTitle = workbook.createCellStyle();
		styleTitle.setFont(fontTitle);
		styleTitle.setFillForegroundColor(HSSFColor.HSSFColorPredefined.LIGHT_CORNFLOWER_BLUE.getIndex());
		styleTitle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
		styleTitle.setAlignment(HorizontalAlignment.CENTER);
		styleTitle.setVerticalAlignment(VerticalAlignment.CENTER);
		styleTitle.setBorderBottom(BorderStyle.THIN);
		styleTitle.setBorderRight(BorderStyle.THIN);
		CellStyle styleData = workbook.createCellStyle();
		styleData.setVerticalAlignment(VerticalAlignment.CENTER);
		styleData.setBorderBottom(BorderStyle.THIN);
		styleData.setBorderRight(BorderStyle.THIN);
		List<ESheet> sheets = excel.getSheets();
		sheets.forEach(sheet -> {
			Sheet xsheet = workbook.createSheet(sheet.getCaption());
			xsheet.setDisplayGridlines(false);
			Map<Integer,Integer> maxWidth = Maps.newHashMap();
			int defWidth = 10  * 256 + 100;
			for(int i=0; i<sheet.getCellCount(); i++) maxWidth.put(i, defWidth);
			List<ERow> rows = sheet.getRows();
			EMap paras = EMap.by("rowIndex", 0);
			rows.forEach(row -> {
				paras.set("rowIndex", paras.getInt("rowIndex") + 1);
				Row xrow = xsheet.createRow(row.getIndex());
				xrow.setHeightInPoints((short)20);
				List<ECell> cells = row.getCells();
				cells.forEach(cell -> {
					Object value = cell.getValue();
					CellType type = CellType.valueOf(cell.getType());
					Cell xcell = xrow.createCell(cell.getCellIndex(), type);
					xcell.setCellStyle(row.isHeader() ? styleTitle : styleData);
					if(value != null){
						String valueStr = value.toString();
						if(type == CellType.FORMULA){
							String valueEx = El.exec(valueStr, paras);
							xcell.setCellFormula(valueEx);
						}else{
							if(type == CellType.NUMERIC) xcell.setCellValue(Double.parseDouble(valueStr));
							else xcell.setCellValue(valueStr);
							int len = valueStr.getBytes().length  * 256 + 100;
							if(len > 15000){ len = 15000; }
							maxWidth.put(cell.getCellIndex(), Math.max(len, maxWidth.get(cell.getCellIndex())));
						}
					}else{
						xcell.setCellValue("");
					}
				});
				for(int i=0; i<sheet.getCellCount(); i++) xsheet.setColumnWidth(i, maxWidth.get(i));
			});
		});
		return this;
	}

	@Override
	public void render() {
		if(workbook == null) error("请先调用generator()方法生成工作表。");
		response.setCharacterEncoding("UTF-8");
        response.setHeader("Content-Disposition", getFilename());
        response.setHeader("Pragma","no-cache");
        response.setHeader("Cache-Control","no-cache");
        response.setDateHeader("Expires", 0);
        response.setContentType("application/vnd.ms-excel");
        ServletOutputStream outs = null;
        try {
        	outs = response.getOutputStream();
        	workbook.write(outs);
        } catch (IOException e) {
            if (getDevMode()) error(e.getMessage());
        } catch (Exception e) {
        	error(e.getMessage());
        } finally {
            if (outs != null) {
                try {outs.close();} catch (IOException e) {error(e.getMessage());}
            }
        }
	}
	
	public String export(String distPath){
		if(workbook == null) error("请先调用generator()方法生成工作表。");
		if(EStr.isEmpty(distPath)) distPath = "/";
		if(!distPath.endsWith("/") && !distPath.endsWith("\\")) distPath = distPath.concat("/");
		String filePath = distPath.concat(fileName);
        File file = new File(filePath);
        if (!file.getParentFile().exists()) file.getParentFile().mkdirs();
        try (OutputStream outs = new FileOutputStream(file)) {
            workbook.write(outs);
        } catch (FileNotFoundException e) {
			error("表格输出目录不存在！");
		} catch (IOException e) {
			error("表格输出失败！");
		}
        return file.getPath();
	}
 
    protected String getFilename() {
        try {
            String agent = request.getHeader("USER-AGENT");
            if (agent.toLowerCase().indexOf("firefox") > 0) {
            	fileName = new String(fileName.getBytes("utf-8"), "iso-8859-1");
            } else {
            	fileName = URLEncoder.encode(fileName, "UTF-8");
            }
        } catch (UnsupportedEncodingException e) {
            error(e.getMessage());
        }
        return "attachment; filename=".concat(fileName);
    }
    
    protected void error(String msg) {
		throw new RuntimeException(msg);
	}
	
	public static void main(String[] args) {
		
		//初始化数据库连接池，仅JAVA调试下需要，WEB环境不需要该行代码
//		DbManager.me.initDBPlugins(null);
//		List<Record> data = Db.findAll("base_comp");
//		
//		XlsWriter xls = XlsWriter.create("test.xls");
//		List<XlsHeaderItem> items = Lists.newArrayList();
//		items.add(XlsHeaderItem.create("domain", "子域"));
//		items.add(XlsHeaderItem.create("caption", "名称"));
//		items.add(XlsHeaderItem.create("cdate", "创建时间"));
//		items.add(XlsHeaderItem.create("used", "可用").setCellType(CellType.NUMERIC));
//		XlsHeader header = XlsHeader.create(items);
//		xls.putSheet("公司信息", data, header);
//		xls.generator().export("D:/exportPic/excel");
		
//		File file = new File("C:\\Users\\Lenovo\\Downloads\\字典（外语语种）.xls");
//		try {
//			Excel excel = EPoi.load(file);
//			excel.getSheet(0).getRow(0).setHeader(true);
//			excel.getSheet(0).getRow(1).setHeader(true);
//			XlsWriter xls = new XlsWriter("测试.xls", excel);
//			xls.generator();
//			xls.export("D:/exportPic/excel");
//		} catch (Exception e) {
//			e.printStackTrace();
//		}
		
	}
	
}
