package itez.kit.pay;

import java.util.Map;

import com.alibaba.fastjson.JSONObject;
import com.beust.jcommander.internal.Maps;

import itez.kit.EPara;
import itez.kit.pay.alipay.PayAli;
import itez.kit.pay.ccb.PayCCB;
import itez.kit.pay.wechat.PayWechat;

/**
 * 支付工具类
 * @author netwild
 *
 */
public class PayKit {

	/**
	 * 支持的支付渠道
	 * 
	 *   1、CCB：建行聚合支付
	 * 
	 * @author netwild
	 */
	public static enum Channel {
		CCB("建行聚合支付", "https://common.itez.com.cn/pic/payment/jsyh.jpg"),
		WX("微信支付", "https://common.itez.com.cn/pic/payment/wx.jpg"),
		ALI("支付宝", "https://common.itez.com.cn/pic/payment/zfb.jpg");
		private String caption = "";
		private String pic = "";
		private Channel(String caption, String pic){
			this.caption = caption;
			this.pic = pic;
		}
		public String getCaption(){
			return this.caption;
		}
		public String getPic(){
			return this.pic;
		}
		public String getDesc(){
			String desc = "请使用手机扫码支付";
			if(this == Channel.CCB){
				desc = "请使用微信、支付宝或建行龙支付App";
			}else if(this == Channel.WX){
				desc = "请使用微信App扫码支付";
			}else if(this == Channel.ALI){
				desc = "请使用支付宝扫码支付";
			}
			return desc;
		}
	}
	
	/**
	 * 支付渠道实现类缓存
	 */
	private Map<Channel, PayBase> pays = Maps.newHashMap();
	
	/**
	 * 单列模式
	 */
	public static final PayKit me = new PayKit();
	private PayKit(){}
	
	/**
	 * 初始化支付渠道
	 * @param channel
	 * @return
	 */
	private PayBase init(Channel channel){
		PayBase pay = pays.get(channel);
		if(pay == null){
			if(channel == Channel.CCB) pay = new PayCCB();
			else if(channel == Channel.WX) pay = new PayWechat();
			else if(channel == Channel.ALI) pay = new PayAli();
			else throw new RuntimeException("无法识别的支付渠道：" + channel.name());
		}
		return pay;
	}
	
	/**
	 * 根据参数动态获取二维码
	 * 
	 * @param params 保存的渠道配置信息，各渠道不相同
	 * @param orderId 订单号
	 * @param payment 支付金额
	 * @param remark1 备注1
	 * @param remark2 备注2
	 * @return
	 */
	public String getQrCode(JSONObject configs, String orderId, String payment, String remark1, String remark2){
		String channelCode = configs.getString("code");
		Channel channel = Channel.valueOf(channelCode);
		PayBase pay = init(channel);
		return pay.getQrCode(configs, orderId, payment, remark1, remark2);
	}

	/**
	 * 支付完成的回调处理
	 * 
	 * @param configs 保存的渠道配置信息，各渠道不相同
	 * @param paras request请求参数封装
	 * @return
	 */
	public PayOver callback(JSONObject configs, EPara paras){
		String channelCode = configs.getString("code");
		Channel channel = Channel.valueOf(channelCode);
		PayBase pay = init(channel);
		return pay.callback(configs, paras);
	}
	
	/**
	 * 返回支付渠道的参数配置项
	 * @param channel
	 * @return
	 */
	public String paramConfig(Channel channel){
		PayBase pay = init(channel);
		return pay.paramsConfig();
	}
	
}
