package itez.kit.fileup;

import java.awt.image.BufferedImage;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.imageio.ImageIO;

import org.apache.commons.io.FileUtils;

import com.google.common.collect.Lists;
import com.jfinal.kit.PathKit;

import itez.kit.EDate;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.EUid;

public class LocalImpl implements IFileUp {
	
	private String Url; //虚拟目录域名
	private String DiskPath; //虚拟目录所在硬盘位置
	private String TempPath; //临时目录硬盘位置
	
	private static final String FileSep = File.separator;
	
	public LocalImpl(){
		TempPath = formatPath(EProp.FileUploadTemp);
		DiskPath = formatPath(EProp.LocalFilePath);
		Url = EProp.LocalFileUrl;
    	if(!DiskPath.startsWith("/") && !DiskPath.startsWith("\\") && !ERegex.has(DiskPath, "^[A-Za-z]\\:")){
    		DiskPath = PathKit.getWebRootPath() + FileSep + DiskPath;
    	}
		if(!DiskPath.endsWith(FileSep)){
			DiskPath += FileSep;
		}
    	if(!TempPath.startsWith("/") && !TempPath.startsWith("\\") && !ERegex.has(TempPath, "^[A-Za-z]\\:")){
    		TempPath = PathKit.getWebRootPath() + FileSep + TempPath;
    	}
		if(!TempPath.endsWith(FileSep)){
			TempPath += FileSep;
		}
	}
	
	@Override
	public String upload(File file, String subPath) {
		return upload(file, subPath, true);
	}
	
	@Override
	public String upload(BufferedImage file, String formatName, String subPath) {
		return upload(file, formatName, subPath, true);
	}
	
	@Override
	public String upload(BufferedImage file, String formatName, String subPath, boolean uuidName) {
		File f = new File(TempPath.concat("thu.jpg"));
		if(EStr.isEmpty(formatName)) formatName = "jpg";
		try {
			ImageIO.write(file, formatName, f);
			return upload(f, subPath, uuidName);
		} catch (Exception e) {
			return null;
		}
	}
	
	@Override
	public String upload(File file, String subPath, boolean uuidName) {
		String fileIndex = EDate.format(EDate.getDate(), "yyyy/MM/dd/");
		String fileName = file.getName();
		if(EStr.notEmpty(subPath) && !subPath.endsWith(FileSep)) subPath += FileSep;
		String key;
		if(uuidName){
			String extName = fileName.substring(fileName.lastIndexOf("."));
			key = EStr.join(subPath, fileIndex, EUid.generator(), extName);
		}else{
			key = EStr.join(subPath, fileName);
		}
	    try {
	    	String dest = formatPath(DiskPath + key);
	    	String toPathStr = dest.substring(0, dest.lastIndexOf(FileSep));
	    	File toPath = new File(toPathStr);
	    	if(!toPath.exists()) toPath.mkdirs();
	    	Files.move(file.toPath(), new File(dest).toPath(), StandardCopyOption.REPLACE_EXISTING);
	        return formatUrl(EStr.join(Url, "/", key));
	    }catch (Exception e) {
	    	System.err.println(e.toString());
	    }
	    return null;
	}
	
	@Override
	public boolean createFolder(String subPath){
		if(subPath.startsWith(FileSep)) subPath = subPath.substring(1);
    	String dest = formatPath(DiskPath + subPath);
    	File toPath = new File(dest);
    	if(toPath.exists()){
    		throw new RuntimeException("目录已存在");
    	}else{
    		return toPath.mkdirs();
    	}
	}
	
	private static String formatPath(String path){
		return path.replace("/", FileSep).replace("\\", FileSep);
	}
	
	private String formatUrl(String url){
		if(EStr.isEmpty(url)) return url;
		return url.replace(FileSep, "/");
	}

	@Override
	public List<FileItem> list(String subPath) {
		return list(subPath, false);
	}

	@Override
	public List<FileItem> list(String subPath, boolean directory) {
		return localList(DiskPath, Url, subPath, directory);
	}

	@Override
	public boolean remove(String fileName) {
		File file = new File(fileName);
		if(!file.exists()) return false;
		if(file.isDirectory()){
			try {
				FileUtils.deleteDirectory(file);
				return true;
			} catch (Exception e) {
				return false;
			}
		}else
			return file.delete();
	}

	@Override
	public File getFile(String filePath){
		if(filePath.startsWith(FileSep)) filePath = filePath.substring(1);
    	filePath = formatPath(DiskPath + filePath);
		return localFile(filePath);
	}

	@Override
	public String getFileStoreUrl() {
		return Url;
	}
	
	public String getUrl() {
		return Url;
	}

	public String getDiskPath() {
		return DiskPath;
	}

	public String getTempPath() {
		return TempPath;
	}
	
	/**
	 * <p>
	 * 返回本地指定目录的子目录及文件列表
	 * </p>
	 * 
	 * @param root 根目录
	 * @param Url 根地址
	 * @param sub 子目录
	 * @param directory 是否包含子目录
	 * @return
	 */
	public static List<FileItem> localList(String root, String Url, String sub, boolean directory){
		List<FileItem> temps = Lists.newArrayList();
    	String path = formatPath(root + sub);
		File dir = new File(path);
		if(!dir.exists()) return temps;
		if(dir.isFile()) return temps;
		File[] list = dir.listFiles();
		if(list == null || list.length == 0) return temps;
		for(File file : list){
			if(file.isDirectory() && !directory) continue;
			FileItem temp = new FileItem(file);
			if(EStr.notEmpty(Url)){
				String url = String.format("%s/%s/%s", Url, sub, file.getName());
				temp.setUrl(url);
			}
			temps.add(temp);
		}
		temps = temps.stream().sorted(Comparator.comparing(FileItem::isDirectory).reversed()).collect(Collectors.toList());
		return temps;
	}

	/**
	 * <p>
	 * 返回本地文件对象
	 * </p>
	 * 
	 * @param filePath
	 * @return
	 */
	public static File localFile(String filePath){
		File file = new File(filePath);
		if(!file.exists()) throw new RuntimeException("文件不存在");
		if(file.isDirectory()) throw new RuntimeException("不能返回文件夹");
		return file;
	}

	public static void main(String[] args) {
		LocalImpl fileio = new LocalImpl();
		File file = new File("D:\\WxAppRoot\\jyt\\resources\\topHeader.jpg");
		String path = fileio.upload(file, "");
		System.out.println(path);
	}

}