package itez.kit.fileup;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;

import com.jfinal.kit.PathKit;

import itez.core.runtime.EContext;
import itez.kit.EDate;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.ERegex;
import itez.kit.EStr;
import itez.kit.EUid;
import itez.kit.log.ELogBase;
import itez.kit.log.ExceptionUtil;

/**
 * 文件上传工具类（过渡期）
 * @author netwild
 *
 */
@SuppressWarnings("unused")
public class EFileUpKit {

	public final static EFileUpKit me = new EFileUpKit();
	private final static ELogBase log = ELog.log(EFileUpKit.class);
	
	private String webRoot;
	private String localTempPath;
	private String localStorePath;
	private String localStoreUrl;
	
	private EFileUpKit(){
		webRoot = PathKit.getWebRootPath();
		localTempPath = formatPath(appendRootPath(EProp.FileUploadTemp));
		localStorePath = formatPath(appendRootPath(EProp.LocalFilePath));
		localStoreUrl = formatUrl(EProp.LocalFileUrl);
	}
	
	public FileItem upload(File file){
		return upload(file, EUid.generator());
	}
	
	public FileItem upload(File file, String fileName){
		String extName = getExtName(file);
		fileName = formatExtName(fileName, extName);
		String domain = EContext.getAttr().getDomain();
		String moduleCode = EContext.getAttr().getModuleCode();
		if(EStr.isEmpty(domain)) throw new RuntimeException("未发现子域（domain）");
		if(EStr.isEmpty(moduleCode)) throw new RuntimeException("未发现模块（moduleCode）");
		return upload(file, fileName, domain, moduleCode);
	}
	
	public FileItem upload(File file, String fileName, String domain, String moduleCode){
		String folder = EDate.format(EDate.getDate(), "yyyy/MM/dd");
		return upload(file, fileName, domain, moduleCode, folder);
	}
	
	public FileItem upload(File file, String fileName, String domain, String moduleCode, String folder){
		String relPath = String.join(EStr.FileSep, domain, moduleCode, folder);
		String destUrl = formatUrl(String.join("/", localStoreUrl, relPath, fileName));
		String destFolderPath = formatPath(String.join(EStr.FileSep, localStorePath, relPath));
		String destFilePath = formatPath(String.join(EStr.FileSep, destFolderPath, fileName));
		FileItem item = null;
		try {
			File destFolder = new File(destFolderPath);
			if(!destFolder.exists()) destFolder.mkdirs();
			File destFile = new File(destFilePath);
			Files.move(file.toPath(), destFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
			item = new FileItem(destFile);
			item.setUrl(destUrl);
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
			else log.error(ExceptionUtil.getMessage(e));
		}
		return item;
	}
	
	private String appendRootPath(String path){
    	if(!path.matches("(\\\\|\\/|[A-Za-z]\\:).+")){
    		path = String.join(EStr.FileSep, webRoot, path);
    	}
    	return path;
	}
	
	private String formatPath(String path){
		if(EStr.isEmpty(path)) return path;
		String sep = EStr.FileSep.equals("/") ? EStr.FileSep : "\\\\";
		return path.replaceAll("\\\\|\\/", sep);
	}
	
	private String formatUrl(String url){
		if(EStr.isEmpty(url)) return url;
		return url.replaceAll("\\\\|\\/", "/");
	}
	
	private String formatExtName(String fileName, String extName){
		if(EStr.isEmpty(extName)) return fileName;
		if(fileName.indexOf(".") > -1) return fileName;
		return String.join(".", fileName, extName);
	}
	
	public String getExtName(File file){
		String fileName = file.getName();
		return getExtName(fileName);
	}
	
	public String getExtName(String fileName){
		if(EStr.isEmpty(fileName)) return "";
		int at = fileName.lastIndexOf(".");
		if(at == -1) return "";
		return fileName.substring(at + 1);
	}

	public String getWebRoot() {
		return webRoot;
	}

	public String getLocalTempPath() {
		return localTempPath;
	}

	public String getLocalStorePath() {
		return localStorePath;
	}

	public String getLocalStoreUrl() {
		return localStoreUrl;
	}
	
}
