package itez.kit;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.servlet.ServletInputStream;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import itez.kit.restful.EMap;

/**
 * WEB工具类
 * @author netwild
 */
public abstract class EWeb {

	private static Logger log = LoggerFactory.getLogger(EWeb.class);

	/**
	 * 获取客户端IP地址
	 * @param request
	 * @return
	 */
	public static String getIpAddr(HttpServletRequest request) {
		String ip = request.getHeader("x-forwarded-for");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("Proxy-Client-IP");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("WL-Proxy-Client-IP");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("HTTP_CLIENT_IP");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("HTTP_X_FORWARDED_FOR");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("HTTP_X_FORWARDED");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("HTTP_FORWARDED_FOR");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getHeader("HTTP_FORWARDED");
		if (EStr.isEmpty(ip) || "unknown".equalsIgnoreCase(ip)) ip = request.getRemoteAddr();
		
		if (EStr.notEmpty(ip)){
			int at = ip.indexOf(",");
			if(at > 0) ip = ip.substring(0, at);
		}
		
		return ip;
	}

	/**
	 * 获取客户端端口号
	 * @param request
	 * @return
	 */
	public static int getRemotePort(HttpServletRequest request) {
		return request.getRemotePort();
	}
	
	/**
	 * <p>
	 * 获取当前请求的协议（http/https）
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static String getScheme(HttpServletRequest request){
		return EStr.findUseful(request.getHeader("X-Forwarded-Scheme"), request.getScheme(), "http");
	}
	
	/**
	 * <p>
	 * 获取当前请求的协议名称及版本（如：HTTP/1.11）
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static String getProtocol(HttpServletRequest request){
		return request.getProtocol();
	}
	
	/**
	 * <p>
	 * 获取当前请求的协议名称及版本（如：HTTP/1.11）
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static String getMethod(HttpServletRequest request){
		return request.getMethod();
	}
	
	/**
	 * <p>
	 * 获取当前请求的数据长度（字节）
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static long getReqLen(HttpServletRequest request){
		long len = request.getContentLengthLong();
		return len == -1 ? 0 : len;
	}
	
	/**
	 * <p>
	 * 获取当前请求头
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static EMap getReqHeader(HttpServletRequest request){
		StringBuffer req = new StringBuffer();
		req.append(getMethod(request));
		req.append(" ");
		req.append(getFullUrl(request, false));
		req.append(" ");
		req.append(getProtocol(request));
		EMap map = EMap.by("Request", req.toString());

		Enumeration<?> enums = request.getHeaderNames();
		while (enums.hasMoreElements()) {
			String key = (String) enums.nextElement();
			String value = request.getHeader(key);
			map.set(key, value);
		}
		return map;
	}
	
	/**
	 * <p>
	 * 获取当前请求体
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static EMap getReqBody(HttpServletRequest request){
		if(getMethod(request).equals("GET")) return null; //GET方式无需返回请求体

		String contentType = request.getContentType();
		if(contentType !=null && contentType.toLowerCase().contains("multipart")) return null; //上传文件方式无法返回请求体
		
		EMap map = EMap.create();
		Enumeration<?> enums = request.getParameterNames();
		while (enums.hasMoreElements()) {
			String key = (String) enums.nextElement();
			String[] values = request.getParameterValues(key);
			String value = Arrays.stream(values).collect(Collectors.joining(","));
			map.set(key, value);
		}
		return map;
	}
	
	/**
	 * 获取当前请求体类型
	 * @param request
	 * @return
	 */
	public static String getReqContentType(HttpServletRequest request){
		String type1 = request.getHeader("Content-Type");
		String type2 = request.getContentType();
		return EStr.findUseful(type1, type2);
	}
	
	/**
	 * <p>
	 * 获取当前响应头
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static EMap getResHeader(HttpServletResponse response){
		EMap map = EMap.create();
		Collection<String> enums = response.getHeaderNames();
		enums.forEach(enu -> {
			String key = (String) enu;
			String value = response.getHeader(key);
			map.set(key, value);
		});
		return map;
	}
	
	/**
	 * <p>
	 * 获取当前响应体
	 * </p>
	 * 
	 * @param request
	 * @return
	 */
	public static String getResBody(HttpServletResponse response){
		return null;
	}

	/**
	 * 获取应用根路径（格式：http://www.domain.com:8888/app）
	 * 
	 * @param request
	 * @return
	 */
	public static String getContextPath(HttpServletRequest request) {
		String scheme, serverName, serverPort, context;
		scheme = getScheme(request);
		serverName = request.getServerName();
		int port = request.getServerPort();
		serverPort = (port == 80 || port == 443) ? "" : ":" + port;
		context = request.getContextPath();
		return EStr.join(scheme, "://", serverName, serverPort, context);
	}

	/**
	 * 获取主机头路径（格式：www.domain.com:8888/app）
	 * 
	 * @param request
	 * @return
	 */
	public static String getHostPath(HttpServletRequest request) {
		String serverName, serverPort, context;
		serverName = request.getServerName();
		int port = request.getServerPort();
		serverPort = (port == 80 || port == 443) ? "" : ":" + port;
		context = request.getContextPath();
		return EStr.join(serverName, serverPort, context);
	}
	
	/**
	 * 获取主域名
	 * 
	 * @param request
	 * @return
	 */
	public static String getDomain(HttpServletRequest request){
		String serverName = request.getServerName();
		if(serverName.indexOf(".") == -1) return serverName; //localhost
		if(ERegex.checkIP(serverName)) return serverName; //IP
		return ERegex.find(serverName, "([^.]*)(.cn|.com|.com.cn|.net|.net.cn|.org|.org.cn|.edu|.edu.cn)$");
	}

	/**
	 * 获取完整请求路径(含内容路径及请求参数)
	 * 
	 * @param request
	 * @return
	 */
	public static String getFullUrl(HttpServletRequest request) {
		return getFullUrl(request, true);
	}

	/**
	 * 获取完整请求路径(含内容路径及请求参数)
	 * 
	 * @param request
	 * @return
	 */
	public static String getFullUrl(HttpServletRequest request, boolean escapeHtml) {
		String resUrl = request.getRequestURL().toString();
		String params = request.getQueryString();
		if(escapeHtml) params = StringEscapeUtils.escapeHtml4(params);
		if(EStr.notEmpty(params)) resUrl += "?" + params;
		return resUrl;
	}
	
	/**
	 * 获取请求来源路径
	 * 
	 * @param request
	 * @return
	 */
	public static String getReferer(HttpServletRequest request){
		return EClean.filterUrl(request.getHeader("referer"));
	}
	
	/**
	 * 地址栏参数编码
	 * 
	 * @param url
	 * @return
	 */
	public static String UrlEncoder(String url){
		String urlEncode = url;
		try {
			urlEncode = URLEncoder.encode(url, EStr.Encoding);
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
		}
		return urlEncode;
	}
	
	/**
	 * 地址栏参数解码
	 * 
	 * @param url
	 * @return
	 */
	public static String UrlDecoder(String url){
		String urlDecode = url;
		try {
			urlDecode = URLDecoder.decode(url, EStr.Encoding);
		} catch (Exception e) {
			if(EProp.DevMode) e.printStackTrace();
		}
		return urlDecode;
	}

	/**
	 * 获取请求参数
	 * 
	 * @param request
	 * @param name
	 * @return
	 */
	public static String getParam(HttpServletRequest request, String name) {
		String value = request.getParameter(name);
		if (null != value && !value.isEmpty()) {
			try {
				return URLDecoder.decode(value, EStr.Encoding).trim();
			} catch (UnsupportedEncodingException e) {
				log.error("decode异常：" + value);
				return value;
			}
		}
		return value;
	}

	/**
	 * 获取ParameterMap
	 * @param request
	 * @return
	 */
	public static Map<String, String> getParamMap(HttpServletRequest request){
		Map<String, String> map = new HashMap<String, String>();
		Enumeration<String> enume = request.getParameterNames();
		while (enume.hasMoreElements()) {
			String name = (String) enume.nextElement();
			map.put(name, getParam(request, name));
		}
		return map;
	}
	
	/**
	 * 判断是否为Ajax请求
	 * @param request
	 * @return
	 */
	public static boolean isAjax(HttpServletRequest request){
        String header = request.getHeader("X-Requested-With");
        return "XMLHttpRequest".equalsIgnoreCase(header);
	}

	/**
	 * 输出servlet文本内容
	 * 
	 * @param response
	 * @param content
	 * @param contentType
	 */
	public static void outPage(HttpServletResponse response, String content) {
		outPage(response, content, EStr.Encoding);
	}

	/**
	 * 输出servlet文本内容
	 * 
	 * @param response
	 * @param content
	 * @param contentType
	 */
	public static void outPage(HttpServletResponse response, String content, String contentType) {
		if(EStr.isEmpty(contentType)) contentType = EStr.Encoding;
		try {
			response.setHeader("Pragma", "no-cache");
			response.setHeader("Cache-Control", "no-cache");
			response.setDateHeader("Expires", 0);
			response.setContentType(String.format("text/html; charset=%s", contentType));
			response.setCharacterEncoding(contentType);
			PrintWriter writer = response.getWriter();
			writer.write(content);
			writer.flush();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * 请求流转字符串
	 * 
	 * @param request
	 * @return
	 */
	public static String requestStream(HttpServletRequest request) {
		InputStream inputStream = null;
		InputStreamReader inputStreamReader = null;
		BufferedReader bufferedReader = null;
		try {
			request.setCharacterEncoding(EStr.Encoding);
			inputStream = (ServletInputStream) request.getInputStream();
			inputStreamReader = new InputStreamReader(inputStream, EStr.Encoding);
			bufferedReader = new BufferedReader(inputStreamReader);
			String line = null;
			StringBuilder sb = new StringBuilder();
			while ((line = bufferedReader.readLine()) != null) {
				sb.append(line);
			}
			return sb.toString();
		} catch (IOException e) {
			log.error("request.getInputStream() to String 异常", e);
			return null;
		} finally { // 释放资源
			if(null != bufferedReader){
				try {
					bufferedReader.close();
				} catch (IOException e) {
					log.error("bufferedReader.close()异常", e);
				}
				bufferedReader = null;
			}
			
			if(null != inputStreamReader){
				try {
					inputStreamReader.close();
				} catch (IOException e) {
					log.error("inputStreamReader.close()异常", e);
				}
				inputStreamReader = null;
			}
			
			if(null != inputStream){
				try {
					inputStream.close();
				} catch (IOException e) {
					log.error("inputStream.close()异常", e);
				}
				inputStream = null;
			}
		}
	}
	
	/**
	 * 
	 * @param response
	 * @param domain		设置cookie所在域
	 * @param path			设置cookie所在路径
	 * @param isHttpOnly	是否只读
	 * @param name			cookie的名称
	 * @param value			cookie的值
	 * @param maxAge		cookie存放的时间(以秒为单位,假如存放三天,即3*24*60*60; 如果值为0,cookie将随浏览器关闭而清除)
	 */
	public static void addCookie(HttpServletRequest request, HttpServletResponse response, 
			String domain, String path, boolean isHttpOnly, 
			String name, String value, int maxAge) {
		Cookie cookie = new Cookie(name, value);

		// 所在域：比如a1.4bu4.com 和 a2.4bu4.com 共享cookie
		if(null != domain && !domain.isEmpty()){
			cookie.setDomain(domain);
		}
		
		// 设置cookie所在路径
		cookie.setPath("/");
		if(null != path && !path.isEmpty()){
			cookie.setPath(path);				
		}
		
		// 是否只读
		try {
			cookie.setHttpOnly(isHttpOnly);
		} catch (Exception e) {
			log.error("servlet容器版本太低，servlet3.0以前不支持设置cookie只读" + e.getMessage());
		}
		
		// 设置cookie的过期时间
		if (maxAge > 0){
			cookie.setMaxAge(maxAge);
		}
		
		// 添加cookie
		response.addCookie(cookie);
		/*
		System.out.println("cookie.getDomain：" + cookie.getDomain());
		System.out.println("cookie.getPath：" + cookie.getPath());
		System.out.println("cookie.getMaxAge：" + cookie.getMaxAge());
		System.out.println("cookie.getName：" + cookie.getName());
		System.out.println("cookie.getValue：" + cookie.getValue());
		Cookie[] cookies = request.getCookies();
		System.out.println("cookies：" + cookies.length);
		*/
	}

	/**
	 * 获取cookie的值
	 * 
	 * @param request
	 * @param name cookie的名称
	 * @return
	 */
	public static String getCookieValueByName(HttpServletRequest request, String name) {
		Cookie cookie = getCookieByName(request, name);
		if(cookie != null) return cookie.getValue();
		return null;
	}
	
	/**
	 * 清除cookie
	 * @param request
	 * @param response
	 * @param name cookie的名称
	 */
	public static void clearCookieByName(HttpServletRequest request, HttpServletResponse response, String name){
		Cookie cookie = getCookieByName(request, name);
		if(cookie != null){
			cookie.setValue(null);
			cookie.setPath("/");
			cookie.setMaxAge(0);
			response.addCookie(cookie);
		}
	}

	/**
	 * 获得cookie
	 * 
	 * @param request
	 * @param name cookie的名称
	 * @return
	 */
	public static Cookie getCookieByName(HttpServletRequest request, String name) {
		Map<String, Cookie> cookieMap = EWeb.readCookieMap(request);
		// 判断cookie集合中是否有我们像要的cookie对象 如果有返回它的值
		if (cookieMap.containsKey(name)) {
			Cookie cookie = (Cookie) cookieMap.get(name);
			return cookie;
		} else {
			return null;
		}
	}

	/**
	 * 获得所有cookie
	 * 
	 * @param request
	 * @return
	 */
	public static Map<String, Cookie> readCookieMap(HttpServletRequest request) {
		Map<String, Cookie> cookieMap = new HashMap<String, Cookie>();
		// 从request范围中得到cookie数组 然后遍历放入map集合中
		Cookie[] cookies = request.getCookies();
		//System.out.println("cookies：" + cookies.length);
		if (null != cookies) {
			for (int i = 0; i < cookies.length; i++) {
				cookieMap.put(cookies[i].getName(), cookies[i]);
			}
		}
		return cookieMap;
	}

	/**
	 * 去除HTML代码
	 * 
	 * @param inputString
	 * @return
	 */
	public static String HtmltoText(String inputString) {
		String htmlStr = inputString; // 含HTML标签的字符串
		String textStr = "";
		java.util.regex.Pattern p_script;
		java.util.regex.Matcher m_script;
		java.util.regex.Pattern p_style;
		java.util.regex.Matcher m_style;
		java.util.regex.Pattern p_html;
		java.util.regex.Matcher m_html;
		java.util.regex.Pattern p_ba;
		java.util.regex.Matcher m_ba;

		try {
			String regEx_script = "<[\\s]*?script[^>]*?>[\\s\\S]*?<[\\s]*?\\/[\\s]*?script[\\s]*?>"; // 定义script的正则表达式{或<script[^>]*?>[\\s\\S]*?<\\/script>
																										// }
			String regEx_style = "<[\\s]*?style[^>]*?>[\\s\\S]*?<[\\s]*?\\/[\\s]*?style[\\s]*?>"; // 定义style的正则表达式{或<style[^>]*?>[\\s\\S]*?<\\/style>
																									// }
			String regEx_html = "<[^>]+>"; // 定义HTML标签的正则表达式
			String patternStr = "\\s+";

			p_script = Pattern.compile(regEx_script, Pattern.CASE_INSENSITIVE);
			m_script = p_script.matcher(htmlStr);
			htmlStr = m_script.replaceAll(""); // 过滤script标签

			p_style = Pattern.compile(regEx_style, Pattern.CASE_INSENSITIVE);
			m_style = p_style.matcher(htmlStr);
			htmlStr = m_style.replaceAll(""); // 过滤style标签

			p_html = Pattern.compile(regEx_html, Pattern.CASE_INSENSITIVE);
			m_html = p_html.matcher(htmlStr);
			htmlStr = m_html.replaceAll(""); // 过滤html标签

			p_ba = Pattern.compile(patternStr, Pattern.CASE_INSENSITIVE);
			m_ba = p_ba.matcher(htmlStr);
			htmlStr = m_ba.replaceAll(""); // 过滤空格

			textStr = htmlStr;

		} catch (Exception e) {
			System.err.println("Html2Text: " + e.getMessage());
		}
		return textStr;// 返回文本字符串
	}

	/**
	 * 把页面的信息替换成我们想要的信息存放数据库里面
	 * 
	 * @param sourcestr
	 *            页面得到的信息
	 * @return
	 */
	public static String getHTMLToString(String sourcestr) {
		if (sourcestr == null) {
			return "";
		}
		sourcestr = sourcestr.replaceAll("\\x26", "&amp;");// &
		sourcestr = sourcestr.replaceAll("\\x3c", "&lt;");// <
		sourcestr = sourcestr.replaceAll("\\x3e", "&gt;");// >
		sourcestr = sourcestr.replaceAll("\\x09", "&nbsp;&nbsp;&nbsp;&nbsp;");// tab键
		sourcestr = sourcestr.replaceAll("\\x20", "&nbsp;");// 空格
		sourcestr = sourcestr.replaceAll("\\x22", "&quot;");// "

		sourcestr = sourcestr.replaceAll("\r\n", "<br>");// 回车换行
		sourcestr = sourcestr.replaceAll("\r", "<br>");// 回车
		sourcestr = sourcestr.replaceAll("\n", "<br>");// 换行
		return sourcestr;
	}

	/**
	 * 把数据库里面的信息回显到页面上
	 * 
	 * @param sourcestr
	 *            数据库取得的信息
	 * @return
	 */
	public static String getStringToHTML(String sourcestr) {
		if (sourcestr == null) {
			return "";
		}
		sourcestr = sourcestr.replaceAll("&amp;", "\\x26");// &
		sourcestr = sourcestr.replaceAll("&lt;", "\\x3c");// <
		sourcestr = sourcestr.replaceAll("&gt;", "\\x3e");// >
		sourcestr = sourcestr.replaceAll("&nbsp;&nbsp;&nbsp;&nbsp;", "\\x09");// tab键
		sourcestr = sourcestr.replaceAll("&nbsp;", "\\x20");// 空格
		sourcestr = sourcestr.replaceAll("&quot;", "\\x22");// "

		sourcestr = sourcestr.replaceAll("<br>", "\r\n");// 回车换行
		sourcestr = sourcestr.replaceAll("<br>", "\r");// 回车
		sourcestr = sourcestr.replaceAll("<br>", "\n");// 换行
		return sourcestr;
	}

}
