package itez.kit;

import okhttp3.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * HTTP操作工具类
 * 
 * @author netwild
 *
 */
public class EHttp {

	private final static Logger log = LoggerFactory.getLogger(EHttp.class);
	public final static EHttp me = new EHttp();
	
	public static enum Method { GET, POST, PUT, DELETE, PATCH }
	private final static Charset CharSet = Charset.forName("UTF-8");
	private static final MediaType ContentType = MediaType.parse("application/octet-stream");
	private final HashMap<String, List<Cookie>> cookieStore = new HashMap<>();
	
	private OkHttpClient client;
	
	private EHttp(){
		Dispatcher dispatcher = new Dispatcher();
        dispatcher.setMaxRequests(64); //底层HTTP库所有的并发执行的请求数量
        dispatcher.setMaxRequestsPerHost(16); //底层HTTP库对每个独立的Host进行并发请求的数量
        //连接池：底层HTTP库中复用连接对象的最大空闲数量（默认32），底层HTTP库中复用连接对象的回收周期（默认5分钟）
		ConnectionPool connectionPool = new ConnectionPool(32, 5, TimeUnit.MINUTES);
		client = new OkHttpClient.Builder()
				.dispatcher(dispatcher)
				.connectionPool(connectionPool) //连接池
				.connectTimeout(10, TimeUnit.SECONDS) //连接超时，默认10秒
				.readTimeout(30, TimeUnit.SECONDS) //读取超时，默认30秒
				.writeTimeout(0,TimeUnit.SECONDS) //写入超时，0为不超时
				.cookieJar(new CookieJar(){ //自动携带Cookie，并回写到本地
					@Override
					public void saveFromResponse(HttpUrl url, List<Cookie> cookies) {
						cookieStore.put(url.host(), cookies);
					}
					@Override
					public List<Cookie> loadForRequest(HttpUrl url) {
						List<Cookie> cookies = cookieStore.get(url.host());
						return cookies != null ? cookies : new ArrayList<Cookie>();
					}
				})
				.build();
	}

	/**
	 * Get请求
	 * @category Get请求
	 * @param url 请求地址
	 * @return 页面body
	 */
	public String get(String url){
		Request request = httpBuilder(Method.GET, url, null, null);
		return body(request);
	}

	/**
	 * Get请求
	 * @category Get请求
	 * @param url 请求地址
	 * @return 页面字节数组
	 */
	public byte[] getByte(String url){
		Request request = httpBuilder(Method.GET, url, null, null);
		return bytes(request);
	}

	/**
	 * Get请求
	 * @category Get请求
	 * @param url 请求地址
	 * @param params 参数集合
	 * @return 页面body
	 */
	public String get(String url, Map<String, String> params){
		url = formatGetParams(url, params);
		Request request = httpBuilder(Method.GET, url, null, null);
		return body(request);
	}
	
	/**
	 * Get请求
	 * @category Get请求
	 * @param url 请求地址
	 * @param params 参数集合
	 * @param headers 页头集合
	 * @return 页面body
	 */
	public String get(String url, Map<String, String> params, Map<String, String> headers){
		url = formatGetParams(url, params);
		Request request = httpBuilder(Method.GET, url, headers, null);
		return body(request);
	}
	
	/**
	 * 将get请求中的参数列表转换为字符串附加到url
	 * @param url 请求地址
	 * @param params 参数集合
	 * @return 字符串
	 */
	private String formatGetParams(String url, Map<String, String> params){
		String paramsStr = "";
		if(params != null && !params.isEmpty()) {
			paramsStr = params.entrySet().stream()
					.map(e -> String.format("%s=%s", e.getKey(), e.getValue()))
					.collect(Collectors.joining("&"));
		}
		if(EStr.notEmpty(paramsStr)){
			url = String.format("%s%s%s", url, url.indexOf("?") > 0 ? '&' : '?', paramsStr);
		}
		return url;
	}
	
	/**
	 * Post方式提交Json请求
	 * @category Post Json请求
	 * @param url 请求地址
	 * @param json 请求数据
	 * @return 返回内容
	 */
	public String postJson(String url, String json){
		MediaType mediaType = MediaType.parse(String.format("application/json; charset=%s", CharSet));
		RequestBody data = RequestBody.create(mediaType, json);
		Request request = httpBuilder(Method.POST, url, null, data);
		return body(request);
	}

	/**
	 * Post方式提交Json请求
	 * @category Post Json请求
	 * @param url 请求地址
	 * @param json 请求数据
	 * @param headers 页头集合
	 * @return 返回内容
	 */
	public String postJson(String url, String json, Map<String, String> headers){
		MediaType mediaType = MediaType.parse(String.format("application/json; charset=%s", CharSet));
		RequestBody data = RequestBody.create(mediaType, json);
		Request request = httpBuilder(Method.POST, url, headers, data);
		return body(request);
	}
	
	/**
	 * Post方式提交xml请求
	 * @category Post xml请求
	 * @param url 请求地址
	 * @param xml 请求数据
	 * @return 返回内容
	 */
	public String postXml(String url, String xml){
		MediaType mediaType = MediaType.parse(String.format("application/x-www-form-urlencoded; charset=%s", CharSet));
		RequestBody data = RequestBody.create(mediaType, xml);
		Request request = httpBuilder(Method.POST, url, null, data);
		return body(request);
	}

	/**
	 * Post方式提交xml请求
	 * @category Post xml请求
	 * @param url 请求地址
	 * @param xml 请求数据
	 * @param headers 页头集合
	 * @return 返回内容
	 */
	public String postXml(String url, String xml, Map<String, String> headers){
		MediaType mediaType = MediaType.parse(String.format("application/x-www-form-urlencoded; charset=%s", CharSet));
		RequestBody data = RequestBody.create(mediaType, xml);
		Request request = httpBuilder(Method.POST, url, headers, data);
		return body(request);
	}
	
	/**
	 * Post Form请求
	 * @category Post Form请求
	 * @param url 请求地址
	 * @param params 参数集合
	 * @return 返回内容
	 */
	public String postForm(String url, Map<String, String> params){
		FormBody.Builder builder = getFormBody(params);
		Request request = httpBuilder(Method.POST, url, null, builder.build());
		return body(request);
	}
	
	/**
	 * Post Form请求
	 * @category Post Form请求
	 * @param url 请求地址
	 * @param params 参数集合
	 * @param headers 页头集合
	 * @return 返回内容
	 */
	public String postForm(String url, Map<String, String> params, Map<String, String> headers){
		FormBody.Builder builder = getFormBody(params);
		Request request = httpBuilder(Method.POST, url, headers, builder.build());
		return body(request);
	}
	
	/**
	 * Post Form请求
	 * @category Post Form请求
	 * @param url 请求地址
	 * @param params 参数集合
	 * @param headers 页头集合
	 * @return 响应对象
	 */
	public Response postFormResponse(String url, Map<String, String> params, Map<String, String> headers){
		FormBody.Builder builder = getFormBody(params);
		Request request = httpBuilder(Method.POST, url, headers, builder.build());
		Response response = null;
		try {
			response = execute(request);
		} catch (Exception e) {
			log.error(e.getMessage());
		}
		return response;
	}
	
	/**
	 * Post File 请求
	 * @category Post File 请求
	 * @param url 请求地址
	 * @param files 对象集合
	 * @return 返回内容
	 */
	public String postFile(String url, Map<String, File> files){
		MultipartBody.Builder builder = getMultiBody(files, null);
		Request request = httpBuilder(Method.POST, url, null, builder.build());
		return body(request);
	}
	
	/**
	 * Post File 请求
	 * @category Post File 请求
	 * @param url 请求地址
	 * @param files 对象集合
	 * @param params 参数集合
	 * @return 返回内容
	 */
	public String postFile(String url, Map<String, File> files, Map<String, String> params){
		MultipartBody.Builder builder = getMultiBody(files, params);
		Request request = httpBuilder(Method.POST, url, null, builder.build());
		return body(request);
	}
	
	/**
	 * Post File 请求
	 * @category Post File 请求
	 * @param url 请求地址
	 * @param files 对象集合
	 * @param params 参数集合
	 * @param headers 页头集合
	 * @return 返回内容
	 */
	public String postFile(String url, Map<String, File> files, Map<String, String> params, Map<String, String> headers){
		MultipartBody.Builder builder = getMultiBody(files, params);
		Request request = httpBuilder(Method.POST, url, headers, builder.build());
		return body(request);
	}
	
	/**
	 * @category 生成FormBody.Builder
	 * @param params
	 * @return
	 */
	private FormBody.Builder getFormBody(Map<String, String> params){
		FormBody.Builder builder = new FormBody.Builder(CharSet);
		if(params != null && !params.isEmpty()) {
			params.forEach((k, v) -> builder.add(k, v));
		}
		return builder;
	}
	
	/**
	 * @category 生成MultipartBody.Builder
	 * @param files
	 * @param params
	 * @return
	 */
	private MultipartBody.Builder getMultiBody(Map<String, File> files, Map<String, String> params){
		MultipartBody.Builder builder = new MultipartBody.Builder();
		builder.setType(MultipartBody.FORM);
		if(files != null && !files.isEmpty()) {
			files.forEach((k, v) -> builder.addFormDataPart(k, v.getName(), getFileBody(v)));
		}
		if(params != null && !params.isEmpty()) {
			params.forEach((k, v) -> builder.addFormDataPart(k, v));
		}
		return builder;
	}
	
	/**
	 * @category 生成文件数据体
	 * @param file
	 * @return
	 */
	private RequestBody getFileBody(File file){
		RequestBody fileBody = RequestBody.create(ContentType, file);
		return fileBody;
	}
	
	/**
	 * @category 生成请求主体
	 * @param method 请求类型
	 * @param url 请求地址
	 * @param headers 请求头
	 * @param params 参数
	 * @param data 数据
	 * @return
	 */
	private Request httpBuilder(Method method, String url, Map<String, String> headers, RequestBody data) {
		Request.Builder builder = new Request.Builder();
		builder.url(url);		
		builder.header("User-Agent", "JWinner EHttp Kit");
		builder.addHeader("Accept-Charset", CharSet.toString());
		builder.addHeader("Accept-Language", "en-us,zh-ch");
		builder.addHeader("Content-Type", String.format("text/html; charset=%s;", CharSet));
		if(headers != null && !headers.isEmpty() && headers.size() > 0) {
			headers.forEach((k, v) ->  builder.addHeader(k, v));
		}
		if(method == Method.GET){
			builder.get();
		}else{
			builder.method(method.name(), data);
		}
		Request req = builder.build();
		return req;
	}
	
	/**
	 * @category 执行请求
	 * @param request
	 * @return
	 */
	private Response execute(Request request) throws Exception{
		Response response = client.newCall(request).execute();
		if(!response.isSuccessful()) throw new Exception(response.body().string());
		return response;
	}
	
	/**
	 * @category 执行请求
	 * @param request
	 * @return
	 */
	private String body(Request request){
		String result = "";
		try {
			Response response = execute(request);
			result = response.body().string();
		} catch (Exception e) {
			log.error("请求错误：{}", e.getMessage());
			if(EProp.DevMode) e.printStackTrace();
		}
		return result;
	}
	
	/**
	 * @category 执行请求
	 * @param request
	 * @return
	 */
	private byte[] bytes(Request request){
		InputStream result = null;
		ByteArrayOutputStream bot = null;
		try {
			Response response = execute(request);
			result = response.body().byteStream();
			bot = new ByteArrayOutputStream();  
	        byte[] buff = new byte[1024];  
	        int rc = 0;  
	        while ((rc = result.read(buff, 0, 100)) > 0) {  
	        	bot.write(buff, 0, rc);  
	        }  
	        return bot.toByteArray();  
		} catch (Exception e) {
			log.error("请求错误：{}", e.getMessage());
			return null;
		} finally {
			if(result != null){
				try {
					result.close();
				} catch (Exception e2) {}
			}
			if(bot != null){
				try {
					bot.close();
				} catch (Exception e2) {}
			}
		}
	}
	
}