package itez.kit;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * 日期处理工具类
 * 
 * @category 日期处理工具类
 * @author netwild
 *
 */
public class EDate {

	public static final String PATTERN_Y = "yyyy"; // 年
	public static final String PATTERN_YM = "yyyy-MM"; // 年-月
	public static final String PATTERN_MD = "MM-dd"; // 月-日
	public static final String PATTERN_YMD = "yyyy-MM-dd"; // 年-月-日
	public static final String PATTERN_YMD_HM = "yyyy-MM-dd HH:mm"; // 年-月-日 时：分
	public static final String PATTERN_HM = "HH:mm"; // 时：分
	public static final String PATTERN_YMD_HMS = "yyyy-MM-dd HH:mm:ss"; // 年-月-日 时：分：秒
	public static final String PATTERN_YMD_HMS_S = "yyyy-MM-dd HH:mm:ss:SSS"; // 年-月-日 时：分：秒：毫秒
	
	@Deprecated
	public static enum DateUtil { DAY, HOUR, MINUTE, SECOND }
	@Deprecated
	public static enum DatePart { YEAR, MONTH, DAY, HOUR, MINUTE, SECOND }

	/**
	 * 获取当前时间
	 * 
	 * @category 获取当前时间
	 * @return 日期
	 */
	public static Date getDate() {
		return new Date();
	}

	/**
	 * 获取当前时间的时间戳
	 * 
	 * @category 获取当前时间的时间戳
	 * @return 时间戳
	 */
	public static long getTime() {
		return new Date().getTime();
	}

	/**
	 * 获取当前的UTC时间戳（秒）
	 * 
	 * @category 获取当前的UTC时间戳（秒）
	 * @return 时间戳
	 */
	public static long getUTCTime() {
		Calendar cal = Calendar.getInstance();
		return cal.getTimeInMillis() / 1000;
	}
	
	/**
	 * 日期时间格式化（按默认的yyyy-MM-dd格式）
	 * 
	 * @category 日期时间格式化
	 * @param date 待格式化的日期
	 * @return 日期字符串
	 */
	public static String format(Date date) {
		return format(date, PATTERN_YMD);
	}
	
	/**
	 * 日期时间格式化
	 * 
	 * @category 日期时间格式化
	 * @param date 日期
	 * @param pattern 格式模板
	 * @return 日期字符串
	 */
	public static String format(Date date, String pattern) {
		DateFormat format = new SimpleDateFormat(pattern);
		return format.format(date);
	}

	/**
	 * 将字符串转换为日期时间
	 * 
	 * @category 将字符串转换为日期时间
	 * @param date 日期字符串
	 * @param pattern 格式模板
	 * @return 日期
	 */
	public static Date parse(String date, String pattern) {
		SimpleDateFormat format = new SimpleDateFormat(pattern);
		try {
			return format.parse(date);
		} catch (Exception e) {
			return getDate();
		}
	}
	
	/**
	 * 自动判断日期格式后，将字符串转换为日期时间
	 * 
	 * @param date 日期字符串
	 * @return 日期
	 */
	public static Date parse(String date){
		String pattern = parsePattern(date);
		return EStr.isNull(pattern) ? getDate() : parse(date, pattern);
	}
	
	/**
	 * 自动判断指定字符串与相应日期时间格式匹配的模板
	 * 无法匹配时，返回null
	 * 
	 * @param date
	 * @return
	 */
	private static String parsePattern(String date){
		String pattern = null;
		if(ERegex.has(date, "^\\d{4}$")){// 年
			pattern = PATTERN_Y;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}$")){// 年-月
			pattern = PATTERN_YM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}$")){// 年-月-日
			pattern = PATTERN_YMD;
		}else if(ERegex.has(date, "^\\d{1,2}\\-\\d{1,2}$")){// 月-日
			pattern = PATTERN_MD;
		}else if(ERegex.has(date, "^\\d{1,2}\\:\\d{1,2}$")){// 时：分
			pattern = PATTERN_HM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分
			pattern = PATTERN_YMD_HM;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}$")){// 年-月-日 时：分：秒
			pattern = PATTERN_YMD_HMS;
		}else if(ERegex.has(date, "^\\d{4}\\-\\d{1,2}\\-\\d{1,2}\\s\\d{1,2}\\:\\d{1,2}\\:\\d{1,2}\\:\\d{1,3}$")){// 年-月-日 时：分：秒：毫秒
			pattern = PATTERN_YMD_HMS_S;
		}
		return pattern;
	}
	
	/**
	 * 返回日期中的某个区间
	 * 
	 * @category 返回日期中的某个区间
	 * @param dp 区间名称
	 * @param dt 日期
	 * @return 字符
	 */
	@Deprecated
	public static String part(DatePart dp, Date dt){
		if(dp == DatePart.YEAR){// 年
			return format(dt, "yyyy");
		}else if(dp == DatePart.MONTH){// 月
			return format(dt, "MM");
		}else if(dp == DatePart.DAY){// 日
			return format(dt, "dd");
		}else if(dp == DatePart.HOUR){// 时
			return format(dt, "HH");
		}else if(dp == DatePart.MINUTE){// 分
			return format(dt, "mm");
		}else{// 秒
			return format(dt, "ss");
		}
	}

	/**
	 * 两个日期的时间差，返回"X天X小时X分X秒"
	 * 
	 * @category 两个日期的时间差，返回"X天X小时X分X秒"
	 * @param start 起始日期
	 * @param end 截止日期
	 * @return 描述
	 */
	public static String diffMsg(Date start, Date end) {
		long diff = (end.getTime() - start.getTime()) / 1000;// 转换成秒
		long day = diff / (24 * 3600);
		long hour = diff % (24 * 3600) / 3600;
		long minute = diff % 3600 / 60;
		long second = diff % 60;
		StringBuilder sb = new StringBuilder();
		sb.append(day);
		sb.append("天");
		sb.append(hour);
		sb.append("小时");
		sb.append(minute);
		sb.append("分");
		sb.append(second);
		sb.append("秒");
		return sb.toString();
	}
	
	/**
	 * 返回两个日期的间隔
	 * 
	 * @category 返回两个日期的间隔
	 * @param dateUtil 间隔单位：DateUtil.DAY（天）/ DateUtil.HOUR（小时）/ DateUtil.MINUTE（分钟）/ DateUtil.SECOND（秒）
	 * @param start 起始日期
	 * @param end 截止日期
	 * @return 间隔
	 */
	@Deprecated
	public static int diff(DateUtil dateUtil, Date start, Date end){
		int diffLen = 0;
		if(dateUtil == DateUtil.DAY){// 相差天数
			diffLen = 24 * 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.HOUR){// 相差小时数
			diffLen = 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.MINUTE){// 相差分钟数
			diffLen = 60 * 1000;
		}else if(dateUtil == DateUtil.SECOND){// 相差秒数
			diffLen = 1000;
		}
		return (int)((end.getTime() - start.getTime()) / diffLen);
	}
	
	/**
	 * 返回两个日期的间隔
	 * 
	 * @category 返回两个日期的间隔
	 * @param start 起始日期
	 * @param end 截止日期
	 * @return 间隔对象
	 */
	public static Duration diff(Date start, Date end){
		LocalDateTime dt1 = toLocalDateTime(start);
		LocalDateTime dt2 = toLocalDateTime(end);
		return diff(dt1, dt2);
	}
	
	/**
	 * 日期计算
	 * 
	 * @category 日期计算
	 * @param dateUtil 间隔单位：DateUtil.DAY（天）/ DateUtil.HOUR（小时）/ DateUtil.MINUTE（分钟）/ DateUtil.SECOND（秒）
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	@Deprecated
	public static Date add(DateUtil dateUtil, Date start, int diff){
		int diffLen = 0;
		if(dateUtil == DateUtil.DAY){// 相差天数
			diffLen = 24 * 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.HOUR){// 相差小时数
			diffLen = 60 * 60 * 1000;
		}else if(dateUtil == DateUtil.MINUTE){// 相差分钟数
			diffLen = 60 * 1000;
		}else if(dateUtil == DateUtil.SECOND){// 相差秒数
			diffLen = 1000;
		}
		return new Date(start.getTime() + diff * diffLen);
	}
	
	/**
	 * 日期计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @param calendarType 计算类型（Calendar.YEAR等）
	 * @return 计算之后的日期
	 */
	private static Date addUtil(Date start, int diff, int calendarType){
		Calendar calendar = new GregorianCalendar();
		calendar.setTime(start);
		calendar.add(calendarType, diff);
		return calendar.getTime();
	}
	
	/**
	 * 年份计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addYear(Date start, int diff){
		return addUtil(start, diff, Calendar.YEAR);
	}
	
	/**
	 * 月份计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addMonth(Date start, int diff){
		return addUtil(start, diff, Calendar.MONTH);
	}
	
	/**
	 * 天数计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addDay(Date start, int diff){
		return addUtil(start, diff, Calendar.DATE);
	}
	
	/**
	 * 小时计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addHour(Date start, int diff){
		return addUtil(start, diff, Calendar.HOUR);
	}
	
	/**
	 * 分钟计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addMinute(Date start, int diff){
		return addUtil(start, diff, Calendar.MINUTE);
	}
	
	/**
	 * 秒数计算
	 * @param start 起始日期
	 * @param diff 计算值
	 * @return 计算之后的日期
	 */
	public static Date addSecond(Date start, int diff){
		return addUtil(start, diff, Calendar.SECOND);
	}

	/**
	 * 得到某一天是星期几
	 * 
	 * @category 得到某一天是星期几
	 * @param date 日期
	 * @return String 星期几
	 */
	@SuppressWarnings("static-access")
	@Deprecated
	public static String getDateInWeek(Date date) {
		String[] weekDays = { "星期日", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六" };
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		int dayIndex = calendar.get(calendar.DAY_OF_WEEK) - calendar.SUNDAY;
		if (dayIndex < 0) dayIndex = 0;
		return weekDays[dayIndex];
	}
	
	/**
	 * 根据日期所在月份，获取该月份的开始和结束日期
	 * 
	 * @param date 日期
	 * @return 日期数组（0：开始日期，1：结束日期）
	 */
	@Deprecated
	public static Date[] getMonthDate(Date date) {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(date);
		calendar.set(Calendar.HOUR_OF_DAY, 0);
		calendar.set(Calendar.MINUTE, 0);
		calendar.set(Calendar.SECOND, 0);
		calendar.set(Calendar.MILLISECOND, 0);

		// 得到前一个月的第一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
		Date start = calendar.getTime();

		// 得到前一个月的最后一天
		calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
		Date end = calendar.getTime();

		return new Date[]{start, end};
	}
	
    /**
     * 判断传入的时间是否在当前时间之后，返回boolean值
     *
     * @param date 要比较的日期
     * @return true: 已过期（即当前日期大于比较日期），false: 未过期（即当前日期小于比较日期）
     */
    public static boolean isExpire(Date date) {
    	return date.before(getDate());
    }
    
    
    /************************************************************************************************************/
    
    /*										JDK 8 中的新日期类型：LocalXXX											*/

    /************************************************************************************************************/
    
    
    
    /**
     * <p>
     * 返回当前时间戳
     * </p>
     * 
     * @return 当前日期（JDK8）
     */
    public static Instant getInstant(){
    	return Instant.now();
    }
    
    /**
     * <p>
     * 返回当前日期
     * </p>
     * 
     * @return 当前日期（JDK8）
     */
    public static LocalDate getLocalDate(){
    	return LocalDate.now();
    }
    
    /**
     * <p>
     * 返回当前日期及时间
     * </p>
     * 
     * @return 当前日期及时间（JDK8）
     */
    public static LocalDateTime getLocalDateTime(){
    	return LocalDateTime.now();
    }
	
	/**
	 * 格式化为LocalDate类型
	 * @param date 日期
	 * @return LocalDate日期
	 */
	public static LocalDate toLocalDate(Date date){
		Instant instant = date.toInstant();
		ZoneId zoneId = ZoneId.systemDefault();
		return instant.atZone(zoneId).toLocalDate();
	}
	
	/**
	 * 格式化为LocalDateTime类型
	 * @param date 日期
	 * @return LocalDateTime日期时间
	 */
	public static LocalDateTime toLocalDateTime(Date date){
		Instant instant = date.toInstant();
		ZoneId zoneId = ZoneId.systemDefault();
		return instant.atZone(zoneId).toLocalDateTime();
	}
	
	/**
	 * <p>
	 * 转换为Date类型
	 * </p>
	 * 
	 * @param dt LocalDate日期
	 * @return 老版本日期对象
	 */
	public static Date toDate(LocalDate dt){
		ZoneId zoneId = ZoneId.systemDefault();
		ZonedDateTime zdt = dt.atStartOfDay(zoneId);
		return Date.from(zdt.toInstant());
	}
	
	/**
	 * <p>
	 * 转换为Date类型
	 * </p>
	 * 
	 * @param dt LocalDateTime日期时间
	 * @return 老版本日期对象
	 */
	public static Date toDate(LocalDateTime dt){
		ZoneId zoneId = ZoneId.systemDefault();
		ZonedDateTime zdt = dt.atZone(zoneId);
		return Date.from(zdt.toInstant());
	}
	
	/**
	 * 返回两个日期的间隔
	 * @param start 起始日期
	 * @param end 截止日期
	 * @return 间隔
	 */
	public static Period diff(LocalDate start, LocalDate end){
		Period period = Period.between(start, end);
		return period;
	}
	
	/**
	 * 返回两个日期时间的间隔
	 * @param start 其实日期时间
	 * @param end 截止日期时间
	 * @return 间隔
	 */
	public static Duration diff(LocalDateTime start, LocalDateTime end){
		Duration duration = Duration.between(start, end);
		return duration;
	}
	
	/**
	 * <p>
	 * 获取格式化模板对象
	 * </p>
	 * 
	 * @param pattern 模板字符串
	 * @return 模板对象
	 */
	public static DateTimeFormatter getPattern(String pattern){
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
		return formatter;
	}
	
	/**
	 * <p>
	 * 日期格式化
	 * </p>
	 * 
	 * @param dt 日期
	 * @return 日期字符串
	 */
	public static String format(LocalDate dt){
		return format(dt, "yyyy-MM-dd");
	}
	
	/**
	 * <p>
	 * 日期格式化
	 * </p>
	 * 
	 * @param dt 日期
	 * @param pattern 模板字符串
	 * @return 日期字符串
	 */
	public static String format(LocalDate dt, String pattern){
		DateTimeFormatter formatter = getPattern(pattern);
		return format(dt, formatter);
	}
	
	/**
	 * <p>
	 * 日期格式化
	 * </p>
	 * 
	 * @param dt 日期
	 * @param formatter 模板对象
	 * @return 日期字符串
	 */
	public static String format(LocalDate dt, DateTimeFormatter formatter){
		return formatter.format(dt);
	}
	
	/**
	 * <p>
	 * 日期时间格式化
	 * </p>
	 * 
	 * @param dt 日期时间
	 * @return 字符串
	 */
	public static String format(LocalDateTime dt){
		return format(dt, "yyyy-MM-dd HH:mm:ss");
	}
	
	/**
	 * <p>
	 * 日期时间格式化
	 * </p>
	 * 
	 * @param dt 日期时间
	 * @param pattern 模板字符串
	 * @return 日期时间字符串
	 */
	public static String format(LocalDateTime dt, String pattern){
		DateTimeFormatter formatter = getPattern(pattern);
		return format(dt, formatter);
	}
	
	/**
	 * <p>
	 * 日期时间格式化
	 * </p>
	 * 
	 * @param dt 日期时间
	 * @param formatter 模板对象
	 * @return 日期时间字符串
	 */
	public static String format(LocalDateTime dt, DateTimeFormatter formatter){
		return formatter.format(dt);
	}
	
	/**
	 * <p>
	 * 日期格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @return 日期对象
	 */
	public static LocalDate parseLocalDate(String dt){
		return parseLocalDate(dt, "yyyy-MM-dd");
	}
	
	/**
	 * <p>
	 * 日期格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @param pattern 模板字符串
	 * @return 日期对象
	 */
	public static LocalDate parseLocalDate(String dt, String pattern){
		DateTimeFormatter formatter = getPattern(pattern);
		return parseLocalDate(dt, formatter);
	}
	
	/**
	 * <p>
	 * 日期格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @param formatter 模板字符串
	 * @return 日期对象
	 */
	public static LocalDate parseLocalDate(String dt, DateTimeFormatter formatter){
		return LocalDate.parse(dt, formatter);
	}
	
	/**
	 * <p>
	 * 日期时间格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @return 日期时间对象
	 */
	public static LocalDateTime parseLocalDateTime(String dt){
		return parseLocalDateTime(dt, "yyyy-MM-dd");
	}
	
	/**
	 * <p>
	 * 日期时间格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @param pattern 模板字符串
	 * @return 日期时间对象
	 */
	public static LocalDateTime parseLocalDateTime(String dt, String pattern){
		DateTimeFormatter formatter = getPattern(pattern);
		return parseLocalDateTime(dt, formatter);
	}
	
	/**
	 * <p>
	 * 日期时间格式解析
	 * </p>
	 * 
	 * @param dt 日期字符串
	 * @param formatter 模板对象
	 * @return 日期时间字符串
	 */
	public static LocalDateTime parseLocalDateTime(String dt, DateTimeFormatter formatter){
		return LocalDateTime.parse(dt, formatter);
	}

}
