/**
 * 
 */
package itez.kit;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Arrays;

import itez.kit.exception.ExceptionKit;
import itez.kit.log.ELogBase;
import itez.kit.restful.Result;

/**
 * <p>
 * 执行本地命令工具类
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月5日 上午12:51:30
 */
public class ECommand {

	static final ELogBase log = ELog.log(ECommand.class);
	
	public static enum OS {
		WIN("<i class=\"fa fa-windows\"></i> Windows"), LINUX("<i class=\"fa fa-linux\"></i> Linux"), OTHER("其他");
		private String caption;
		OS(String caption){
			this.caption = caption;
		}
		public String getCaption(){
			return caption;
		}
	}
	
	private OS os;
	private Runtime run;
	private static ECommand ins = null;
	
	public static ECommand getInstance(){
		if(ins == null) ins = new ECommand();
		return ins;
	}
	
	private ECommand(){
		String osName = System.getProperty("os.name").toLowerCase();
		os = osName.contains("windows") ? OS.WIN : osName.contains("linux") ? OS.LINUX : OS.OTHER;
		run = Runtime.getRuntime();
	}
	
//	private String getOsCmd(String command){
//		return (os == OS.WIN ? "cmd /c " : os == OS.LINUX ? "/bin/sh -c " : "") + command;
//	}
	
	private String[] getOsCmdArr(String command){
		String[] cmds = new String[3];
		if(os == OS.WIN){
			cmds[0] = "cmd";
			cmds[1] = "/c";
		}else{
			cmds[0] = "/bin/sh";
			cmds[1] = "-c";
		}
		cmds[2] = command;
		return cmds;
	}
	
	/**
	 * <p>
	 * 命令行模式下执行命令
	 * </p>
	 * 
	 * @param command 命令名称及参数
	 * @return
	 */
	public Result cmd(String command){
		return exec(getOsCmdArr(command), null, null);
	}
	
	/**
	 * <p>
	 * 命令行模式下执行命令，并设置环境变量
	 * </p>
	 * 
	 * @param command
	 * @param envp 字符串数组，每个变量的格式：varName=varValue
	 * @return
	 */
	public Result cmd(String command, String [] envp){
		return exec(getOsCmdArr(command), envp, null);
	}
	
	/**
	 * <p>
	 * 命令行模式下执行命令，并设置当前所在的目录位置
	 * </p>
	 * 
	 * @param command
	 * @param dir
	 * @return
	 */
	public Result cmd(String command, File dir){
		return exec(getOsCmdArr(command), null, dir);
	}
	
	/**
	 * <p>
	 * 命令行模式下执行命令，同时设置环境变量以及目录位置
	 * </p>
	 * 
	 * @param command
	 * @param envp 字符串数组，每个变量的格式：varName=varValue
	 * @param dir
	 * @return
	 */
	public Result cmd(String command, String [] envp, File dir){
		return exec(getOsCmdArr(command), null, dir);
	}
	
	/**
	 * <p>
	 * 执行普通命令
	 * </p>
	 * 
	 * @param command
	 * @return
	 */
	public Result exec(String[] command){
		return exec(command, null, null);
	}
	
	/**
	 * <p>
	 * 执行普通命令，同时设置环境变量
	 * </p>
	 * 
	 * @param command
	 * @param envp 字符串数组，每个变量的格式：varName=varValue
	 * @return
	 */
	public Result exec(String[] command, String [] envp){
		return exec(command, envp, null);
	}
	
	/**
	 * <p>
	 * 执行普通命令，同时设置目录位置
	 * </p>
	 * 
	 * @param command
	 * @param dir
	 * @return
	 */
	public Result exec(String[] command, File dir){
		return exec(command, null, dir);
	}
	
	/**
	 * <p>
	 * 执行普通命令，同时指定环境变量及目录位置
	 * </p>
	 * 
	 * @param command
	 * @param envp
	 * @param dir
	 * @return
	 */
	public Result exec(String[] command, String[] envp, File dir){
		try {
			//log.info("执行命令内容：{}", command[2]);
			Process proc = run.exec(command, envp, dir);
            InputStream ins = proc.getInputStream();
            String[] msg = {""};
            new Thread(new inputStreamThread(ins, msg)).start();
            int exitVal = proc.waitFor();
            if(exitVal == 0){
            	return Result.success("res", msg[0]);
            }else{
            	log.error("执行命令时返回错误信息，命令：{}", Arrays.toString(command));
            	if(dir != null) log.error("执行命令时返回错误信息，位置：{}", dir.getAbsolutePath());
            	log.error("执行命令时返回错误信息，返回：{}", msg[0]);
            	return Result.fail(msg[0]);
            }
		} catch (IOException e) {
			String expMsg = ExceptionKit.getExceptionMore(e);
			if(EProp.DevMode) e.printStackTrace();
			else log.error(expMsg);
            return Result.fail(expMsg);
		} catch (InterruptedException e) {
			String expMsg = ExceptionKit.getExceptionMore(e);
			if(EProp.DevMode) e.printStackTrace();
			else log.error(expMsg);
            return Result.fail(expMsg);
		}
	}
	
	public static void main(String[] args) {
		ECommand cmd = new ECommand();
		Result ret = cmd.cmd("dir > D:/dir.txt");
		System.out.println(ret.getState());
		System.out.println(ret.getData().get("res"));
	}

	public OS getOs() {
		return os;
	}

	public Runtime getRun() {
		return run;
	}
	
	/**
	 * <p>
	 * 获取执行结果
	 * </p>
	 * 
	 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
	 * 
	 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
	 * @date		2021年4月5日 上午12:53:49
	 */
	class inputStreamThread implements Runnable{
		
        private BufferedReader br = null;
        private String[] msg;
        
        public inputStreamThread(InputStream ins, String[] msg){
        	Charset charset = ECommand.getInstance().os == OS.WIN ? EStr.GBK : EStr.UTF_8;
            this.br = new BufferedReader(new InputStreamReader(ins, charset));
            this.msg = msg;
        }
        
        @Override
        public void run() {
        	StringBuffer sb = new StringBuffer();
        	String line = null;
            try {
            	while ((line = br.readLine()) != null) {
                	sb.append(line + "\n");
                }
            } catch (IOException e) {
            	e.printStackTrace();
            }
            msg[0] = sb.toString();
        }
    }
	
}
