package itez.core.wrapper.interceptor;

import java.lang.reflect.Method;

import com.jfinal.aop.Interceptor;
import com.jfinal.aop.Invocation;

import itez.core.runtime.EContext;
import itez.core.runtime.auth.AuthMode;
import itez.core.runtime.auth.AuthRequire;
import itez.core.runtime.auth.AuthService;
import itez.core.runtime.modules.ModuleManager;
import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.session.EAttr;
import itez.core.runtime.session.ESessionBase;
import itez.core.wrapper.controller.EController;
import itez.kit.EWeb;
import itez.kit.restful.Result;

public class AuthInterceptor implements Interceptor {
	
	@Override
	public void intercept(Invocation inv) {
		EController controller = (EController)inv.getController();
		EAttr attr = EContext.getAttr();

        String header = controller.getRequest().getHeader("content-type");
		boolean isJson = (header != null && header.contains("json"));
		
		/**
		 * 获取登录状态
		 */
		ESessionBase session = controller.session();
		Boolean logined = session.getLogined();

		/**
		 * 解析鉴权模式：
		 * 1）先判断Action是否要求鉴权；
		 * 2）如果没有，则继续判断Controller是否要求鉴权；
		 * 3）如果仍然没有，则默认允许匿名访问
		 */
		Method method = inv.getMethod(); //当前访问的Action
		AuthMode mode = new AuthMode(); //鉴权模式
		getMethodAuthMode(mode, method, controller.getClass()); //先判断Action的鉴权模式
		
		boolean authAccept = false; //鉴权结果，默认为失败
		String authMsg = ""; //错误提示

		if(mode.getAuthCode() == AuthCode.REQ_GUEST){ //允许匿名
			authAccept = true; //鉴权通过
		}else if(!logined){ //不允许匿名，且当前未登录
			if(isJson){
				controller.renderJson(Result.fail("请先登录！"));
			}else{
				//判断是否记忆登录前的来源地址
				String jumpUrl = attr.getLoginUrl();
				String fromUrl = inv.getActionKey();
				String defPath = ModuleManager.me.getDefaultModule().getModulePath();
				if(!fromUrl.equals(defPath)) jumpUrl = jumpUrl.concat("?from=").concat(EWeb.UrlEncoder(attr.getRequestUrl()));
				controller.setFlashMsg("请先登录");
				controller.redirect(jumpUrl);
			}
			return;
		}else{
			Result valiRet = AuthService.authValidator(mode);
			if(valiRet.getState()){
				authAccept = true;
			}else{
				authMsg = valiRet.getMsg();
			}
		}
		
		if(!authAccept){
			String msg = String.format("所需的角色或权限：%s", authMsg);
			if(isJson){
				controller.renderJson(Result.fail(msg));
			}else{
				controller.renderErrMsg(mode.getAuthCode(), msg);
			}
		}else{
			inv.invoke();
		}
	}
	
	/**
	 * 优先进行Action层面的鉴权
	 * @param method
	 * @param ctrl
	 */
	private void getMethodAuthMode(AuthMode mode, Method method, Class<? extends EController> ctrlClass){
		if(method.isAnnotationPresent(AuthRequire.Roles.class)){
			mode.setAuthCode(AuthCode.REQ_ROLES);
			mode.setAuthIds(method.getAnnotation(AuthRequire.Roles.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.Role.class)){
			mode.setAuthCode(AuthCode.REQ_ROLE);
			mode.setAuthId(method.getAnnotation(AuthRequire.Role.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.RolesAny.class)){
			mode.setAuthCode(AuthCode.REQ_ROLES_ANY);
			mode.setAuthIds(method.getAnnotation(AuthRequire.RolesAny.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.Perms.class)){
			mode.setAuthCode(AuthCode.REQ_PERMS);
			mode.setAuthIds(method.getAnnotation(AuthRequire.Perms.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.Perm.class)){
			mode.setAuthCode(AuthCode.REQ_PERM);
			mode.setAuthId(method.getAnnotation(AuthRequire.Perm.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.PermsAny.class)){
			mode.setAuthCode(AuthCode.REQ_PERMS_ANY);
			mode.setAuthIds(method.getAnnotation(AuthRequire.PermsAny.class).value());
		}else if(method.isAnnotationPresent(AuthRequire.Logined.class)){
			mode.setAuthCode(AuthCode.REQ_LOGIN);
		}else if(method.isAnnotationPresent(AuthRequire.Guest.class)){
			mode.setAuthCode(AuthCode.REQ_GUEST);
		}else{
			getControllerAuthMode(mode, ctrlClass);
		}
	}
	
	/**
	 * 进行Controller层面的鉴权，只有当Action未设置时有效
	 * @param ctrl
	 */
	private void getControllerAuthMode(AuthMode mode, Class<?> ctrl){
		if(ctrl.isAnnotationPresent(AuthRequire.Roles.class)){
			mode.setAuthCode(AuthCode.REQ_ROLES);
			mode.setAuthIds(ctrl.getAnnotation(AuthRequire.Roles.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.Role.class)){
			mode.setAuthCode(AuthCode.REQ_ROLE);
			mode.setAuthId(ctrl.getAnnotation(AuthRequire.Role.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.RolesAny.class)){
			mode.setAuthCode(AuthCode.REQ_ROLES_ANY);
			mode.setAuthIds(ctrl.getAnnotation(AuthRequire.RolesAny.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.Perms.class)){
			mode.setAuthCode(AuthCode.REQ_PERMS);
			mode.setAuthIds(ctrl.getAnnotation(AuthRequire.Perms.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.Perm.class)){
			mode.setAuthCode(AuthCode.REQ_PERM);
			mode.setAuthId(ctrl.getAnnotation(AuthRequire.Perm.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.PermsAny.class)){
			mode.setAuthCode(AuthCode.REQ_PERMS_ANY);
			mode.setAuthIds(ctrl.getAnnotation(AuthRequire.PermsAny.class).value());
		}else if(ctrl.isAnnotationPresent(AuthRequire.Logined.class)){
			mode.setAuthCode(AuthCode.REQ_LOGIN);
		}else{
			mode.setAuthCode(AuthCode.REQ_GUEST);
		}
	}
	
}
