package itez.core.wrapper.handler;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import itez.core.runtime.EContext;
import itez.core.runtime.service.Ioc;
import itez.core.runtime.service.common.IComp;
import itez.core.runtime.service.common.ICompService;
import itez.core.runtime.service.common.ISiteDomain;
import itez.core.runtime.session.EAttr;
import itez.core.waf.IWaf;
import itez.core.waf.IWafInfo;
import itez.kit.EClass;
import itez.kit.ELog;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.log.ELogBase;

/**
 * 二级域名解析
 * 支持两种域名模式：
 * 1）前缀：（http://[domain].xxxxx.com/...），要求域名开通泛解析
 * 2）后缀：（http://www.xxxxx.com/[domain]/...），无需域名泛解析
 * 
 * @author netwild
 *
 */
public class EDomainHandler extends EHandler {

	private final static ELogBase log = ELog.log(EDomainHandler.class);
	
	//默认的顶级域名（为空时根据请求地址自动获取；否则以该顶级域名生成各种链接地址串）
	//private final static String DomainName = EProp.DomainName;
	//是否开启泛域名解析（true：开启，false：关闭）
	private final static boolean DomainUsed = EProp.DomainHandle;
	//域名解析方式（前缀：prefix；后缀：postfix，其中前缀模式需服务器支持泛域名解析）
	private final static String DomainMode = EProp.DomainMode.toUpperCase();
	//默认域名（当泛域名解析关闭，或未匹配到域名时，将使用默认域名）
	private final static String DomainDefault = EProp.DomainDefault;
	
	/**
	 * 域名解析模式枚举
	 * PREFIX：前缀（http://[domain].xxxxx.com/...），要求域名开通泛解析
	 * POSTFIX：后缀（http://www.xxxxx.com/[domain]/...），无需域名泛解析
	 */
	private static enum DomainModeDict { PREFIX, POSTFIX }
	
	private final static Pattern IpAddrPattern = Pattern.compile("^(\\d){1,3}(\\.(\\d){1,3}){3}$");									//IP地址
	private final static Pattern TopDomainPattern = Pattern.compile("^[\\w]+(.com|.net|.cn|.com.cn|.net.cn|.org.cn|.edu.cn)$");		//缺省主机域名模式
	private final static Pattern DomainPrefixPattern = Pattern.compile("^([^.]+)\\.");												//前缀子域匹配（从serverName中获取子域）
	private final static Pattern DomainPostfixPattern = Pattern.compile("^(\\/)([^/]+)");											//后缀子域匹配（从target中获取子域）
	
	// domain和target在Request.Attr中的键名
	public final static String DomainAttrName = "JW_DOMAIN";
	public final static String TargetAttrName = "JW_TARGET";
	
	/**
	 * WEB防火墙相关
	 */
	private static boolean wafExist = false;		//WEB防火墙是否存在
	private static IWaf wafLoader = null;			//WEB防火墙加载对象
	
	private final ICompService compSer = Ioc.getComp();
	private final ISiteDomain siteDomainSer = Ioc.get(ISiteDomain.class);
	
	public EDomainHandler() {
		
		/**
		 * 自动检测是否存在Web防火墙
		 */
		String className = "itez.plat.waf.util.WafLoader";
		if(null != EClass.forName(className)){
			log.info("WAF：已发现 WafLoader");
			wafLoader = EClass.newInstance(className);
			if(wafLoader != null){
				wafExist = true;
				log.info("WAF：WafLoader初始化成功");
			}else{
				log.info("WAF：WafLoader初始化失败");
			}
		}else{
			log.info("WAF：未发现 WafLoader");
		}
		
	}
	
	/**
	 * 执行过滤处理
	 * isHandled：标识该请求是否已被终止（true：已终止，无需继续发送到容器；false：未终止，由容器继续提供响应）
	 */
	@Override
	public void handle(String target, HttpServletRequest request, HttpServletResponse response, boolean[] isHandled) {

		//静态资源直接跳出，由容器提供响应（如果需要由WAF保护静态资源，需要注释该行）
		if(target.indexOf('.') != -1) return;
		//客户端资源直接跳出，由容器提供响应
		if(target.startsWith("/client/")) return;
		//静态网站资源直接跳出，由容器提供响应
		if(target.startsWith("/sitePage/")) return;
				
		//获得默认租户
		String domainDef = EProp.DomainDefault;
		IComp compDef = compSer.findByDomain(domainDef);
		if(compDef == null){
			String errMsg = String.format("初始化错误：默认子域 <b>%s</b> 无效！", domainDef);
			renderErr(response, errMsg);
			log.error(errMsg);
			isHandled[0] = true;
			return;
		}
		
		//设置解析相关的基本变量
		DomainModeDict domainMode = DomainModeDict.valueOf(DomainMode);	//域名模式
		String serverName = request.getServerName();					//主机名
		String domain = DomainDefault;				 					//目标子域
		Boolean bind = false;											//是否已绑定自定义域名
		
		//已开启域名解析，开始解析目标子域
		if(DomainUsed){
			
			String sysDomain = siteDomainSer.getBind(serverName);		//检索自定义域名绑定情况
			if(EStr.notEmpty(sysDomain)){								//已为子域绑定自定义域名
				bind = true;
				domain = sysDomain;
			}else{														//按主机名自动解析域名
				
				//如果是计算机名或者IP地址，只能采用后缀模式
				if(serverName.indexOf(".") < 1 || IpAddrPattern.matcher(serverName).matches()) domainMode = DomainModeDict.POSTFIX;
				
				Matcher matcher;
				if(domainMode == DomainModeDict.PREFIX){				//前缀模式
					if(checkTopDomain(serverName)){						//如果是缺省主机模式，例如：https://itez.com.cn
						domain = domainDef;
					}else{												//普通模式
						matcher = DomainPrefixPattern.matcher(serverName);
						if(matcher.find() && matcher.groupCount() > 0){
							domain = matcher.group(1).toLowerCase();
						}
					}
				}else{													//后缀模式
					matcher = DomainPostfixPattern.matcher(target);
					if(matcher.find() && matcher.groupCount() > 0){
						domain = matcher.group(2).toLowerCase();
						target = matcher.replaceAll("");
					}
				}
			}
			
		}
				
		//根据domain返回公司信息
		IComp comp = domain.equals(domainDef) ? compDef : compSer.findByDomain(domain);
		if(comp == null){
			String errMsg = String.format("请求错误：该子域 <b>%s</b> 不存在！", domain);
			renderErr(response, errMsg);
			isHandled[0] = true;
			return;
		}else if(!bind && comp.getDisableDomain()){ //该租户禁止使用二级子域访问
			String errMsg = String.format("请求错误：该子域 <b>%s</b> 已停止使用！", domain);
			renderErr(response, errMsg);
			isHandled[0] = true;
			return;
		}
		
		//将domain和target存入Request
		request.setAttribute(DomainAttrName, domain); 
		request.setAttribute(TargetAttrName, target);
		
		//WAF防火墙
		IWafInfo wafInfo = null;
		
		try {
			if(wafExist){ //存在WAF模块
				wafInfo = wafLoader.gather(domain, target, request, response); //收集信息
				if(!wafLoader.before(wafInfo)){ //执行防火墙请求拦截
					wafLoader.saveLog(wafInfo); //保存溯源日志
					renderErr(response, EStr.ifEmpty(wafInfo.getDesc(), "异常访问，已被拦截！"));
					isHandled[0] = true;
					return;
				}
			}
			
			boolean appendDomain = (domainMode == DomainModeDict.POSTFIX && !bind);
			EAttr attr = new EAttr(request, compDef, comp, appendDomain); //业务环境参数
            EContext.bind(attr, request, response); //将业务环境参数存入线程
			next.handle(target, request, response, isHandled); //访问资源
		}finally {
			if(wafInfo != null){
				if(!wafLoader.after(wafInfo)){ //执行防火墙响应拦截
					wafLoader.saveLog(wafInfo); //保存溯源日志
					renderErr(response, EStr.ifEmpty(wafInfo.getDesc(), "异常访问，已被拦截！"));
					isHandled[0] = true;
					return;
				}
				wafLoader.saveLog(wafInfo); //保存溯源日志
			}
			
			EContext.release(); //销毁业务环境线程
		}
	}
	
	/**
	 * 判断是否为缺省主机方式（例如：https://itez.com.cn）
	 * @param serverName
	 * @return
	 */
	private boolean checkTopDomain(String serverName){
		return TopDomainPattern.matcher(serverName).matches();
	}

}
