package itez.core.wrapper.dbo.model;

import itez.core.runtime.EContext;
import itez.core.runtime.service.common.IComp;
import itez.core.runtime.service.common.IUser;
import itez.core.wrapper.dbo.DboException;
import itez.core.wrapper.dbo.dialect.IDialect;
import itez.kit.ELog;
import itez.kit.EStr;
import itez.kit.EUid;
import itez.kit.exception.EAssert;
import itez.kit.log.ELogBase;
import itez.kit.rsa.RsaKit;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.alibaba.fastjson.annotation.JSONField;
import com.beust.jcommander.internal.Sets;
import com.jfinal.plugin.activerecord.ActiveRecordException;
import com.jfinal.plugin.activerecord.Config;
import com.jfinal.plugin.activerecord.Db;
import com.jfinal.plugin.activerecord.DbPro;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Page;
import com.jfinal.plugin.activerecord.Table;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class EModel<M extends EModel> extends Model<M> {

	private static final long serialVersionUID = 1L;
	private static final ELogBase log = ELog.log(EModel.class);
	
	//======================= 约定字段名称 ==================================

	protected static final String COLUMN_DOMAIN = "domain"; //租户子域
	protected static final String COLUMN_CREATED = "cdate"; //创建时间
	protected static final String COLUMN_MODIFIED = "mdate"; //修改时间
	protected static final String COLUMN_REMOVED = "rdate"; //逻辑删除时间
	protected static final String COLUMN_USED = "used"; //逻辑删除标识
	protected static final String COLUMN_SORT = "sort"; //序号
	
	//======================= ORM对象初始化 ==================================
	
	@Override
	public Map<String, Object> _getAttrs() {
		return super._getAttrs();
	}
	
	/**
	 * 返回Model的配置
	 * @return
	 */
	public Config _getConfig() {
		return super._getConfig();
	}
	
	@Override
	public Table _getTable() {
		return super._getTable();
	}
	
	/**
	 * 返回Model的映射表名
	 * @return
	 */
	protected String _getTableName() {
		return _getTable().getName();
	}
	
	/**
	 * 返回当前数据库方言
	 * @return
	 */
	private IDialect _getDialect(){
		return (IDialect) _getConfig().getDialect();
	}
	
	//======================= 上下文环境 ==================================
	
	/**
	 * 获得UUID
	 * @return
	 */
	protected String $UUID(){
		return EUid.generator();
	}

	/**
	 * 获得当前租户的子域
	 * @return
	 */
	protected String $domain() {
		return EContext.getAttr() == null ? null : EContext.getAttr().getDomain();
	}

	/**
	 * 获得当前租户的实例
	 * @return
	 */
	protected IComp $comp() {
		return EContext.getAttr() == null ? null : EContext.getAttr().getComp();
	}
	
	/**
	 * 获得当前用户的登录状态
	 * @return
	 */
	protected Boolean $logined(){
		return EContext.getSession() == null ? null : EContext.getSession().getLogined();
	}

	/**
	 * 获得当前登录用户的ID
	 * @return
	 */
	protected String $uid() {
		return EContext.getSession() == null ? null : EContext.getSession().getUid();
	}

	/**
	 * 获得当前登录用户的实例
	 * @return
	 */
	protected IUser $user() {
		return EContext.getSession() == null ? null : EContext.getSession().getUser();
	}
	
	//======================= 非空校验 ==================================
	
	/**
	 * 非空校验
	 * @param cols 要校验的字段列表，为NULL时校验全部字段
	 * @return
	 */
	public boolean required(String...colsRequires){
		Set<String> cols = Sets.newHashSet();
		if(null == colsRequires){
			cols = _getTable().getColumnNameSet();
		}else{
			cols = Stream.of(colsRequires).collect(Collectors.toSet());
		}
		if(cols.size() == 0) return true;
		for(String col : cols){
			Object val = this.get(col);
			if(EStr.isEmpty(val)) return false;
		};
		return true;
	}
	
	//======================= 基本数据检索 ==================================
	
	/**
	 * 根据主键数组返回对象列表
	 * @param ids
	 * @return
	 */
	public List<M> findByIda(String...ids){
		Querys qs = Querys.and(Query.in(getPk(), EStr.ida2sqlIn(ids)));
		return select(qs, null, null, false);
	}
	
	/**
	 * 根据主键字符串返回对象列表
	 * @param ids
	 * @return
	 */
	public List<M> findByIds(String ids){
		Querys qs = Querys.and(Query.in(getPk(), EStr.ids2sqlIn(ids)));
		return select(qs, null, null, false);
	}
	
	public M findByCode(String code){
    	Querys qs = Querys.and(Query.eq("code", code));
    	return selectFirst(qs);
	}
	
	//======================= 按条件检索，返回单个字段的列表形式 ==================================
	
	/**
	 * 返回指定列的数据
	 * @param querys
	 * @param orderBy
	 * @param limit
	 * @param column
	 * @return
	 */
    public <T> List<T> selectCol(Querys querys, String orderBy, Object limit, String column){
    	if(querys == null) querys = Querys.and();
    	checkQuerys(querys);
    	if(EStr.isEmpty(orderBy) && hasColumn(COLUMN_SORT)) orderBy = "sort";
    	String sql = _getDialect().genQuery(_getTableName(), column, querys, null, orderBy, limit);
    	DbPro db = Db.use(_getConfig().getName());
    	return db.query(sql, querys.getArgs());
    }
	
	//======================= 按条件检索第一条 ==================================

    /**
     * 返回第一条
     * @return
     */
    public M selectFirst(){
    	Querys qs = Querys.and();
    	return selectFirst(qs, null);
    }

    /**
     * 返回第一条，指定列
     * @param columnNames
     * @return
     */
    public M selectFirst(String...columnNames){
    	Querys qs = Querys.and();
    	return selectFirst(qs, null, columnNames);
    }
    
    /**
     * 按检索条件返回第一条
     * @param querys
     * @return
     */
    public M selectFirst(Querys querys){
    	return selectFirst(querys, null);
    }
    
    /**
     * 按检索条件、排序方式返回第一条
     * @param querys
     * @param orderBy
     * @return
     */
    public M selectFirst(Querys querys, String orderBy){
    	List<M> list = select(querys, orderBy, 1);
    	if(list != null && list.size() > 0){
    		return list.get(0);
    	}else{
    		return null;
    	}
    }
    
    /**
     * 按检索条件、排序方式、指定列返回第一条
     * @param querys
     * @param orderBy
     * @param columnNames
     * @return
     */
    public M selectFirst(Querys querys, String orderBy, String...columnNames){
    	List<M> list = select(querys, orderBy, 1, columnNames);
    	if(list != null && list.size() > 0){
    		return list.get(0);
    	}else{
    		return null;
    	}
    }
    
    /**
     * 按检索条件、排序方式、指定列返回第一条
     * @param querys
     * @param orderBy
     * @param columnNames
     * @return
     */
    public M selectFirst(Querys querys, String orderBy, Boolean autoCheck, String...columnNames){
    	List<M> list = select(querys, orderBy, 1, autoCheck, columnNames);
    	if(list != null && list.size() > 0){
    		return list.get(0);
    	}else{
    		return null;
    	}
    }
    
    /**
     * 使用SQL选择器检索
     * @param selector
     * @return
     */
    public M selectFirst(Selector selector){
    	List<M> list = select(selector.setLimit(1));
    	if(list != null && list.size() > 0){
    		return list.get(0);
    	}else{
    		return null;
    	}
    }
    
	//======================= 按条件检索 ==================================

    /**
     * 按条件检索
     * @param querys
     * @param orderBy
     * @param limit
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String orderBy, Object limit, String...loadColumns) {
    	return select(querys, orderBy, limit, true, loadColumns);
    }
    
    /**
     * 按条件检索
     * @param querys 检索条件
     * @param orderBy 排序方式
     * @param limit 返回条数限制
     * @param autoCheck 是否自动附加其他条件（租户子域、租户ID、可用标识）
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String orderBy, Object limit, Boolean autoCheck, String...loadColumns) {
    	querys = querys == null ? Querys.and() : Querys.and(querys);
    	if(autoCheck) checkQuerys(querys);
    	if(EStr.isEmpty(orderBy) && hasColumn(COLUMN_SORT)) orderBy = "sort";
    	String cols = Arrays.stream(loadColumns).collect(Collectors.joining(","));
    	IDialect dialect = _getDialect();
    	String tableName = _getTableName();
    	String sql = dialect.genQuery(tableName, cols, querys, null, orderBy, limit);
    	return querys == null ? find(sql) : find(sql, querys.getArgs());
    }
    
    /**
     * 按条件检索
     * @param querys 检索条件
     * @param groupBy 分组方式
     * @param orderBy 排序方式
     * @param limit 返回条数限制
     * @param autoCheck 是否自动附加其他条件（租户子域、租户ID、可用标识）
     * @param loadColumns
     * @return
     */
    public List<M> select(Querys querys, String groupBy, String orderBy, Object limit, Boolean autoCheck, String...loadColumns) {
    	querys = querys == null ? Querys.and() : Querys.and(querys);
    	if(autoCheck) checkQuerys(querys);
    	if(EStr.isEmpty(orderBy) && hasColumn(COLUMN_SORT)) orderBy = "sort";
    	String cols = Arrays.stream(loadColumns).collect(Collectors.joining(","));
    	String sql = _getDialect().genQuery(_getTableName(), cols, querys, groupBy, orderBy, limit);
    	return querys == null ? find(sql) : find(sql, querys.getArgs());
    }
    
    /**
     * 按条件检索
     * @param selector SQL选择器
     * @return
     */
    public List<M> select(Selector selector){
    	return select(selector, true);
    }
    
    /**
     * 按条件检索
     * @param selector SQL选择器
     * @return
     */
    public List<M> select(Selector selector, Boolean autoCheck){
    	Querys querys = selector.getQuerys();
    	querys = querys == null ? Querys.and() : Querys.and(querys);
    	if(autoCheck) checkQuerys(querys);
    	String orderBy = selector.getOrderBy();
    	if(EStr.isEmpty(orderBy) && hasColumn(COLUMN_SORT)) orderBy = "sort";
    	String[] columns = selector.getColumns();
    	String cols = columns != null ? Arrays.stream(columns).collect(Collectors.joining(",")) : null;
    	String sql = _getDialect().genQuery(_getTableName(), cols, querys, selector.getGroupBy(), orderBy, selector.getLimit());
    	return querys == null ? find(sql) : find(sql, querys.getArgs());
    }
    
    /**
     * 分页检索
     * @param pageNumber
     * @param pageSize
     * @param selector
     * @return
     */
    public Page<M> paginate(int pageNumber, int pageSize, Selector selector){
    	return paginate(pageNumber, pageSize, selector, true);
    }
    
    /**
     * 分页检索
     * @param pageNumber
     * @param pageSize
     * @param selector
     * @param autoCheck 是否自动附加其他条件（租户子域、租户ID、可用标识）
     * @return
     */
    public Page<M> paginate(int pageNumber, int pageSize, Selector selector, Boolean autoCheck){
    	Querys querys = selector.getQuerys();
    	querys = querys == null ? Querys.and() : Querys.and(querys);
    	if(autoCheck) checkQuerys(querys);
    	String groupBy = selector.getGroupBy();
    	String orderBy = selector.getOrderBy();
    	if(EStr.isEmpty(orderBy) && hasColumn(COLUMN_SORT)) orderBy = "sort";
    	String[] columns = selector.getColumns();
    	String cols = columns != null ? Arrays.stream(columns).collect(Collectors.joining(",")) : "*";
    	String sql_select = _getDialect().genPageSelect(cols);
    	String sql_query = _getDialect().genPageQuery(_getTableName(), querys, groupBy, orderBy);
    	if(querys == null)
    		if(EStr.isEmpty(groupBy))
    			return paginate(pageNumber, pageSize, sql_select, sql_query);
    		else
    			return paginate(pageNumber, pageSize, true, sql_select, sql_query);
    	else
    		if(EStr.isEmpty(groupBy))
    			return paginate(pageNumber, pageSize, sql_select, sql_query, querys.getArgs());
    		else
    			return paginate(pageNumber, pageSize, true, sql_select, sql_query, querys.getArgs());
    }
	
	//======================= 保存及更新 ==================================

    /**
     * 更新或者保存
     * 有主键就更新，没有就保存
     *
     * @return
     */
    public boolean saveOrUpdate() {
    	return (null == get(getPk())) ? this.save() : this.update();
    }

    /**
     * 保存数据
     *
     * @return
     */
    @Override
    public boolean save() {
        if (null == get(getPk()) && String.class == getPkType()) set(getPk(), $UUID());
        if (hasColumn(COLUMN_DOMAIN) && EStr.isEmpty(get(COLUMN_DOMAIN))) set(COLUMN_DOMAIN, $domain());
        if (hasColumn(COLUMN_CREATED) && get(COLUMN_CREATED) == null) set(COLUMN_CREATED, new Date());
        if (hasColumn(COLUMN_MODIFIED) && get(COLUMN_MODIFIED) == null) set(COLUMN_MODIFIED, new Date());
        if (hasColumn(COLUMN_USED) && get(COLUMN_USED) == null) set(COLUMN_USED, 1);
        return super.save();
    }
    
    /**
     * 更新
     *
     * @return
     */
    @Override
    public boolean update() {
        if (hasColumn(COLUMN_MODIFIED)) set(COLUMN_MODIFIED, new Date());
        return super.update();
    }
	
	//======================= 物理删除及逻辑删除 ==================================

    /**
     * 根据ID删除
     * @param idValue 主键值
     * @param logic 是否逻辑删除
     * @return
     */
    @Override
    public boolean deleteById(Object idValue) {
        M model = findById(idValue);
        return model == null ? true : model.delete();
    }
    
    /**
     * 是否可用
     * @return
     */
    public boolean able(){
        if (hasColumn(COLUMN_USED)) {
            Integer used = get(COLUMN_USED);
            if(used == null || used == 0) return false;
            else return true;
        }
        return false;
    }
    
    /**
     * 禁用
     * @return
     */
    public boolean disable() {
    	if(!able()) return true;
        if (hasColumn(COLUMN_REMOVED)) set(COLUMN_REMOVED, new Date());
        if (hasColumn(COLUMN_USED)) set(COLUMN_USED, 0);
        return super.update();
    }
    
    /**
     * 启用
     * @return
     */
    public boolean enable() {
    	if(able()) return true;
        if (hasColumn(COLUMN_MODIFIED)) set(COLUMN_REMOVED, new Date());
        if (hasColumn(COLUMN_USED)) set(COLUMN_USED, 1);
        return super.update();
    }
	
	//======================= 元数据 ==================================
    
    /**
     * 在检索条件中自动补充租户标识（租户子域、租户ID、可用标识）
     * @param querys
     */
    private void checkQuerys(Querys querys){
    	if(hasColumn(COLUMN_USED)){
    		querys.add(Query.eq(COLUMN_USED, 1));
    	}
    	if(hasColumn(COLUMN_DOMAIN)){
    		String domain = $domain();
    		if(domain != null){
    			Querys qsor = Querys.or(Query.nu(COLUMN_DOMAIN)).add(Query.eq(COLUMN_DOMAIN, "")).add(Query.eq(COLUMN_DOMAIN, domain));
    			querys.add(qsor);
    		}
    	}
    }

    /**
     * 返回主键信息
     */
    private transient String primaryKey;
    
    @JSONField(serialize = false)
    protected String getPk() {
        if (primaryKey != null) {
            return primaryKey;
        }
        String[] primaryKeys = getPks();
        if (null != primaryKeys && primaryKeys.length == 1) {
            primaryKey = primaryKeys[0];
        }

        EAssert.assertTrue(primaryKey != null, String.format("get PrimaryKey is error in[%s]", getClass()));
        return primaryKey;
    }

    private transient Class<?> primaryType;

    @JSONField(serialize = false)
    protected Class<?> getPkType() {
        if (primaryType == null) {
            primaryType = _getTable().getColumnType(getPk());
        }
        return primaryType;
    }

    @JSONField(serialize = false)
    protected String[] getPks() {
        Table t = _getTable();
        if (t == null) {
            throw new RuntimeException("can't get table of " + _getUsefulClass() + " , maybe jboot install incorrect");
        }
        return t.getPrimaryKey();
    }
    
    /**
     * 判断是否包含指定列名
     * @param columnLabel
     * @return
     */
    protected boolean hasColumn(String columnLabel) {
        return _getTable().hasColumnLabel(columnLabel);
    }

    /**
     * 复制一个新的model
     * 主要是用在 从缓存取出数据的时候，如果直接修改，在ehcache会抛异常
     * 如果要对model进行修改，可以先copy一份新的，然后再修改
     *
     * @return
     */
    public M copy() {
        M m = null;
        try {
            m = (M) _getUsefulClass().newInstance();
            m.put(_getAttrs());
        } catch (Throwable e) {
            e.printStackTrace();
        }
        return m;
    }
    
    /**
     * ============== 原生SQL检索不建议使用！ =============================
     */

	@Override
	@Deprecated
	public List<M> find(String sql) {
		try {
			return super.find(sql);
		} catch (ActiveRecordException e) {
			log.error("SQL错误：\r\n--------------------------------\r\n{}\r\n--------------------------------", sql);
			throw new DboException(e.getCause().getMessage());
		}
	}
	
	@Override
	@Deprecated
	public List<M> find(String sql, Object... paras) {
		try {
			return super.find(sql, paras);
		} catch (ActiveRecordException e) {
			log.error("SQL错误：\r\n--------------------------------\r\n{}\r\n--------------------------------", sql);
			throw new DboException(e.getCause().getMessage());
		}
	}
	
	@Override
	@Deprecated
	public M findFirst(String sql) {
		try {
			return super.findFirst(sql);
		} catch (ActiveRecordException e) {
			log.error("SQL错误：\r\n--------------------------------\r\n{}\r\n--------------------------------", sql);
			throw new DboException(e.getCause().getMessage());
		}
	}
	
	@Override
	@Deprecated
	public M findFirst(String sql, Object... paras) {
		try {
			return super.findFirst(sql, paras);
		} catch (ActiveRecordException e) {
			log.error("SQL错误：\r\n--------------------------------\r\n{}\r\n--------------------------------", sql);
			throw new DboException(e.getCause().getMessage());
		}
	}
	

	//*******************************************  解密   ***********************************************/

	public M decryptAll(){
		Set<String> cols = _getTable().getColumnNameSet();
		for(String col: cols){
			String val = getStr(col);
			if(null == val) continue;
			val = RsaKit.decryptByDefPrivateKey(val);
			set(col, val);
		}
		return (M)this;
	}
	
	public M decrypt(String...keys){
		for(String key: keys){
			String val = getStr(key);
			if(null == val) continue;
			val = RsaKit.decryptByDefPrivateKey(val);
			set(key, val);
		}
		return (M)this;
	}
	
}
