/**
 * 
 */
package itez.core.wrapper.dbo.exec;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import com.beust.jcommander.internal.Maps;
import com.jfinal.plugin.activerecord.ActiveRecordException;
import com.jfinal.plugin.activerecord.Config;
import com.jfinal.plugin.activerecord.DbKit;
import com.jfinal.plugin.activerecord.Record;

import itez.core.runtime.modules.ModuleManager;
import itez.core.wrapper.dbo.DbManager;
import itez.core.wrapper.dbo.DbProp;
import itez.kit.EStr;

/**
 * <p>
 * SQL执行分析器
 * </p>
 * 
 * <p>Copyright(C) 2017-2021 <a href="http://www.itez.com.cn">上游科技</a></p>
 * 
 * @author		<a href="mailto:netwild@qq.com">Z.Mingyu</a>
 * @date		2021年4月12日 下午10:29:26
 */
public class DbExec {

	private static Map<String, DbExec> insMap = Maps.newHashMap();
	
	private String moduleCode;
	
	/**
	 * <p>
	 * 执行SQL
	 * </p>
	 * 
	 * @param moduleCode
	 * @param sql
	 * @return
	 */
	public static DbResult execute(String moduleCode, String sql){
		DbExec exec = insMap.get(moduleCode);
		if(exec == null){
			exec = new DbExec(moduleCode);
			insMap.put(moduleCode, exec);
		}
		return exec.execute(sql);
	}
	
	/**
	 * 构造方法
	 * 
	 * @param moduleCode 模块别名
	 */
	public DbExec(String moduleCode){
		this.moduleCode = moduleCode;
	}
	
	/**
	 * <p>
	 * 执行SQL
	 * </p>
	 * 
	 * @param sql SQL语句，多条语句中间用分号分隔。
	 * @return
	 */
	public DbResult execute(String sql){
		if(EStr.isEmpty(sql)) throw new RuntimeException("未发现SQL语句");
		Config config = getConfig();
		DataSource ds = getDataSource();
		Connection conn = null;
		try {
			conn = ds.getConnection();
			Statement stmt = conn.createStatement();
			stmt.execute(sql);
			DbResult dbRes = new DbResult();
			getNext(config, stmt, dbRes);
			close(stmt);
			return dbRes;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			close(conn);
		}
	}
	
	/**
	 * <p>
	 * 获取下一个结果集（更新计数）
	 * </p>
	 * 
	 * @param config
	 * @param stmt
	 * @param dbRes
	 */
	private void getNext(Config config, Statement stmt, DbResult dbRes){
		try {
			int rowCount = stmt.getUpdateCount();
			if(rowCount >= 0) { //有返回更新计数、DDL命令或 0 个更新
				dbRes.appendUpdateCount(rowCount);
				stmt.getMoreResults();
				getNext(config, stmt, dbRes);
			}
			ResultSet rs = stmt.getResultSet();
			if(rs != null){ //有返回结果集
				ResultSetMetaData metas = rs.getMetaData();
				List<Record> recs = config.getDialect().buildRecordList(config, rs);
				DbTable dbTable = new DbTable(metas, recs);
				dbRes.appendTable(dbTable);
				stmt.getMoreResults();
				close(rs);
				getNext(config, stmt, dbRes);
			}
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * <p>
	 * 关闭ResultSet
	 * </p>
	 * 
	 * @param rs
	 */
	private void close(ResultSet rs) {
		if (rs != null) {try {rs.close();} catch (SQLException e) { throw new ActiveRecordException(e);}}
	}
	
	/**
	 * <p>
	 * 关闭Statement
	 * </p>
	 * 
	 * @param st
	 */
	private void close(Statement st) {
		if (st != null) {try {st.close();} catch (SQLException e) { throw new ActiveRecordException(e);}}
	}
	
	/**
	 * <p>
	 * 关闭连接Connection
	 * </p>
	 * 
	 * @param conn
	 */
	private void close(Connection conn) {
		if (conn != null) {try {conn.close();} catch (SQLException e) { throw new ActiveRecordException(e);}}
	}
	
	/**
	 * <p>
	 * 返回当前模块的Db配置对象
	 * </p>
	 * 
	 * @return
	 */
	private Config getConfig(){
		DbProp dbProp = ModuleManager.me.getModuleDbProp(moduleCode);
		if(dbProp == null) dbProp = ModuleManager.me.getModuleDbProp(DbManager.MAIN_DATABASE_NAME);
		String dbName = dbProp.getReName();
		return DbKit.getConfig(dbName);
	}
	
	/**
	 * <p>
	 * 返回当前模块的数据源
	 * </p>
	 * 
	 * @return
	 */
	private DataSource getDataSource(){
		Config config = getConfig();
		return config.getDataSource();
	}
	
}
