package itez.core.wrapper.controller;

import java.awt.image.BufferedImage;
import java.io.File;
import java.net.URLDecoder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.Maps;
import com.jfinal.core.Controller;
import com.jfinal.core.NotAction;
import com.jfinal.kit.Kv;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Record;

import itez.core.runtime.EContext;
import itez.core.runtime.auth.AuthCode;
import itez.core.runtime.injector.EInjector;
import itez.core.runtime.session.EAttr;
import itez.core.runtime.session.ESessionBase;
import itez.core.wrapper.handler.EDomainHandler;
import itez.kit.EBase64;
import itez.kit.EClean;
import itez.kit.EFile;
import itez.kit.EJson;
import itez.kit.EPara;
import itez.kit.EProp;
import itez.kit.ERet;
import itez.kit.EStr;
import itez.kit.fileup.EFileFactory;
import itez.kit.fileup.EFileKit;
import itez.kit.restful.EMap;
import itez.kit.rsa.RsaKit;

/**
 * 根控制器
 * 自定义控制器需要继承该根控制器实现
 * 
 * @author netwild
 *
 */
public abstract class EController extends Controller {

	/**
	 * FlashMessage：向前端发送服务器消息，通常用于反馈表单提交结果
	 */
    protected HashMap<String, Object> flashMessages;

    /**
     * 获得当前子域
     */
	@NotAction
    public String getDomain(){
    	return getAttr(EDomainHandler.DomainAttrName);
    }

    /**
     * 获得当前的请求路径
     */
	@NotAction
    public String getTarget(){
    	return getAttr(EDomainHandler.TargetAttrName);
    }

    /**
     * 获得请求类型：GET / POST
     */
	@NotAction
    public HTTP getMethod(){
    	String method = getRequest().getMethod().toUpperCase();
    	return HTTP.valueOf(method);
    }
    
    /**
     * 获得参数，同时进行Url解码，避免汉字乱码
     */
	@NotAction
	@Override
	public String getPara() {
		String para = super.getPara();
		return UrlDecoder(para);
	}

	/**
	 * 按索引获得参数，同时进行Url解码，避免汉字乱码
	 */
	@NotAction
	@Override
	public String getPara(int index) {
		String para = super.getPara(index);
		return UrlDecoder(para);
	}

	/**
	 * 按索引获得参数，设置默认值，同时进行Url解码，避免汉字乱码
	 */
	@NotAction
	@Override
	public String getPara(int index, String defaultValue) {
		String para = super.getPara(index, defaultValue);
		return UrlDecoder(para);
	}

	/**
	 * 按别名获得参数，同时进行Url解码，避免汉字乱码
	 */
	@NotAction
	@Override
	public String getPara(String name) {
		String para = super.getPara(name);
		return UrlDecoder(para);
	}

	/**
	 * 按别名获得参数，设置默认值，同时进行Url解码，避免汉字乱码
	 */
	@NotAction
	@Override
	public String getPara(String name, String defaultValue) {
		String para = super.getPara(name, defaultValue);
		return UrlDecoder(para);
	}
	
	/**
	 * <p>
	 * 返回净化之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @return
	 */
	@NotAction
	public String getClean(String name) {
		return getClean(name, null);
	}
	/**
	 * <p>
	 * 返回净化之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	@NotAction
	public String getClean(String name, String defaultValue) {
		String val = get(name, defaultValue);
		return EClean.clean(val, EClean.text);
	}
	
	/**
	 * <p>
	 * 返回净化之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @return
	 */
	@NotAction
	public String getClean(int index) {
		return getClean(index, null);
	}
	/**
	 * <p>
	 * 返回净化之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	@NotAction
	public String getClean(int index, String defaultValue) {
		String val = get(index, defaultValue);
		return EClean.clean(val, EClean.text);
	}
	
	/**
	 * <p>
	 * 返回净化并解密之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @return
	 */
	@NotAction
	public String getDecrypt(String name) {
		return getDecrypt(name, null);
	}
	/**
	 * <p>
	 * 返回净化并解密之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	@NotAction
	public String getDecrypt(String name, String defaultValue) {
		String val = getClean(name, defaultValue);
		return RsaKit.decryptByDefPrivateKey(val);
	}
	
	/**
	 * <p>
	 * 返回净化并解密之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @return
	 */
	@NotAction
	public String getDecrypt(int index) {
		return getDecrypt(index, null);
	}
	/**
	 * <p>
	 * 返回净化并解密之后的参数值
	 * </p>
	 * 
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	@NotAction
	public String getDecrypt(int index, String defaultValue) {
		String val = getClean(index, defaultValue);
		return RsaKit.decryptByDefPrivateKey(val);
	}

	/**
	 * 获得参数表
	 */
	@NotAction
	public EMap getParaData(){
		EMap map = EMap.create();
		Map<String, String[]> requestParams = getRequest().getParameterMap();
		for (Entry<String, String[]> entry : requestParams.entrySet()) {
			String[] values = entry.getValue();
			String value = null;
			if(values != null && values.length > 0) value = Arrays.stream(values).collect(Collectors.joining(","));
			map.set(entry.getKey(), value);
		}
		return map;
	}
	
	/**
	 * 将接收到的表单数据解析成指定的Model对象实例
	 * @param modelClass
	 * @return
	 */
	@NotAction
	public <T> T parseBean(Class<T> beanClass) {
		return (T)EInjector.injectBean(beanClass);
	}
	
	/**
	 * URL编码
	 * @param paraValue
	 * @return
	 */
	@NotAction
	protected String UrlDecoder(String paraValue){
		if(getMethod() == HTTP.POST) return paraValue;
		try {
			paraValue = URLDecoder.decode(paraValue, EProp.Charset);
		} catch (Exception e) {}
		return paraValue;
	}
	
	/**
	 * 渲染到错误页面
	 */
	@NotAction
	public void renderErr(AuthCode authCode){
		renderErrMsg(authCode, null);
	}
	
	/**
	 * 渲染到错误页面
	 */
	@NotAction
	public void renderErrMsg(AuthCode authCode, String errMsg){
		renderErrHelp(authCode, errMsg, null);
	}
	
	/**
	 * 渲染到错误页面，包括推荐链接
	 */
	@NotAction
	public void renderErrHelp(AuthCode authCode, String errMsg, String helpUrl){
		setAttr("authCode", authCode);
		setAttr("errMsg", errMsg);
		setAttr("helpUrl", helpUrl);
		render(EProp.TempErrorPath);
	}
	
	/**
	 * 设置单项FlashMessage
	 * @param name
	 * @param value
	 * @return
	 */
    @NotAction
    public Controller setFlashMsg(String msg) {
        if (flashMessages == null) {
        	flashMessages = new HashMap<>();
        }
        flashMessages.put(ERet.ATTR_MSG, msg);
        return this;
    }
	
	/**
	 * 设置单项FlashMessage
	 * @param name
	 * @param value
	 * @return
	 */
    @NotAction
    public Controller setFlashAttr(String name, Object value) {
        if (flashMessages == null) {
        	flashMessages = new HashMap<>();
        }
        flashMessages.put(name, value);
        return this;
    }

    /**
     * 设置多项FlashMessage
     * @param map
     * @return
     */
    @SuppressWarnings({ "rawtypes", "unchecked" })
	@NotAction
    public Controller setFlashMap(Map map) {
        if (map == null) {
            throw new NullPointerException("map is null");
        }
        if (flashMessages == null) {
        	flashMessages = new HashMap<>();
        }
        flashMessages.putAll(map);
        return this;
    }

    /**
     * 返回单项FlashMessage
     * @param name
     * @return
     */
	@NotAction
    public String getFlashMsg() {
        return flashMessages == null ? null : flashMessages.get(ERet.ATTR_MSG).toString();
    }

    /**
     * 返回单项FlashMessage
     * @param name
     * @return
     */
    @SuppressWarnings("unchecked")
	@NotAction
    public <T> T getFlashAttr(String name) {
        return flashMessages == null ? null : (T) flashMessages.get(name);
    }

    /**
     * 返回多项FlashMessage
     * @return
     */
    @NotAction
    public HashMap<String, Object> getFlashMap() {
        return flashMessages;
    }
    
    /**
     * 获得Attr
     */
    @NotAction
    public EAttr attr(){
    	return EContext.getAttr();
    }
    
    /**
     * 获得Response
     */
    @NotAction
    public HttpServletRequest request(){
    	return EContext.getRequest();
    }
    
    /**
     * 获得Response
     */
    @NotAction
    public HttpServletResponse response(){
    	return EContext.getResponse();
    }
    
    /**
     * 获得Session
     */
    @NotAction
    public ESessionBase session(){
    	return EContext.getSession();
    }

    @NotAction
	@SuppressWarnings("unchecked")
	public <T> T getRawDataTo(Class<?> targetClass) {
		return (T) EJson.parse(getRawData(), targetClass);
	}

    @NotAction
	public Kv getRawDataToKv() {
		return getRawDataTo(Kv.class);
	}
    
    /*===================================== 文件上传（旧方法） ===================================================*/
	
	/**
	 * 文件上传，并返回url
	 * @param file
	 * @return
	 */
    @NotAction
	protected String getUpload(File file){
		String rootPath = String.join(EStr.FileSep, attr().getDomain(), attr().getModuleCode());
		return EFileKit.upload(file, rootPath);
	}
	
	/**
	 * 文件上传，并返回url
	 * @param file
	 * @param subPath
	 * @return
	 */
    @NotAction
	protected String getUpload(File file, String subPath){
		String rootPath = String.join(EStr.FileSep, attr().getDomain(), subPath);
		return EFileKit.upload(file, rootPath);
	}
	
	/**
	 * 文件上传，并返回url
	 * @param file
	 * @return
	 */
    @NotAction
	protected String getUpload(BufferedImage file, String formatName){
		String rootPath = String.join(EStr.FileSep, attr().getDomain(), attr().getModuleCode());
		return EFileKit.upload(file, formatName, rootPath);
	}
    
    /*===================================== 文件上传（新方法） ===================================================*/
    
    /*===================================== 参数序列化 ===================================================*/
	
	/**
	 * 将接收到的参数序列化
	 * @return
	 */
    @NotAction
	protected String paramSerialize(String...ignore){
		Map<String, String[]> params = getRequest().getParameterMap();
		if(params.size() == 0) return "";
		Map<String, String> paramsNew = Maps.newHashMap();
		params.forEach((k, v) -> {
			if(ignore.length == 0){
				paramsNew.put(k, v[0]);
			}else{
				boolean isIgnore = Arrays.stream(ignore).anyMatch(i -> i.equals(k));
				if(!isIgnore) paramsNew.put(k, v[0]);
			}
		});
		String json = EJson.toJson(paramsNew);
		return EBase64.encode(json);
	}
	
	/**
	 * 将参数反序列化
	 * @param params
	 * @return
	 */
    @NotAction
	@SuppressWarnings("unchecked")
	protected Map<String, String> paramDeserialize(String params){
		Map<String, String> paramsNew = Maps.newHashMap();
		if(EStr.isEmpty(params)) return paramsNew;
		String json = EBase64.decode(params);
		paramsNew = EJson.parse(json, Map.class);
		return paramsNew;
	}
	
	/**
	 * 将参数打包成对象，便于传递与获取
	 * @return
	 */
    @NotAction
	protected EPara paramPack(){
		return new EPara(getRequest());
	}
	
	/**
	 * 将参数打包成对象，便于传递与获取
	 * @return
	 */
    @NotAction
	protected EPara paramPack(boolean autoUpload){
		return new EPara(getRequest(), autoUpload);
	}
	
	/**
	 * 将参数打包成对象，便于传递与获取
	 * @param fileSubPath 上传文件子路径，默认为：files
	 * @return
	 */
    @NotAction
	protected EPara paramPack(String fileSubPath){
		return new EPara(getRequest(), fileSubPath);
	}
	
	/**
	 * 将Model列表转换为Record列表
	 * @param models
	 * @return
	 */
    @NotAction
	@SuppressWarnings("rawtypes")
	protected List<Record> parseRecords(List<? extends Model> models){
		return models.stream().map(r -> r.toRecord()).collect(Collectors.toList());
	}
	
	/**
	 * <p>
	 * 将HTML写入静态文件
	 * </p>
	 * 
	 * @param html
	 * @param filePath
	 * @return 静态文件访问地址
	 */
    @NotAction
	protected String generatorHtml(String html, String filePath){
		int lastStep = filePath.lastIndexOf("/");
		String folderPath = lastStep == -1 ? "" : EStr.FileSep + EStr.left(filePath, lastStep).replace("/", EStr.FileSep).replace("\\", EStr.FileSep);
		String fileName = lastStep == -1 ? filePath : EStr.right(filePath, filePath.length() - lastStep - 1);
		String htmlPath = String.format("%s%s%shtml%s", EFileFactory.me.getLocalDiskPath(), attr().getDomain(), EStr.FileSep, folderPath);
		File htmlFolder = new File(htmlPath);
		if(!htmlFolder.exists()) htmlFolder.mkdirs();
		File htmlFile = new File(String.format("%s%s%s", htmlPath, EStr.FileSep, fileName));
		if(!htmlFile.exists()){
			try {
				htmlFile.createNewFile();
			} catch (Exception e) {
				if(EProp.DevMode) e.printStackTrace();
			}
		}
		EFile.write(htmlFile, html);
		String url = String.format("%s%s/html/%s", EFileFactory.me.getLocalUrl(), attr().getDomain(), filePath);
		if(url.indexOf("http") != 0) url = attr().getWeb().concat(url); 
		return url;
	}

    /**
     * <p>
     * 图片验证码
     * </p>
     * 
     * @param paraVal
     * @return
     */
    @NotAction
	public boolean validateCaptchaVal(String paraVal) {
		return com.jfinal.captcha.CaptchaRender.validate(this, paraVal);
	}
	
	/**
	 * 服务已关闭
	 */
    @NotAction
	public void off(String msg){
    	setAttr("msg", EStr.ifEmpty(msg, "请关注官网说明或联系管理员获取具体信息"));
		render("/main/temp/off.html");
	}

}
