package itez.core.util.inject;

import java.lang.reflect.Method;
import java.util.Set;

import com.jfinal.core.converter.TypeConverter;
import com.jfinal.plugin.activerecord.ActiveRecordException;
import com.jfinal.plugin.activerecord.Model;
import com.jfinal.plugin.activerecord.Table;
import com.jfinal.plugin.activerecord.TableMapping;

import itez.kit.EStr;
import itez.kit.restful.EMap;

public class Injector {

	private static <T> T createInstance(Class<T> objClass) {
		try {
			return objClass.newInstance();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	public static <T> T injectModel(Class<T> modelClass, EMap params, boolean skipConvertError) {
		String modelName = modelClass.getSimpleName();
		return (T)injectModel(modelClass, EStr.toLowerCaseFirst(modelName), params, skipConvertError);
	}
	
	public static <T> T injectBean(Class<T> beanClass, EMap params, boolean skipConvertError) {
		String beanName = beanClass.getSimpleName();
		return (T)injectBean(beanClass, EStr.toLowerCaseFirst(beanName), params, skipConvertError);
	}
	
	@SuppressWarnings("unchecked")
	public static final <T> T injectBean(Class<T> beanClass, String beanName, EMap params, boolean skipConvertError) {
		Object bean = createInstance(beanClass);
		String modelNameAndDot = EStr.notEmpty(beanName) ? beanName + "." : null;
		TypeConverter converter = TypeConverter.me();
		Method[] methods = beanClass.getMethods();
		for (Method method : methods) {
			String methodName = method.getName();
			if (methodName.startsWith("set") == false || methodName.length() <= 3) {	// only setter method
				continue;
			}
			Class<?>[] types = method.getParameterTypes();
			if (types.length != 1) {						// only one parameter
				continue;
			}
			
			String attrName = EStr.toLowerCaseFirst(methodName.substring(3));
			String paraName = modelNameAndDot != null ? modelNameAndDot + attrName : attrName;
			if (params.containsKey(paraName)) {
				try {
					String paraValue = params.getStr(paraName);
					Object value = paraValue != null ? converter.convert(types[0], paraValue) : null;
					method.invoke(bean, value);
				} catch (Exception e) {
					if (skipConvertError == false) {
						throw new RuntimeException(e);
					}
				}
			}
		}
		
		return (T)bean;
	}
	
	@SuppressWarnings("unchecked")
	public static final <T> T injectModel(Class<T> modelClass, String modelName, EMap params, boolean skipConvertError) {
		Object temp = createInstance(modelClass);
		if (temp instanceof Model == false) {
			throw new IllegalArgumentException("getModel only support class of Model, using getBean for other class.");
		}
		
		Model<?> model = (Model<?>)temp;
		Table table = TableMapping.me().getTable(model.getClass());
		if (table == null) {
			throw new ActiveRecordException("The Table mapping of model: " + modelClass.getName() + 
					" not exists or the ActiveRecordPlugin not start.");
		}
		
		String modelNameAndDot = EStr.notEmpty(modelName) ? modelName + "." : null;
		TypeConverter converter = TypeConverter.me();
		// 对 paraMap进行遍历而不是对table.getColumnTypeMapEntrySet()进行遍历，以便支持 CaseInsensitiveContainerFactory
		// 以及支持界面的 attrName有误时可以感知并抛出异常避免出错
		
		Set<Object> keys = params.keySet();
		for(Object key : keys){
			String paraName = key.toString();
			String attrName;
			if (modelNameAndDot != null) {
				if (paraName.startsWith(modelNameAndDot)) {
					attrName = paraName.substring(modelNameAndDot.length());
				} else {
					continue ;
				}
			} else {
				attrName = paraName;
			}

			Class<?> colType = table.getColumnType(attrName);
			if (colType == null) {
				if (skipConvertError) {
					continue ;
				} else {
					throw new ActiveRecordException("The model attribute " + attrName + " is not exists.");
				}
			}

			try {
				String paraValue = params.getStr(key);				
				Object value = paraValue != null ? converter.convert(colType, paraValue) : null;
				model.set(attrName, value);
			} catch (Exception e) {
				if (skipConvertError == false) {
					throw new RuntimeException("Can not convert parameter: " + paraName, e);
				}
			}
		}
		
		return (T)model;
	}
}
