package itez.core.util.grouping;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.beust.jcommander.internal.Lists;
import com.beust.jcommander.internal.Maps;
import com.jfinal.kit.Kv;
import com.jfinal.plugin.activerecord.Record;

import itez.core.util.grouping.GroupingField.Type;
import itez.kit.ENum;

public class GroupingItem {

	private Integer groupIndex;						//分组索引
	private List<Record> data;						//分组数据
	private GroupingPlanBase plan;					//分组方案
	private Map<String, GroupingField> fields;		//分组字段配置
	private Set<String> fieldNames;					//分组字段名称列表
	
	/**
	 * 字段区间统计
	 * 
	 * Key：字段名（fieldName）
	 * Kv：字段内各区间的数量统计（例如：性别字段“sex”去重之后，有“男”和“女”两种取值（区间），则该Kv内需要分别累计两个区间各自的数量）
	 * 备注：区间统计只适用于字符型的字段，数值型字段只在Kv中累计“sum”即可，待分组完成后，再根据 dataCount 计算“avg”。
	 */
	private Map<String, Kv> fieldEnumStis;

	/**
	 * 字段区间差异
	 * 
	 * Key：字段名（fieldName）
	 * Kv：字段内各区间的数量统计与方案平均数差值
	 * 备注：字符型字段需要按照区间分别进行比较，数值型字段只需要比较avg即可。
	 */
	private Map<String, Kv> fieldEnumDiff;

	/**
	 * 字段差异统计
	 * 
	 * Key：字段名（fieldName）
	 * Kv：字段差异统计结果
	 * 备注：在Kv中需要为每个字段计算以下属性：
	 * 
	 *  1、fieldDiff：字段差异总数（字段内各区间差异求和）
	 *  2、enumCount：字段区间个数（数值型为1）
	 *  3、weight：字段权重
	 *  4、dataCount：分组数据条数
	 *  5、avgCount：分组平均数据条数（在方案对象中计算）
	 *  6、countDiff：数据条数差异 = abs(dataCount - avgCount)
	 *  7、countWeight：数据条数权重
	 *  8、groupCount：分组个数（在方案对象中计算）
	 *  9、rate：字段差异率 = (diffCount × weight + countDiff × countWeight)  / (groupCount × enumCount × dataCount)
	 */
	private Map<String, Kv> fieldDiffStis;

	public GroupingItem(GroupingPlanBase plan, Integer groupIndex) {
		this.groupIndex = groupIndex;
		this.plan = plan;
		data = Lists.newArrayList();
		fieldEnumStis = Maps.newHashMap();
		fieldEnumDiff = Maps.newHashMap();
		fieldDiffStis = Maps.newHashMap();
		
		fields = plan.getFields();
		fieldNames = fields.keySet();
		for(String fieldName: fieldNames) fieldEnumStis.put(fieldName, Kv.create());
	}
	
	/**
	 * 添加数据
	 * 
	 * @param rec
	 * @return
	 */
	public GroupingItem putData(Record rec){
		data.add(rec);
		calcEnumStis(rec);
		return this;
	}
	
	/**
	 * 字段区间统计
	 * @param rec
	 */
	private void calcEnumStis(Record rec){
		for(String fieldName: fieldNames){
			GroupingField fieldCfg = fields.get(fieldName);
			Kv enumv = fieldEnumStis.get(fieldName);
			if(fieldCfg.getType() == Type.NUMBER){
				if(enumv.get("sum") == null) enumv.set("sum", 0d);
				Double v1 = enumv.getAs("sum");
				Double v2 = rec.getDouble(fieldName);
				enumv.set("sum", ENum.add(v1, v2));
				plan.addFieldEnumStis(fieldName, v2);
			}else{
				String val = rec.getStr(fieldName);
				if(enumv.get(val) == null) enumv.set(val, 0);
				enumv.set(val.toString(), enumv.getInt(val) + 1);
				plan.addFieldEnumStis(fieldName, val);
			}
		}
	}
	
	/**
	 * 字段区间差值统计
	 */
	@SuppressWarnings("unchecked")
	public void calcEnumDiff(){
		Map<String, Kv> planEnumAvgs = plan.getFieldEnumAvg();
		Double dataCount = Double.valueOf(data.size());
		for(String fieldName : fieldNames){
			Kv stis = fieldEnumStis.get(fieldName);
			Kv avgs = planEnumAvgs.get(fieldName);
			GroupingField.Type type = fields.get(fieldName).getType();
			if(type == Type.NUMBER){
				Double sum = stis.getAs("sum");
				Double avg = ENum.div(sum, dataCount);
				Double avgPlan = avgs.getAs("avg");
				Double diff = Math.abs(ENum.sub(avg, avgPlan));
				fieldEnumDiff.put(fieldName, Kv.by("avg", diff));
			}else{
				Kv diffs = Kv.create();
				Set<String> enums = stis.keySet();
				for(String enumv : enums){
					Integer enumCnt = stis.getInt(enumv);
					Double avgPlan = avgs.getAs(enumv);
					diffs.set(enumv.toString(), Math.abs(ENum.sub(Double.valueOf(enumCnt), avgPlan)));
				}
				fieldEnumDiff.put(fieldName, diffs);
			}
		}
		calcDiffStis();
	}
	
	/**
	 * 字段差异统计
	 */
	@SuppressWarnings("unchecked")
	private void calcDiffStis(){
		Double dataCount = Double.valueOf(data.size());
		Double avgCount = plan.getAvgCount();
		Double countDiff = Math.abs(ENum.sub(dataCount, avgCount));
		Double countWeight = Double.valueOf(plan.getCountWeight());
		Double countDiffStis = ENum.mul(countDiff, countWeight);
		Double groupCount = Double.valueOf(plan.getGroupCount());
		for(String fieldName : fieldNames){
			GroupingField fieldCfg = fields.get(fieldName);
			GroupingField.Type type = fieldCfg.getType();
			Kv diffs = fieldEnumDiff.get(fieldName);
			Double fieldDiff = 0d;
			Double enumCount = 1d;
			Double weight = Double.valueOf(fieldCfg.getWeight());
			if(type == Type.NUMBER){
				fieldDiff = diffs.getAs("avg");
			}else{
				Set<String> enums = diffs.keySet();
				for(String enumv : enums){
					Double enumDiff = diffs.getAs(enumv);
					fieldDiff = ENum.add(fieldDiff, enumDiff);
					enumCount = ENum.add(enumCount, 1d);
				}
			}
			Double rate1 = ENum.mul(fieldDiff, weight);					//当前分组字段差异统计
			Double rate2 = ENum.add(rate1, countDiffStis);				//当前分组数据条数差异统计
			Double rate3 = ENum.mul(dataCount, groupCount, enumCount);	//分母
			Double rateVal = ENum.div(rate2, rate3);					//字段差异值
			Double ratePer = ENum.mul(rateVal, 100d);					//字段差异率
			Kv diffRate = Kv.by("dataCount", dataCount).set("avgCount", avgCount).set("countDiff", countDiff).set("groupCount", groupCount).set("type", type.name());
			diffRate.set("fieldDiff", fieldDiff).set("enumCount", enumCount).set("weight", weight).set("rateVal", rateVal).set("ratePer", ratePer);
			fieldDiffStis.put(fieldName, diffRate);
		}
	}
	
	/**
	 * 返回分组数据
	 * 
	 * @return
	 */
	public List<Record> getData() {
		return data;
	}
	
	/**
	 * 返回分组数据条数
	 * 
	 * @return
	 */
	public Integer getDataCount(){
		return data.size();
	}

	public Map<String, Kv> getFieldEnumStis() {
		return fieldEnumStis;
	}

	public Map<String, Kv> getFieldEnumDiff() {
		return fieldEnumDiff;
	}

	public Map<String, Kv> getFieldDiffStis() {
		return fieldDiffStis;
	}

	public Integer getGroupIndex() {
		return groupIndex;
	}
	
}
