package itez.core.runtime.session;

import java.util.concurrent.TimeUnit;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import itez.core.launch.JW;
import itez.core.wrapper.handler.EDomainHandler;
import itez.kit.ECookie;
import itez.kit.EProp;
import itez.kit.EStr;
import itez.kit.EUid;

public class TokenManager {

	public final static TokenManager me = new TokenManager();
	private TokenManager(){}

	public final static Integer SessionTimeOutMinutes = EProp.SessionTimeOutMinutes; // Session保持时间，默认30分钟
	public final static String SessionHeaderName = "X-Auth-Token"; // Token在Request.Header中的键名
	public final static String SessionAttrName = "JWSID"; // Token在Request.Attr及Params中的键名
	public final static String SessionCacheName = "JWSESSION"; // Token在Cache中的存储器名称
	public final static String TokenUuid = "uuid"; // Token中的UUID键名
	
	/**
	 * 从请求中获得Token
	 * 
	 * @param request
	 * @return
	 */
	public String getToken(HttpServletRequest request, HttpServletResponse response){
		if(request == null) return null;
		Object reqAttrSess = request.getAttribute(SessionAttrName);
		String id4Cookie = ECookie.getCookie(request, getTokenCookieKey(request));
		String id4Attribute = reqAttrSess != null ? (String) reqAttrSess : null;
		String id4Params = request.getParameter(SessionAttrName);
		String id4Header = request.getHeader(SessionHeaderName);
		String token = EStr.findUseful(id4Cookie, id4Attribute, id4Params, id4Header);
		if(EStr.isEmpty(token)) token = createToken(request, response);
		return token;
	}
	
	/**
	 * 生成新Token
	 * 
	 * @param request
	 * @return
	 */
	public String createToken(HttpServletRequest request, HttpServletResponse response){
        String sessionId = EUid.generator();
        String token = encodeToken(sessionId);
        ECookie.setCookie(response, getTokenCookieKey(request), token, -1, TimeUnit.SECONDS);
        request.setAttribute(SessionAttrName, token);
        return token;
	}
	
	/**
	 * 基于uuid生成加密之后的Token
	 * @param uuid
	 * @return
	 */
	public String encodeToken(String uuid){
		Claims claims = Jwts.claims();
		claims.setId(uuid);
		return Jwts.builder().setClaims(claims).signWith(SignatureAlgorithm.HS512, JW.TokenSecret).compact();
	}
	
	/**
	 * 解析Token，返回uuid
	 * @param token
	 * @return
	 */
	public String decodeToken(String token) {
		if(EStr.isEmpty(token)) return null;
		String uuid = null;
		try {
			Jws<Claims> jws = Jwts.parser().setSigningKey(JW.TokenSecret).parseClaimsJws(token);
			Claims claims = jws.getBody();
			uuid = claims.getId();
		} catch (Exception e) {
		}
		return uuid;
	}

	public String getTokenCookieKey(HttpServletRequest request){
		Object reqAttrDomain = request.getAttribute(EDomainHandler.DomainAttrName);
		String domain = reqAttrDomain != null ? (String) reqAttrDomain : EProp.DomainDefault;
		return String.format("_%s_%s_", SessionAttrName, domain);
	}
	
	public String getSessionCookieKey(HttpServletRequest request, String name){
		Object reqAttrDomain = request.getAttribute(EDomainHandler.DomainAttrName);
		String domain = reqAttrDomain != null ? (String) reqAttrDomain : EProp.DomainDefault;
		return String.format("_%s_%s_%s_", SessionAttrName, domain, name);
	}
	
}
