package itez.core.runtime.session;

import java.util.Collections;
import java.util.Enumeration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.servlet.ServletContext;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import itez.core.runtime.EContext;
import itez.kit.ECookie;
import itez.kit.cache.ECacheFactory;
import itez.kit.cache.ICache;

/**
 * 自定义封装HttpSession
 * 主要使用Cache替换默认的Session实现，最终支持集群/分布式部署
 * 
 * @author netwild
 *
 */
public class ESessionCache extends ESessionBase{
	
	protected final Map<String, Object> newAttributes = Maps.newHashMap();
	protected final Set<String> deleteAttribute = Sets.newHashSet();
	protected final Map<String, Object> sessionStore;

	protected volatile boolean empty;
	protected volatile boolean dataChanged;
	
	public ESessionCache(ServletContext context) {
		id = generateSessionId();
		creationTime = System.currentTimeMillis();
		lastAccessedTime = creationTime;
		maxInactiveInterval  = (int)TimeUnit.MINUTES.toSeconds(SessionTimeOutMinutes);
		servletContext = context;
		sessionStore = generateSessionStore();
		empty = sessionStore.isEmpty();
	}
	
	/**
	 * 获取缓存对象
	 * @return
	 */
	private ICache getCache(){
		return ECacheFactory.me.getCache();
	}

	/**
	 * 生成或者获取SessionStore
	 * @return
	 */
    private Map<String, Object> generateSessionStore() {
        Map<String, Object> store = getCache().get(SessionCacheName, id);
        if (store == null){
        	store = Collections.emptyMap();
        }
        return store;
    }
	
	/**
	 * 更新Session的声明周期
	 */
    @Override
	public void refresh() {
        if (invalid) { //session已经被整体删除，调用了session.invalidate()
            getCache().remove(SessionCacheName, id);
            ECookie.setCookie(EContext.getResponse(), getCookieKey(), null, 0, TimeUnit.MINUTES);
        } else if (empty) { // 空的httpSession数据
        	getCache().remove(SessionCacheName, id);
            ECookie.setCookie(EContext.getResponse(), getCookieKey(), null, 0, TimeUnit.MINUTES);
        } else if (dataChanged) { //session 已经被修改(session数据的增删改查)
            Map<String, Object> snapshot = snapshot();
            // 数据已经全部被删除了
            if (snapshot.isEmpty()) {
            	getCache().remove(SessionCacheName, id);
                ECookie.setCookie(EContext.getResponse(), getCookieKey(), null, 0, TimeUnit.MINUTES);
            } else {
                getCache().put(SessionCacheName, id, snapshot, maxInactiveInterval);
            }
        } else { //更新session存储时间
            getCache().setTtl(SessionCacheName, id, maxInactiveInterval);
        }
	}
	
	/**
	 * 获取当前的Session数据快照
	 * @return
	 */
    public Map<String, Object> snapshot() {
        Map<String, Object> snap = Maps.newHashMap();
        snap.putAll(sessionStore);
        snap.putAll(newAttributes);
        for (String name : deleteAttribute) {
            snap.remove(name);
        }
        return snap;
    }

    @Override
    public Object getAttribute(String name) {
        checkValid();
        if (newAttributes.containsKey(name)) {
            return newAttributes.get(name);
        } else if (deleteAttribute.contains(name)) {
            return null;
        }
        return sessionStore.get(name);
    }
    
    @Override
    public void setAttribute(String name, Object value) {
        checkValid();
        if (value != null) {
            newAttributes.put(name, value);
            deleteAttribute.remove(name);
            empty = false;
            dataChanged = true;
        } else {
            removeAttribute(name);
        }
    }

    @Override
    public void removeAttribute(String name) {
        checkValid();
        if (empty && newAttributes.isEmpty()) {
            return;
        }

        if (!newAttributes.containsKey(name) && !sessionStore.containsKey(name)){
            return;
        }

        deleteAttribute.add(name);
        newAttributes.remove(name);
        dataChanged = true;
    }

	@Override
	public void invalidate() {
		invalid = true;
		dataChanged = true;
	}

	@Override
	public Enumeration<String> getAttributeNames() {
        checkValid();
        Set<String> names = Sets.newHashSet(sessionStore.keySet());
        names.addAll(newAttributes.keySet());
        names.removeAll(deleteAttribute);
        return Collections.enumeration(names);
	}
	
    protected void checkValid() throws IllegalStateException {
        if (invalid) {
            throw new IllegalStateException("http session has invalidate");
        }
    }

}
